// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of APE.
// 
// The exception is that, if you link the APE library with other files
// to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the APE library code into it.
//
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
//
// This exception applies only to the code released under the 
// name APE.  If you copy code from other releases into a copy of
// APE, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for APE, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include "config.h"
#include "macros.h"
#include "thread.h"
#include "socket.h"
#include <errno.h>
#include <fcntl.h>

TimerPort::TimerPort()
{
	active = false;
	gettimeofday(&timer, NULL);
}

void TimerPort::setTimer(timeout_t timeout)
{
	gettimeofday(&timer, NULL);
	active = false;
	if(timeout)
		incTimer(timeout);
}

void TimerPort::incTimer(timeout_t timeout)
{
	int secs = timeout / 1000;
	int usecs = (timeout % 1000) * 1000;

	timer.tv_usec += usecs;
	if(timer.tv_usec > 1000000l)
	{
		++timer.tv_sec;
		timer.tv_usec %= 1000000l;
	}
	timer.tv_sec += secs;
	active = true;
}

void TimerPort::endTimer(void)
{
	active = false;
}

timeout_t TimerPort::getTimer(void)
{
	struct timeval now;
	long diff;

	if(!active)
		return ~0;

	gettimeofday(&now, NULL);
	diff = (timer.tv_sec - now.tv_sec) * 1000l;
	diff += (timer.tv_usec - now.tv_usec) / 1000l;
	if(diff < 0)
		return 0l;

	return diff;
}

SocketPort::SocketPort(SocketService *svc, TCPSocket &tcp) :
Socket(accept(tcp.so, NULL, NULL))
{
	next = prev = NULL;
	service = NULL;

	if(so > -1)
	{
		service = svc;
		svc->Attach(this);
	}
}

SocketPort::SocketPort(SocketService *svc, InetAddress &ia, short port) :
Socket(AF_INET, SOCK_DGRAM, 0)
{
	struct sockaddr_in addr;

	next = prev = NULL;
	service = NULL;
	addr.sin_family = AF_INET;
	addr.sin_addr = getaddress(ia);
	addr.sin_port = htons(port);
	if(bind(so, (struct sockaddr *)&addr, (socklen_t)sizeof(addr)))
	{
		endSocket();
		Error(SOCKET_BINDING_FAILED);
		return;
	}
	state = SOCKET_BOUND;
	service = svc;
	svc->Attach(this);
}

SocketPort::~SocketPort()
{
	if(service)
		service->Detach(this);
	endSocket();
}

sockerror_t SocketPort::Connect(InetAddress &ia, short port)
{
	struct sockaddr_in addr;
	long opts;
	sockerror_t rtn = SOCKET_SUCCESS;

	addr.sin_family = AF_INET;
	addr.sin_addr = getaddress(ia);
	addr.sin_port = htons(port);

	opts = fcntl(so, F_GETFL);
	fcntl(so, F_SETFL, opts | O_NDELAY);
	if(connect(so, (struct sockaddr *)&addr, (socklen_t)sizeof(addr)))
		rtn = connectError();
	fcntl(so, F_SETFL, opts);
	return rtn;
}


void SocketPort::setTimer(timeout_t ptimer)
{
	TimerPort::setTimer(ptimer);
	service->Update();
}

void SocketPort::incTimer(timeout_t ptimer)
{
	TimerPort::incTimer(ptimer);
	service->Update();
}

SocketService::SocketService(int pri) :
Thread(NULL, pri), Mutex()
{
	long opt;

	first = last = NULL;
	count = 0;
	FD_ZERO(&connect);
	::pipe(iosync);
	hiwater = iosync[0] + 1;
	FD_SET(iosync[0], &connect);

	opt = fcntl(iosync[0], F_GETFL);
	fcntl(iosync[0], F_SETFL, opt | O_NDELAY);	
}

SocketService::~SocketService()
{
	char buf[1];

	buf[0] = 0;
	::write(iosync[1], &buf, 1);
	Terminate();

	while(first)
		delete first;
}

void SocketService::Attach(SocketPort *port)
{
	EnterMutex();
	if(last)
		last->next = port;

	port->prev = last;
	last = port;
	FD_SET(port->so, &connect);
	if(port->so >= hiwater)
		hiwater = port->so + 1;

	if(!first)
	{
		first = port;
		LeaveMutex();
		Start();
	}
	else
	{
		LeaveMutex();
		Update();
	}			
	++count;
}

void SocketService::Detach(SocketPort *port)
{
	EnterMutex();
	FD_CLR(port->so, &connect);
	if(port == first && port == last)
	{
		first = last = NULL;
		LeaveMutex();
		return;
	}
	if(port->prev)
		port->prev->next = port->next;
	if(port->next)
		port->next->prev = port->prev;
	if(port == first)
		first = port->next;
	if(port == last)
		last = port->prev;
	LeaveMutex();
	Update();
	--count;
}

void SocketService::Update(void)
{
	char buf[1];
	buf[0] = 1;
	::write(iosync[1], &buf, 1);
}

void SocketService::Run(void)
{
	struct timeval timeout, *tvp;
	timeout_t timer, expires;
	SocketPort *port;
	fd_set inp, out, err;
	char buf;
	int so;

	FD_ZERO(&inp);
	FD_ZERO(&out);
	FD_ZERO(&err);

	setCancel(THREAD_CANCEL_DEFERRED);
	for(;;)
	{
		timer = ~0;
		while(1 == ::read(iosync[0], &buf, 1))
		{
			if(buf)
				continue;

			setCancel(THREAD_CANCEL_IMMEDIATE);
			Sleep(~0);
			Exit();
		}
	
		EnterMutex();
		port = first;
		while(port)
		{
			so = port->so;
			if(FD_ISSET(so, &err))
				port->Disconnect();

			if(FD_ISSET(so, &inp))
				port->Pending();

			expires = port->getTimer();
			if(expires > 0)
				if(expires < timer)
					timer = expires;

			if(!expires)
			{
				port->endTimer();
				port->Expired();
			}

			port = port->next;
		}
		LeaveMutex();
		memcpy(&err, &connect, sizeof(err));
		memcpy(&inp, &connect, sizeof(inp));
		if(timer == ~0)
			tvp = NULL;
		else
		{
			tvp = &timeout;
			timeout.tv_sec = timer / 1000;
			timeout.tv_usec = (timer % 1000) * 1000;
		}
		select(hiwater, &inp, &out, &err, tvp);		
	}
}						
		
	
