// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of Common C++.
// 
// The exception is that, if you link the Common C++ library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the Common C++ library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name Common C++.  If you copy code from other releases into a copy of
// Common C++, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for Common C++, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include "config.h"
#include "macros.h"
#include "thread.h"

ThreadLock::ThreadLock()
{
#ifdef HAVE_PTHREAD_RWLOCK
	pthread_rwlockattr_t attr;
	
	pthread_rwlockattr_init(&attr);
	if(pthread_rwlock_init(&_lock, &attr) && getException() == THROW_OBJECT)
		throw(this);
#else
	pthread_mutexattr_t attr;
	
	pthread_mutexattr_init(&attr);
	if(pthread_mutex_init(&_lock, &attr) && getException() == THROW_OBJECT)
		throw(this);
#endif
}

ThreadLock::~ThreadLock()
{
#ifdef HAVE_PTHREAD_RWLOCK
	pthread_rwlock_destroy(&_lock);
#else
	pthread_mutex_destroy(&_lock);
#endif
}

void ThreadLock::ReadLock(void)
{
#ifdef HAVE_PTHREAD_RWLOCK
	pthread_rwlock_rdlock(&_lock);
#else
	pthread_mutex_lock(&_lock);
#endif
}

void ThreadLock::WriteLock(void)
{
#ifdef HAVE_PTHREAD_RWLOCK
	pthread_rwlock_wrlock(&_lock);
#else
	pthread_mutex_lock(&_lock);
#endif
}

void ThreadLock::Unlock(void)
{
#ifdef HAVE_PTHREAD_RWLOCK
	pthread_rwlock_unlock(&_lock);
#else
	pthread_mutex_unlock(&_lock);
#endif
}

bool ThreadLock::TryReadLock(void)
{
#ifdef	HAVE_PTHREAD_RWLOCK
	if(pthread_rwlock_tryrdlock(&_lock))
		return false;
#else
	if(pthread_mutex_trylock(&_lock))
		return false;
#endif
	return true;
}

bool ThreadLock::TryWriteLock(void)
{
#ifdef	HAVE_PTHREAD_RWLOCK
	if(pthread_rwlock_trywrlock(&_lock))
		return false;
#else
	if(pthread_mutex_trylock(&_lock))
		return false;
#endif
	return true;
}

Mutex::Mutex()
{
	pthread_mutexattr_t _attr;

	pthread_mutexattr_init(&_attr);
#ifdef	PTHREAD_MUTEXTYPE_RECURSIVE
	pthread_mutexattr_settype(&_attr, PTHREAD_MUTEXTYPE_RECURSIVE);
#endif
	if(pthread_mutex_init(&_mutex, &_attr) && getException() == THROW_OBJECT)
		throw(this);

#ifndef	PTHREAD_MUTEXTYPE_RECURSIVE
	_level = 0;
	_tid = NULL;
#endif
}

#ifdef PTHREAD_MUTEXTYPE_RECURSIVE
bool Mutex::TryEnterMutex(void)
{
	return (pthread_mutex_trylock(&_mutex) == 0) ? true : false;
}

#else
void Mutex::EnterMutex(void)
{
	if(_level)
		if(_tid == getThread())
		{
			++_level;
			return;
		}
	pthread_mutex_lock(&_mutex);
	++_level;
	_tid = getThread();
}

bool Mutex::TryEnterMutex(void)
{
	if(_level)
	{
		if(_tid == getThread())
		{
			++_level;
			return true;
		}
		else
			return false;
	}
	if ( pthread_mutex_trylock(&_mutex) != 0 )
		return false;
	else
	{
		_tid = getThread();
		++_level;
		return true;
	}
}

void Mutex::LeaveMutex(void)
{
	if(_level > 1)
	{
		--_level;
		return;
	}
	--_level;
	_tid = NULL;
	pthread_mutex_unlock(&_mutex);
}
#endif

MutexCounter::MutexCounter(int initial) : Mutex()
{
	counter = initial;
};

int operator++(MutexCounter &mc)
{
	int rtn;

	mc.EnterMutex();
	rtn = mc.counter++;
	mc.LeaveMutex();
	return rtn;
}

int operator--(MutexCounter &mc)
{
	int rtn = 0;
	
	mc.EnterMutex();
	if(mc.counter)
	{
		rtn = --mc.counter;
		if(!rtn)
			throw(mc);
	}
	else	// assume called in the catch
		mc.LeaveMutex();
	mc.LeaveMutex();
	return rtn;
}

#ifdef	HAVE_ATOMIC

AtomicCounter::AtomicCounter()
{
	atomic.counter = 0;
}

AtomicCounter::AtomicCounter(int value)
{
	atomic.counter = value;
}

int AtomicCounter::operator++(void)
{
	atomic_inc(&atomic);
	return atomic_read(&atomic);
}

int AtomicCounter::operator--(void)
{
	int chk = atomic_dec_and_test(&atomic);
	if(chk)
		return 0;
	chk = atomic_read(&atomic);
	if(!chk)
		++chk;
	return chk;
}

int AtomicCounter::operator+=(int change)
{
	atomic_add(change, &atomic);
	return atomic_read(&atomic);
}

int AtomicCounter::operator-=(int change)
{
	atomic_sub(change, &atomic);
	return atomic_read(&atomic);
}

int AtomicCounter::operator+(int change)
{
	return atomic_read(&atomic) + change;
}

int AtomicCounter::operator-(int change)
{
	return atomic_read(&atomic) - change;
}

int AtomicCounter::operator=(int value)
{
	atomic_set(&atomic, value);
	return atomic_read(&atomic);
}

bool AtomicCounter::operator!(void)
{
	int value = atomic_read(&atomic);
	if(value)
		return false;
	return true;
}

AtomicCounter::operator int()
{
	return atomic_read(&atomic);
}

#else

AtomicCounter::AtomicCounter()
{
	counter = 0;
}

AtomicCounter::AtomicCounter(int value)
{
	counter = value;
}

int AtomicCounter::operator++(void)
{
	int value;

	++lock;
	value = ++counter;
	--lock;
	return value;
}

int AtomicCounter::operator--(void)
{
	int value;
	++lock;
	value = --counter;
	--lock;
	return value;
}

int AtomicCounter::operator+=(int change)
{
	int value;
	++lock;
	counter += change;
	value = counter;
	--lock;
	return value;
}

int AtomicCounter::operator-=(int change)
{
	int value;
	++lock;
	counter -= change;
	value = counter;
	--lock;
	return value;
}

int AtomicCounter::operator+(int change)
{
	int value;
	++lock;
	value = counter + change;
	--lock;
	return value;
}

int AtomicCounter::operator-(int change)
{
	int value;
	++lock;
	value = counter - change;
	--lock;
	return value;
}

AtomicCounter::operator int()
{
	int value;
	++lock;
	value = counter;
	--lock;
	return value;
}

int AtomicCounter::operator=(int value)
{
	int ret;
	++lock;
	ret = counter;
	counter = value;
	--lock;
	return ret;
}

bool AtomicCounter::operator!(void)
{
	int value;
	++lock;
	value = counter;
	--lock;
	if(value)
		return false;
	return true;
}

#endif
/** EMACS **
 * Local variables:
 * mode: c++
 * c-basic-offset: 8
 * End:
 */
