/* xbm2back.c
   Author: Marko Meyer
   Date: 09/30 1995
   Time-stamp: <95/11/14 20:24:42 mme>
   
   A tool for creating .lrn and .mst files from .xbm files. 
   Calling convention is:

   xbm2back [-T mapped_size -F mst/lrn_name -C -R -V] xbm_dir

   -T is for setting a size [xdimxydim] to map the xbm's to;
      if not specified, the size of the first xbm in xbm_dir is taken.
   -F specifies a name for the mst/lrn-file;
      if not specified, the name of the xbm_dir is taken.
   -C take seperate columns
   -R take seperate rows
      if none of the last two is specified, xbm2back will take separate rows.
      if both are specified, the xbm's are read 'pixel by pixel' and written
      as well, whereas the pixels are ordered after the rows.
   -V turn on verbose mode, otherwise the program will not tell you anything,
      but errors.

   -N normalize on [0,1].

   If you omit the xbm_dir, an usage - information will be printed.

   This program is delivered mainly with BACKNET. Please check doc/DISCLAIMER
   and all the other README and doc - files there.

   This program is (C) Marko Meyer 1995

*/

#include<stdio.h>
#include<string.h>
#include<dirent.h>
#include<sys/types.h>
#include<sys/stat.h>
#include<fcntl.h>
#include<backnet.h>

char usage[] = 
"usage: xbm2back [-L xdimxydim -M xdimxydim -F file_name -CRVN] xbm_dir\n";

char copyright[] = 	
"xbm2back V 0.1  (C) Marko Meyer 1995.\n";

char *l_mapxy = NULL;
char *m_mapxy;
int l_mapx = 0;
int l_mapy = 0;
int m_mapx = 0;
int m_mapy = 0;

int c_set = 0;
int r_set = 0;
int cr_set = 0;
int v_set = 0;
int n_set = 0;
char *f_name = NULL;
char *d_name = NULL;
char *p_name = NULL;

#define min(a,b) ((a) < (b) ? (a) : (b))


#define NO_ERROR 0  /* Just in case we need some 'good' return value. */
#define GEN_ERROR (-1) /* some General Failure ;-] */
#define SZ_MATCHES 0 /* For 'get_xbm_size(...)'. */
#define SZ_NTMATCH (-1) /* For 'get_xbm_size(...)'.*/

#define MAX_XBM_LLEN 80  /* Maybe this is not a standard value; it works. */


int 
howtouse()
{
	printf("%s\n",usage);
	exit(1);
}


void
comment(char *com_str)
{
	if(v_set && (com_str!=NULL)) fprintf(stderr,"%s: %s\n",p_name,com_str);
}

void
write_header(FILE *file)
{
	char cm_line[LINELENGTH];
	
	sprintf(cm_line, "# Created by %s\n", copyright);
	
	fputs(cm_line,file);

	sprintf(cm_line, "%d\n%d\n", 0, 0);
	fputs(cm_line,file);
}

int
write_pattern_sect(FILE *file, char **values, int h, int w,
				   int y, int x, char *origin)
{
	/* We write a pattern_section to a either mst or lrn file. (This is not
	   important to us, because the algorithmic things are the same. */

	char cm_line[LINELENGTH];
	double d_val;
/*	int i_val = 0;
	double **dvals;
	int cpos_x, cpos_y, dpos_x, dpos_y;
	int bi_cnt_y, bi_cnt_x;
	int bi_place_y, bi_place_x;*/
	int i,j;
	
	sprintf(cm_line, "# from: %s\n",origin);
	fputs(cm_line,file);

	/* There are three cases: c_set is set, r_set is set or cr_set is set.
	   We think about the last at first. */

	if( cr_set )
	{
		/* We have to write all the values into the file. There's some 
		   problem when having less fields than values. So we must do
		   something. */

		if( (h==y) && (w==x))
		{
			/* We have to write. */
			for(i = 0 ; i < h; i++)
				for(j = 0; j < w; j++)
				{
					sprintf(cm_line,"%e\n", (double) values[i][j]);
					fputs(cm_line,file);
				}
		}
		else
		{
			/* This case  is not supported, yet. */
			comment("Entered unsupported case L143");
		}
		return NO_ERROR;
	}

	if(c_set)
	{
		if( (h == y) && (w == x) )
		{
			for(i = 0; i < x; i++)
			{
				d_val = 0;
				for(j = 0; j < y; j++)
					d_val += (values[j][i] << j);
				
				if(n_set) d_val /= ((1 << y) - 1);
				sprintf(cm_line,"%e\n", d_val);
				fputs(cm_line,file);
			}
		}
		else
		{
			/* This case is not supported, yet. */
			comment("Entered unsupported case L166");
		}
		return NO_ERROR;
		
	}
	
	if(r_set)
	{
		if( (h == y) && (w == x))
		{
			for( i = 0; i < y; i++)
			{
				d_val = 0;
				for(j = 0; j < x; j++)
					d_val += (values[i][j] << j);
				if(n_set) d_val /= ((1 << x) - 1);
				
				sprintf(cm_line,"%e\n",d_val);
				fputs(cm_line,file);
			}
		}
		else
		{
			/* This case is not supported, yet. */
			comment("Entered unsupported case L190");
		}
		
		return NO_ERROR;
		
	}
	
	/* Should never get this far. */
	
	return GEN_ERROR;
}


int
get_xbm_size(FILE *xbm_file, int *height, int *width)
{
	/* This one gets the size of the image contained in the xbm_file.
	   We just have two ways: we can rely on the structure of XBM or we
	   can rely just on the entries in a xbm file. I decided to do the last.
	*/

	/* As far as I know, height and width of an xbm image are stored as
	   ' #define xxxx_{height | width} NUMBER'. 
	   As far as I see, there's no rule for the place of this lines within
	   the file [I haven't seen any xbm-standard, but I may have been not too
	   carefully looking for it.], although they usually seem to be in the 
	   first two lines. I don't rely on the place, but on the structure.
	   So let's go. */

	char buf[MAX_XBM_LLEN];
	char Error[LINELENGTH];
	char *tok;
	int found = 0;

	rewind(xbm_file);
	/* Now we are at the beginning of the file. We should try to find a 
	   line starting with a "#define" - token. */
	do
	{
		if(fgets(buf, MAX_XBM_LLEN, xbm_file) == NULL)
		{
			/* We've got either an Error or EOF. */
			if(errno)
			{
				/* Seems to be an Error. */
				sprintf(Error,"%s: Error on reading xbm_file",p_name);
				perror(Error);
				exit(-1);
			}
			else
			{
				/* What we do on EOF? 
				   Hmm, that's difficult, since we don't expect to get
				   so far. We just return a GEN_ERROR :-). */
				comment("Unexpected EOF in xbm_file.\n");
				return GEN_ERROR;
			}
			
		}
		else
		{
			/* The buffer is filled with a line. We check for "#define". 
			   We assume that this is the first token, while tokens are
			   separated by ' '. 
			   This should be reliable, because the #define statement is 
			   like in C, and so there's some standard for this.
			   */
			
			tok = strtok(buf," ");
			if(tok != NULL)
			{
				if( strcmp(tok, "#define") == 0)
				{
					/* We've got it. Let's look for the second token. */
					tok = strtok(NULL, " ");
					if( strstr(tok, "_height") != NULL)
					{
						/* We've got the height. */
						tok = strtok(NULL, " ");
						*height = atoi(tok);
					}
					else
						if(strstr( tok, "_width") != NULL)
						{
							/* We've got the width. */
							tok = strtok(NULL, " ");
							*width = atoi(tok);
						}
					found++;
				}
			}
		}
		
	}
	while(found < 2);
	return NO_ERROR;
}

char **
read_xbm_patterns(FILE *source, int height, int width)
{
	char line[MAX_XBM_LLEN];
	char *token;
	char tok;
	int byte_cnt;
	int newline = 0;
	int i,j,k;
	char **dest;
 	
	/* We rewind and go on, until the first token in a line is 'static'.
	   The next line is the first pattern line, we assume. */

	do
	{
		if(fgets(line, MAX_XBM_LLEN, source) == NULL)
		{
			/* We've got EOF or an error. */
			if(errno)
			{
				perror("Error at fgets");
				exit(-1);
				
			}
			else
			{
				comment("Unexpected EOF in File.");
				return NULL;
			}
		}
		
		token = strtok(line, " ");
		if(token == NULL) continue;
	}
	while(strstr(token," ") != NULL);

	/* If we're here, we've got the correct line. The next one is it.*/
	if(fgets(line, MAX_XBM_LLEN, source) == NULL)
	{
		if(errno)
		{
			perror("Error at 2nd fgets");
			exit(-1);
		}
		else
		{
			comment("Unexpected EOF in file.");
			return NULL;
		}
	}
	newline = 1;
	byte_cnt = (width + 7) / 8;
	
	dest = (char **) calloc(height, sizeof(char*));
	
	for(i = 0; i < height; i++)
	{
	    dest[i] = (char *) calloc(width, sizeof(char));

		for(j = 0; j < byte_cnt; j++)
		{
			/* Get the token from line (they are comma separated, I believe
			   this we can rely on). */
			
			do
			{
				if(newline)
				{
					newline = 0;
					token = strtok(line, ",");
				}
				else token = strtok(NULL, ",");
				
				/* We've got the token or NULL, if this is behind the last. */
				
				if(token == NULL)
				{
					if(fgets(line, MAX_XBM_LLEN, source) == NULL)
					{
						if(errno)
						{
							perror("Error at 3rd fgets");
							exit(-1);
						}
						else
						{
							comment("Unexpected EOF in file.");
							return NULL;
						}

					}
					newline = 1;
				}
			}
			while(token == NULL);
			
			while(*token == ' ') token++;

			/* We've got a valid token now. Let's see, what we do with it. */

			if(strstr(token, "};") == NULL)
			{
				/* This is not the end, good. */
				/* The token now should be numeric, we first transform it. */

				tok = (char) strtol(token, NULL, 0);
				
				/* Now we have a byte, that we have to transform into the
				   bits, which should be set in our matrix. */
				
				for(k = j * 8; k < (min( (j+1)*8, width)); k++)
					dest[i][k] = (char) ((tok & (1 << (k - (j * 8) ))) 
										 >> (k-(j * 8)));
			}
		}
	}
	return dest;
}

					
				

int
process_file(FILE *file, int *mapx, int *mapy, char *s)
{
	FILE *str_xbm;
	char Error[80];
	int orig_width;
	int orig_height;
/*	int norm_fact, sum; */
	int i;
	

	char **from_xbm;
	
	comment("Opening file.");
	
	if((str_xbm = fopen(s, "r")) == NULL)
	{
		sprintf(Error,"%s: Error opening %s",p_name,s);
		perror(Error);
		exit(-1);
	}
	
	comment("File open.");
	
	if(get_xbm_size(str_xbm, &orig_height, &orig_width) != NO_ERROR)
		return GEN_ERROR;

	sprintf(Error,"Got height: %d and width: %d.", orig_height, orig_width);
	comment(Error);
	
	if(!(*mapx && *mapy))
	{
		comment("Setting m_mapx and m_mapy!");
		*mapx = orig_width;
		*mapy = orig_height;
	}

	
	/* We're now ready for reading in the patterns. The orig_* things help
	   us; they give hints for how many reads are needed. We read the values
	   into a byte matrix. */

	
	if((from_xbm = read_xbm_patterns(str_xbm, orig_height, orig_width) )
	   == NULL)
		return GEN_ERROR;
	
	comment("Read XBM Patterns to Matrix.");
	if((orig_height != *mapy) || (orig_width != *mapx))
	{
		comment("Reached an unsupported case; orig_ and map_ defer!");
		return GEN_ERROR;
	}
	
	write_pattern_sect(file, from_xbm, 
					   orig_height, orig_width,
					   *mapy,*mapx,s);

	for(i = 0; i < orig_height; i++)
		free(from_xbm[i]);
	free(from_xbm);
	
	comment("Wrote Patterns to file.");
	fclose(str_xbm);
	return NO_ERROR;
	
}

	
	
int
main(int argc, char **argv)
{

	register char *ch, *ch2;
	register DIR *xbm_dir;
	register struct dirent *xbm_file;
	char Error[80];
	FILE *str_mst, *str_lrn, *test;
	

	p_name = argv[0];

	while((--argc > 0) && ((*++argv)[0] == '-'))
		for(ch = argv[0]+1; *ch != '\0'; ch++)
			switch(*ch)
			{
			  case 'L':

				comment("Flag -L not supported yet.");
				break;
				
				if((--argc <= 0) || ((*++argv)[0] == '-')) 
				{
					fprintf(stderr,"-L requires a parameter.\n");
					howtouse();
				}
				else
					l_mapxy = *argv;
				break;
				
			  case 'M':
				
				comment("Flag -M not supported yet.");
				break;
				
				if((--argc <= 0) || ((*++argv)[0] == '-'))
				{
					fprintf(stderr,"-M requires a parameter.\n");
					howtouse();
				}
				else
					m_mapxy = *argv;
				break;
			  
			  case 'F':
				if((--argc <= 0) || ((*++argv)[0] == '-'))
				{
					fprintf(stderr,"-F requires a parameter.\n");
					howtouse();
				}
				else
					f_name = *argv;
				break;
				
			  case 'C':
				c_set = 1;
				break;
				
			  case 'R':
				r_set = 1;
				break;
				
			  case 'V':
				v_set = 1;
				break;

			  case 'N':
				n_set = 1;
				break;
				
			  default:
				fprintf(stderr,
						"%s: Unknown command line option %c\n",p_name,*ch);
				howtouse();
				break;
			}
		
	if(argc <= 0) 
	{
		fprintf(stderr,"%s: xbm_dir not specified!\n",p_name);
		howtouse();
	}

	if(c_set && r_set) cr_set = 1;

	if(!(c_set | r_set)) r_set = 1;
	
	comment(copyright);

/* Although we coded something we don't support this by now. 

	if(l_mapxy)
	{
		if((ch = strchr(l_mapxy, 'x')) == NULL)
		{
			fprintf(stderr,"%s: Format of parameter to -L not correct: %s\n",
					p_name,l_mapxy);
			howtouse();
		}
		*ch = '\0';
		
		if(((l_mapx = atoi(l_mapxy)) == 0) || ((l_mapy = atoi(++ch)) == 0))
		{
			fprintf(stderr,"%s: Format of parameter to -L not correct: %s\n",
					p_name,l_mapxy);
			howtouse();
		}

	    sprintf(Error,"Got a different learn mapsize: %d by %d!\n",
				l_mapx,l_mapy);
		comment(Error);
	}

	if(m_mapxy)
	{
		if((ch = strchr(m_mapxy, 'x')) == NULL)
		{
			fprintf(stderr,"%s: Format of parameter to -M not correct: %s\n",
					p_name,m_mapxy);
			howtouse();
		}
		*ch = '\0';
		
		if(((m_mapx = atoi(m_mapxy)) == 0) || ((m_mapy = atoi(++ch)) == 0))
		{
			fprintf(stderr,"%s: Format of parameter to -M not correct: %s\n",
					p_name,m_mapxy);
			howtouse();
		}

	    sprintf(Error,"Got a different pattern mapsize: %d by %d!\n",
				m_mapx,m_mapy);
		comment(Error);
	}

*/
	comment("Checking xbm_dir ... ");
	
	if((xbm_dir = opendir(argv[0])) == NULL)
	{
		sprintf(Error,"%s: Error at opening %s",p_name,argv[0]);
		perror(Error);
		exit(-1);
	}

	comment("Ok! \n");
	comment("Creating files ...");

	if(!f_name)
	{
		ch = strrchr(argv[0],'/');
		if(!ch) ch = argv[0];
		else ch++;
		if((f_name = (char *) calloc(strlen(ch), sizeof(char))) == NULL)
		{
			fprintf(stderr,"%s: Virtual Memory exhausted\n",p_name);
			exit(1);
		}
		strcpy(f_name,ch);
	}

	if((ch = (char *) calloc(strlen(f_name) + 5, sizeof(char))) == NULL)
	{
		fprintf(stderr,"%s: Virtual Memory exhausted\n",p_name);
		exit(1);
	}
	
	strcpy(ch, f_name);
	strcat(ch, MST_EXTENSION);
	
	if((str_mst = fopen(ch, "w") ) == NULL)
	{
		sprintf(Error,"%s: Error opening %s for writing",p_name,f_name);
		perror(Error);
		exit(-1);
	}
	
	write_header(str_mst);
	
	comment("MST Ok ...");

	strcpy(ch,f_name);
	strcat(ch, LRN_EXTENSION);

	if((str_lrn = fopen(ch, "w") ) == NULL)
	{
		sprintf(Error,"%s: Error opening %s for writing",p_name,f_name);
		perror(Error);
		exit(-1);
	}

	write_header(str_lrn);
	
	free(ch);
	comment("LRN Ok !\n");
	
	while((xbm_file = readdir(xbm_dir)) != NULL)
	{
		if(strstr(xbm_file->d_name,".xbm") == NULL ) continue;
		if(strstr(xbm_file->d_name,"~.") != 0) continue;

		if((ch = (char *) calloc(strlen(argv[0])+strlen(xbm_file->d_name)+2,
								 sizeof(char))) == NULL)
		{
			fprintf(stderr,"%s: Virtual Memory exhausted\n",p_name);
			exit(1);
		}

		if((ch2 = (char *) calloc(strlen(argv[0])+strlen(xbm_file->d_name)+3,
								 sizeof(char))) == NULL)
		{
			fprintf(stderr,"%s: Virtual Memory exhausted\n",p_name);
			exit(1);
		}
		
		strcpy(ch,argv[0]);
		strcat(ch,"/");
		strcat(ch,xbm_file->d_name);

		strncpy(ch2,ch,strlen(ch) - 4);
		strcat(ch2,"~.xbm");

		sprintf(Error,"Looking for %s\n",ch2);
		comment(Error);
		
		if((test=fopen(ch2,"r")) == NULL)
		{
			perror("fopen error");
		    exit(-1);
		}
		

		fclose(test);

		sprintf(Error,"Processing %s\n",ch);
		comment(Error);
		
		process_file(str_mst,&m_mapx,&m_mapy,ch);
		process_file(str_lrn,&l_mapx,&l_mapy,ch2);
		
		free(ch);
		free(ch2);
	}
	
	fclose(str_mst);
	fclose(str_lrn);
	
	return 0;
}




