//  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: CNCL.c,v 0.31 1996-08-07 17:54:21+02 steppler Exp $
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#include "CNCL.h"

#include <iostream.h>
#include <strstream.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <new.h>



/*
 * Error handling:
 *   Default is fatal errors.
 */
CNCL::ErrorType CNCL::error_flag = CNCL::err_fatal;

// Set/get error handler setting
CNCL::ErrorType CNCL::get_error() 
{
    return error_flag;
}

CNCL::ErrorType CNCL::set_error(ErrorType new_error)
{
    ErrorType old_error = error_flag;
    error_flag = new_error==err_default ? err_fatal : new_error;
    return old_error;
}

// Set function called for terminating program
void CNCL::set_exit_handler(void (*func)())
{
    exit_handler = func;
}



/*
 * Default function for error exits
 */

void CNCL::default_exit_handler()
{
    cerr << "CNCL: program "
	 << ( error_flag == err_abort ? "aborted" : "terminated" )
	 << endl;

    if(error_flag == err_abort)
	abort();
    else
	exit(1);
}

/*
 * CNCL exit handler
 */

void (* CNCL::exit_handler)() = default_exit_handler;



/*
 * Error handler:
 *   Print out messages on cerr
 */

void CNCL::error(ErrorType err,
		 const char *msg1, const char *msg2, const char *msg3,
		 const char *msg4, const char *msg5, const char *msg6)
{
    if(err == err_default)
	err = error_flag;
    
    if(err != err_ignore)
    {
	if(msg1 == NIL)
	{
	    switch(err)
	    {
	    case err_warning:
		cerr << "CNCL warning: ";	break;
	    case err_info:
		cerr << "CNCL: ";		break;
	    default:
		cerr << "CNCL error: ";		break;
	    }
	}
	else
	    cerr << msg1;
	if(msg2)
	    cerr << msg2;
	if(msg3)
	    cerr << msg3;
	if(msg4)
	    cerr << msg4;
	if(msg5)
	    cerr << msg5;
	if(msg6)
	    cerr << msg6;
	if(str_stream)
	{
	    cerr << str_stream->str();
	    delete str_stream;
	    delete [] str_buf;
	    str_stream = NIL;
	    str_buf = NIL;
	}
	cerr << endl;
    }
    
    if(err==err_fatal || err==err_abort)
	exit_handler();
}



/*
 * Error message with default handling
 */

void CNCL::error(const char *msg1, const char *msg2, const char *msg3,
		 const char *msg4, const char *msg5, const char *msg6)
{
    error(err_default, msg1, msg2, msg3, msg4, msg5, msg6);
}



/*
 * Error message with fatal handling
 */

void CNCL::fatal(const char *msg1, const char *msg2, const char *msg3,
		 const char *msg4, const char *msg5, const char *msg6)
{
    error(err_fatal, msg1, msg2, msg3, msg4, msg5, msg6);
}



/*
 * Error message with warning handling
 */

void CNCL::warning(const char *msg1, const char *msg2, const char *msg3,
		   const char *msg4, const char *msg5, const char *msg6)
{
    error(err_warning, msg1, msg2, msg3, msg4, msg5, msg6);
}



/*
 * (Non-)Error message with info handling
 */

void CNCL::info(const char *msg1, const char *msg2, const char *msg3,
		const char *msg4, const char *msg5, const char *msg6)
{
    error(err_info, msg1, msg2, msg3, msg4, msg5, msg6);
}



/*
 * strstream for additional output with error(), ...
 */
char       *CNCL::str_buf = NIL;
ostrstream *CNCL::str_stream = NIL;


ostream& CNCL::msg()
{
    if(str_buf)
	delete [] str_buf;
    if(str_stream)
	delete str_stream;
    str_buf    = new char [STR_BUF_SIZE];
    // This results in a proper EOS '\0' even if `ends' isn't
    // pushed into the stream.
    memset(str_buf, 0, STR_BUF_SIZE);
    str_stream = new ostrstream(str_buf, STR_BUF_SIZE);
    
    return *str_stream;
}

void CNCL::clear()
{
    delete [] str_buf;
    str_buf = NIL;
    delete str_stream;
    str_stream = NIL;
}

#ifndef CNCL_NO_MATHERR			// matherr() may not be available

/*
 * matherr() handling
 */
extern "C"
#ifdef LIBGPP_24			// libg++ <=2.4 uses libm_execption
int matherr(struct libm_exception *x)
#else
int matherr(struct exception *x)
#endif
{
    char buffer[128];

    ostrstream strm(buffer, sizeof(buffer));
    char *err;
    int args = 1;
    
    if(!strcmp(x->name, "atan2")  ||  !strcmp(x->name, "copysign")  ||
       !strcmp(x->name, "drem")   ||  !strcmp(x->name, "fmod")      ||
       !strcmp(x->name, "frexp")  ||  !strcmp(x->name, "hypot")     ||
       !strcmp(x->name, "ldexp")  ||  !strcmp(x->name, "modf")      ||
       !strcmp(x->name, "pow")    ||  !strcmp(x->name, "scalb")     ||
       !strcmp(x->name, "yn")                                         )
	args = 2;
    
    switch(x->type)
    {
    case DOMAIN:
	err = "DOMAIN";
	break;
    case SING:
	err = "SING";
	break;
    case OVERFLOW:
	err = "OVERFLOW";
	break;
    case UNDERFLOW:
	if(!strcmp(x->name, "exp"))	// exp() underflow = 0 is o.k.
	    return 0;
	err = "UNDERFLOW";
	break;
#ifdef TLOSS
    case TLOSS:
	err = "TLOSS";
	break;
#endif
#ifdef PLOSS
    case PLOSS:
	err = "PLOSS";
	break;
#endif
    default:
	err = "unknown";
	break;
    }

    if(args == 2)
	strm << x->name << "(" << x->arg1 << ", " << x->arg2 << "): "
	     << err << " error" << ends;
    else
	strm << x->name << "(" << x->arg1 << "): "
	     << err << " error" << ends;

    CNCL::error(NIL, "matherr: ", buffer);
    
    return 1;
}

#endif /**!CNCL_NO_MATHERR**/



/*
 * new handler called when memory is exhausted
 */
void CNCL::new_handler()
{
    fatal(NIL, "memory exhausted");
    /**NOT REACHED**/
}



/*
 * CNCL initialization, called once
 */

bool CNCL::initialized = FALSE;

void CNCL::initialize()
{
    /* Install new handler */
    set_new_handler(CNCL::new_handler);
    
    initialized = TRUE;
}



/*
 * Global CNCL object to enforce initialization even if no CNCL object
 * is created elsewhere.
 */
static CNCL dummy;
