// -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: T_array3.cP,v 1.2 1996-08-07 18:11:24+02 steppler Exp $
 *
 * Class: CNArray3Double  --- 3 dimensional double array class
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#include "Array3Double.h"


/***** Constructors **********************************************************/

CNArray3Double::CNArray3Double(size_t r, size_t c, size_t d, double def)
{
    def_val = def;
    init(r, c, d);
}


CNArray3Double::CNArray3Double(const CNArray3Double &a)
{
    size_t i, j, k;
    
    init(a.arows, a.acols, a.adepth);
    for(i = 0; i < arows; i++)
        for(j = 0; j < acols; j++)
            for(k = 0; k < adepth; k++)
                array[i][j][k] = a.array[i][j][k];
}


CNArray3Double::~CNArray3Double()
{
    CNArray3Double::delete_array();
}


CNArray3Double &CNArray3Double::operator = (const CNArray3Double &a)
{
    size_t i, j, k;
    
    delete_array();
    init(a.arows, a.acols, a.adepth);
    for(i = 0; i < arows; i++)
        for(j = 0; j < acols; j++)
            for(k = 0; k < adepth; k++)
                array[i][j][k] = a.array[i][j][k];

    return *this;
}    


/***** public functions ******************************************************/

void CNArray3Double::size(size_t r, size_t c, size_t d)  // Set size of array
{
    size_t i, j, k;
    
    if (array == NIL)
    {
	init(r, c, k);
	return;
    }
    if(r == arows && c == acols && d == adepth)	// Nothing to do
	return;
    if(r == arows)			// Resize rows only
    {
        for(i = 0; i < arows; i++)
            array[i].size(c, d);
        acols = c;
        adepth = d;
        return;
    }
    
    CNArray2Double *temp = array;

    array = new CNArray2Double[r];
    for (i = 0; i < r; i++)
    {
        array[i].size(c, d);
        for(j = 0; j < c; j++)
            for(k = 0; k < d; k++)
                if(i < arows && j < acols && k < adepth)
                    array[i][j][k] = temp[i][j][k];
                else
                    array[i][j][k] = def_val;
    }
    delete [] temp;
    arows = r;
    acols = c;
    adepth = d;
}


/***** Private functions *****************************************************/

void CNArray3Double::init(size_t r, size_t c, size_t d)
{
    size_t i, j, k;
    
    arows = r;
    acols = c;
    adepth = d;
    
    array = new CNArray2Double[r];
    for(i = 0; i < r; i++)
    {
        array[i].size(c, d);
        for(j = 0; j < c; j++)
            for(k = 0; k < d; k++)
                array[i][j][k] = def_val;
    }
}


void CNArray3Double::delete_array()
{
    arows = acols = adepth = 0;
    if(array)
	delete [] array;
    array = NIL;
}


/***** Default I/O member function for CNCL classes **************************/

// CNNormal output
void CNArray3Double::print(ostream &strm) const
{
    strm << "[ " << endl;
    for(size_t i=0; i<arows; i++)
    {
	strm << "  " << array[i];
	if(i != arows-1)
	    strm << " ,";
	strm << endl;
    }
    strm << "]" << endl;
}

// Debug output
void CNArray3Double::dump(ostream &strm) const
{
    strm << "CNArray3Double { $Revision: 1.2 $ ... " << endl;
    strm << "rows=" << arows << " cols=" << acols << " depth=" << adepth
         << " array=" << hex << (unsigned long)(array) << dec << endl;
    strm << "values=" << endl << this << endl;
    strm << "}" << endl;
}



/***** CNCL stuff for type ***************************************************/

// Describing object for class CNArray3Double
static CNClass CNArray3Double_desc("CNArray3Double", "$Revision:",
			    CNArray3Double::new_object);

// "Type" for type checking functions
CNClassDesc CN_ARRAY3DOUBLE = &CNArray3Double_desc;
