//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: HashDynamic.c,v 0.31 1996-08-07 17:55:04+02 steppler Exp $
 *
 * CNClass: CNHashDynamic --- hash tables with dynamic number of entries
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#include "HashDynamic.h"

CNHashDynamic::CNHashDynamic(unsigned long cap)
{
    init_table(cap);
}

CNHashDynamic::CNHashDynamic(CNParam *)
{
    init_table(DEFAULT_HASH_TABLE_CAPACITY);
}

CNHashDynamic::~CNHashDynamic()
{
    delete (table);
}

void CNHashDynamic::store_key(CNKey *k)
{
    if (k == NIL) 
	fatal("You must provide a valid key, when trying to store one!\n");

    hash(k, k->hash(capacity));
}

void CNHashDynamic::hash(CNKey *k, unsigned long hash_value)
{
    unsigned long pos = hash_value;
    
    if (pos > capacity) 
	fatal("hash() has been passed a hash value,\n",
	      "which exceeds the hash table's capacity!\n");
    
    // collision?
    if (table[pos].he_CNKey != NIL)
	pos = collision(pos);

    table[pos].he_CNKey = k;
    table[pos].he_HashValue = hash_value;
    entries++;

    // Are 3/4 of the hash table's capacity exhausted?
    // --> resize dynamicly
    if (entries * 4 >= capacity * 3)
	resize();
}

CNKey *CNHashDynamic::get_key(const CNKey *k) const
{
    unsigned long pos;
    
    if (k == NIL)
	fatal("You must provide a valid key, in order to get a key\n");

    pos = find_key(k, k->hash(capacity));

    // find_key failed
    if (pos == capacity)
	return NIL;
    
    return table[pos].he_CNKey;
}

unsigned long CNHashDynamic::find_key(const CNKey *k, unsigned long hash_value)
    const
{
    unsigned long i, pos;
    
    if (k == NIL)
	fatal("You must provide a valid key, when trying to find a key\n");
    
    // quadratic search
    for (i = 0; i <= capacity - 1; i++) {
	pos = (hash_value + i * i) % capacity;
	
	// hash table does not contain the key searched for
	if (table[pos].he_CNKey == NIL) 
	    return capacity;
	// key found
	else if (table[pos].he_CNKey->compare(k)) 
	    return pos;
    }
    
    // unable to find key via quadratic searching
    // use linear searching
    for (i = 1; i <= capacity - 1; i++) {
	pos = (hash_value + i) % capacity;
	// hash table does not contain the key searched for
	if (table[pos].he_CNKey == NIL) 
	    return capacity;
	// key found
	else if (table[pos].he_CNKey->compare(k)) 
	    return pos;
    }

    // unable to find key
    return capacity;
}

CNObject *CNHashDynamic::get_object(const CNKey *k) const
{
    CNKey *key;
    
    key = get_key(k);
    if (key != NIL)
	return key->get_object();
    
    return NIL;
}

bool CNHashDynamic::reset()
{
    unsigned long i;

    if (!entries) {
	warning("Unable to reset() already empty hash table!\n");
	return (FALSE);
    }
    
    entries = 0;
    for (i = 0; i < capacity; i++)
	table[i].he_CNKey = NIL;

    return (TRUE);
}

bool CNHashDynamic::reset_absolutely()
{
    unsigned long i;

    if (entries) 
	for (i = 0; i < capacity; i++)
	    delete table[i].he_CNKey;

    return reset();
}

bool CNHashDynamic::reset_absolutely_w_obj()
{
    unsigned long i;

    if (entries) 
    {
	for (i = 0; i < capacity; i++)
	{
	    if (table[i].he_CNKey)
		delete table[i].he_CNKey->get_object();
	    delete table[i].he_CNKey;
	}
    }
    return reset();
}

bool CNHashDynamic::delete_key(const CNKey *k)
{
    return remove_key(k, clearKeyOnly);
}

bool CNHashDynamic::delete_key_absolutely(const CNKey *k)
{
    return remove_key(k, freeMemOfKey);
}

bool CNHashDynamic::delete_key_absolutely_w_obj(const CNKey *k)
{
    return remove_key(k, freeMemOfKeyAndObject);
}

bool CNHashDynamic::remove_key(const CNKey *k, rmKeyType remove_absolutely)
{
    unsigned long i, pos;
    struct HashEntry *old_table, *help_table;

    if (k == NIL)
	fatal("You must provide a valid key, in order to delete a key\n");
    
    if ((pos = find_key(k, k->hash(capacity))) == capacity) {
	warning("Unable to delete object: CNObject not found!");
	return FALSE;
    }

    // delete entry
    if (remove_absolutely != clearKeyOnly)
    {
	if (remove_absolutely == freeMemOfKeyAndObject)
	{
	    delete table[pos].he_CNKey->get_object();
	}
	delete table[pos].he_CNKey;
    }

    table[pos].he_CNKey = NIL;
    entries = 0;

    old_table = new (struct HashEntry[capacity]);
    if (old_table == NIL) 
	fatal("Out of memory!\n");

    // init hash table	 
    for (i = 0; i < capacity; i++)
	old_table[i].he_CNKey = NIL;

    // exchange
    help_table = old_table;
    old_table = table;
    table = help_table;
    
    for (i = 0; i < capacity; i++) 
	if (old_table[i].he_CNKey != NIL) 
	    hash(old_table[i].he_CNKey, old_table[i].he_HashValue);

    delete old_table;
	    
    return TRUE;
}


/***** Private member functions **********************************************/
unsigned long CNHashDynamic::collision(unsigned long hash_value)
{
    unsigned long i, pos = hash_value;

    // calc new hash value
    // first try quadratic probing
    for (i = 1; i <= capacity - 1; i++) {
	pos = (hash_value + i * i) % capacity;
	// found a free position
	if (table[pos].he_CNKey == NIL) 
	    return (pos);
    }
    
    // unable to find a free postion via quadratic probing
    // use linear probing, which inevitably leads to clustering
    for (i = 1; i <= capacity - 1; i++) {
	pos = (hash_value + i) % capacity;
	// found a free position
	if (table[pos].he_CNKey == NIL) 
	    break;
    }
    
    return pos;
}

void CNHashDynamic::resize()
{
    unsigned long i, cap = capacity;
    struct HashEntry *old_table, *help_table;
    
    capacity = get_prime(capacity + 1);
    // capacity exceeds largest known prime number
    // just double old capacity
    if (capacity == cap + 1)
	capacity = cap * 2;

    old_table = new (struct HashEntry[capacity]);
    if (old_table == NIL) 
	fatal("Out of memory!\n");

    // init hash table	 
    for (i = 0; i < capacity; i++)
	old_table[i].he_CNKey = NIL;

    // exchange
    help_table = old_table;
    old_table = table;
    table = help_table;

    entries = 0;
    
    // hash all valid entries into new table
    for (i = 0; i < cap; i++) 
	if (old_table[i].he_CNKey != NIL) 
	    hash(old_table[i].he_CNKey, old_table[i].he_CNKey->hash(capacity));

    delete old_table;
}


unsigned long CNHashDynamic::get_prime(unsigned long cap)
{
    const unsigned long NumPrimes = 19;
    const unsigned long Primes[NumPrimes] = 
        { 3, 7, 17, 37, 79, 163, 331, 673, 1361,
	  2729, 5471, 10949, 21911, 43853, 87719,
	  175447, 350899, 701819, 999983 
	};
    unsigned long i;

    // find next prime number
    for (i = 0; i < NumPrimes; i++) {
	if (cap <= Primes[i])
	    return Primes[i];
    }
    // capacity exceeds largest known prime number
    return cap;
}
    
void CNHashDynamic::init_table(unsigned long cap)
{
    unsigned long i;

    capacity = (cap != DEFAULT_HASH_TABLE_CAPACITY) ? 
	        get_prime(cap) : cap;
    
    table = new (struct HashEntry[capacity]);
    if (table == NIL) 
	fatal("Out of memory!");

    // init hash table	 
    for (i = 0; i < capacity; i++)
	table[i].he_CNKey = NIL;

    entries = 0;
}


/***** Default I/O member function for CNCL classes **************************/

// CNNormal output
void CNHashDynamic::print(ostream &strm) const
{
    strm << "capacity " << capacity
	 << " entries " << entries
	 << endl;
}

// Debug output
void CNHashDynamic::dump(ostream &strm) const
{
    strm << "CNHashDynamic { $Revision $"
	 << " capacity " << capacity
	 << " entries " << entries	 
	 << " }" << endl;
}



/***** CNCL stuff for type information and exemplar objects ******************/

// Describing object for class CNHashDynamic
static CNClass CNHashDynamic_desc("CNHashDynamic", "$Revision $",
			  CNHashDynamic::new_object);

// "Type" for type checking functions
CNClassDesc CN_HASHDYNAMIC = &CNHashDynamic_desc;






