//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: SLList.c,v 1.6 1996-08-07 17:55:45+02 steppler Exp $
 *
 * Class: CNSLList --- Single Linked List
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 * 
 * As an exception to this rule you may use this template to generate
 * your own classes. This does not cause these classes to be covered by
 * the GNU Library General Public License. This exception does not
 * however invalidate any other reasons why the resulting program must be
 * covered by the GNU Library General Public License.
 *****************************************************************************/

#include <CNCL/Class.h>

#include "SLList.h"

CNSLList::CNSLList()          : pfirst(NIL), plast(NIL), count(0)
{}

CNSLList::CNSLList(CNParam* ) : pfirst(NIL), plast(NIL), count(0)
{}

CNSLList::CNSLList(const CNSLList& ) : pfirst(NIL), plast(NIL), count(0)
{
    fatal(NIL, "can't copy CNSLObject object");
}

CNSLList::~CNSLList()
{
    delete_all();
}

CNSLObject* CNSLList::first() const
{
    return pfirst;
}

CNSLObject* CNSLList::next(CNSLObject* link) const
{
    if (!link)
    {
	fatal("CNSLList", "Invalid pointer!");
    }
    return link->pnext;
}


CNSLObject* CNSLList::last() const
{
    return plast;
}

CNSLObject* CNSLList::prev(CNSLObject* link) const
{
    CNSLObject* p;

    for (p = pfirst; p && p->pnext != link;)
    {
	p = p->pnext;
    }

    return p;
}

// Check for empty list
bool CNSLList::empty() const
{
    return pfirst == NIL;
}

// Return # of items in list
unsigned long CNSLList::length() const
{
    return count;
}

CNSLObject* CNSLList::prepend(CNSLObject* link)
{
    if (!link)
    {
	fatal("CNSLList", "Invalid pointer!");
    }
    if (pfirst)
    {
	link->pnext = pfirst;
	pfirst       = link;
    }
    else
    {
	pfirst = plast = link;
	link->pnext = NIL;
    }
    count++;

    return link;
}

CNSLObject* CNSLList::prepend(CNObject* obj)
{
    return prepend(new CNSLObject(obj));
}

CNSLObject* CNSLList::prepend(CNObject& obj)
{
    return prepend(new CNSLObject(&obj));
}

CNSLObject* CNSLList::append(CNSLObject* link)
{
    if (!link)
    {
	fatal("CNSLList", "Invalid pointer!");
    }
    if (plast)
    {
	plast->pnext = link;
	link->pnext = NIL;
	plast = link;
    }
    else
    {
	pfirst = plast = link;
	link->pnext = NIL;
    }
    count++;

    return link;
}

CNSLObject* CNSLList::append(CNObject* obj)
{
    return append(new CNSLObject(obj));
}

CNSLObject* CNSLList::append(CNObject& obj)
{
    return append(new CNSLObject(&obj));
}

/*
 * Remove node from list, do NOT delete DLObject
 */
CNSLObject* CNSLList::remove_object(CNSLObject* pos)
{
    CNSLObject* p, * n;

    if (!pos)
    {
	fatal("CNSLList", "Invalid pointer!");
    }
    if (!pfirst)
	return NIL;
    for (n = p = pfirst; p != pos; n = p, p = p->pnext)
	;
    // element not found
    if (!p)
    {
	warning("CNSLList", "Cannot remove object! Object not found!");
	return NIL;
    }
    // remove first element
    else if (p == pfirst)
    {
	pfirst = pfirst->pnext;
	// removed single element
	if (p == plast)
	{
	    plast = NIL;
	}
	count--;
	return pfirst;
    }
    // remove last element
    else if (p == plast)
    {
	plast = n;
    }
    n->pnext = p->pnext;

    count--;
    
    return n;
}

/*
 * Remove node from list, delete DLObject
 */
CNSLObject* CNSLList::delete_object(CNSLObject* pos)
{
    CNSLObject* n;

    n = remove_object(pos);
    delete pos;
    
    return n;
}


/*
 * Delete the entire list, all SLObject nodes,
 * but NOT the referenced objects.
 */
void CNSLList::delete_all()
{
    CNSLObject* p,* n;
    
    for (p = pfirst; p != NIL;)
    {
	n = p->pnext;
	delete p;
	p = n;
    }
    pfirst = plast = NIL;
    count  = 0;
}

/*
 * Delete the entire list, all SLObject nodes,
 * AND the referenced objects.
 */
void CNSLList::delete_all_w_obj()
{
    CNSLObject* p, * n;
    
    for (p = pfirst; p;)
    {
	// First, delete object
	p->delete_object();
	n = p->pnext;
	delete p;
	p = n;
    }
    pfirst = plast = NIL;
    count  = 0;
}


CNSLObject* CNSLList::insert_after(CNSLObject* pos, CNObject* obj)
{
    return insert_after(pos, new CNSLObject(obj));
}

CNSLObject* CNSLList::insert_after(CNSLObject* pos, CNObject& obj)
{
    return insert_after(pos, new CNSLObject(&obj));
}

CNSLObject* CNSLList::insert_after(CNSLObject* pos, CNSLObject* obj)
{
    CNSLObject* p;
    
    if (!pos || !obj)
    {
	fatal("CNSLList", "Invalid pointer!");
    }
    for (p = pfirst; p != pos; p = p->pnext)
	;
    // element not found
    if (!p)
    {
	warning("CNSLList",
		"Cannot insert object! Referenced element not found!");
	return obj;
    }
    
    obj->pnext = pos->pnext;
    pos->pnext = obj;
    if (pos == plast)
    {
	plast = obj;
    }
    count++;
    
    return obj;
}

CNSLObject* CNSLList::insert_before(CNSLObject* pos, CNObject* obj)
{
    return insert_before(pos, new CNSLObject(obj));
}

CNSLObject* CNSLList::insert_before(CNSLObject* pos, CNObject& obj)
{
    return insert_before(pos, new CNSLObject(&obj));
}

CNSLObject* CNSLList::insert_before(CNSLObject* pos, CNSLObject* obj)
{
    if (!pos || !obj)
    {
	fatal("CNSLList", "Invalid pointer!");
    }
    if (pos != pfirst)
    {
	CNSLObject* p, * n;
	
	for (n = p = pfirst; p != pos; n = p, p = p->pnext)
	    ;
	// element not found
	if (!p)
	{
	    warning("CNSLList",
		    "Cannot insert object! Referenced element not found!");
	    return obj;
	}
	return insert_after(n, obj);
    }
    else
    {
	
	obj->pnext = pfirst;
	pfirst = obj;
	count++;
    }
    return obj;
}

/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNSLList::print(ostream& strm) const
{
    CNSLObject* p;

    for(p=first(); p; p=p->next())
	strm << p;
}

// Debug output
void CNSLList::dump(ostream& strm) const
{
    CNSLObject* p;
    
    strm << "CNSLList { $Revision: 1.6 $  count=" << count << endl;
    for(p=first(); p; p=p->next())
	p->dump(strm);
    strm<< " }" << endl;
}

// IOStream operator <<
ostream& operator << (ostream& strm, const CNSLList& obj)
{
    obj.print(strm);
    return strm;
}

ostream& operator << (ostream& strm, const CNSLList* obj)
{
    if(obj)
	obj->print(strm);
    else
	strm << "(NIL)";
    return strm;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNSLList
static CNClass CNSLList_desc("CNSLList", "$Revision: 1.6 $",
			    CNSLList::new_object);

// "Type" for type checking functions
CNClassDesc CN_SLLIST = &CNSLList_desc;
