//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: EventHIterator.c,v 0.30 1996-08-07 17:56:28+02 steppler Exp $
 *
 * Class: CNEventHIterator --- ...
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 * 
 * As an exception to this rule you may use this template to generate
 * your own classes. This does not cause these classes to be covered by
 * the GNU Library General Public License. This exception does not
 * however invalidate any other reasons why the resulting program must be
 * covered by the GNU Library General Public License.
 *****************************************************************************/

#include <CNCL/Class.h>

#include "EventHIterator.h"

// initialize iterator with last element of heap
CNEventHIterator::CNEventHIterator(CNEventHeapSched *s)
{
    sched = s;
    index = sched->heap_n;
    act_event = sched->heap[index];
    use_current = TRUE;
}



// traverse event heap (here neither the order of events nor uniqueness is
// guaranteed; a single event may be delivered more than once but at least
// once; of course an event will only be deleted once)
CNEvent *CNEventHIterator::next_event()
{
    if (use_current)
        use_current = FALSE;
    else
        act_event = --index ? sched->heap[index] : NIL;
        
    return act_event; 
}



// delete the event delivered last time, then reuse list position because
// the upheap() function may move a yet undelivered event to the current
// position. Never delete an event not yet delivered
void CNEventHIterator::delete_current_event()
{
    if ((!index) || use_current) return;    // don't delete too far 
    
    if (index == sched->heap_n) 
    {
        delete act_event;               // no upheap or downheap is necessary
        sched->heap_n--;                // when deleting last element of heap,
        index--;                        // so index may decrease (at least i
    }                                   // hope so)
    else
    {
        sched->heap[index] = sched->heap[sched->heap_n--];
        CNEvent *v = sched->heap[index];
        if (act_event->after(v))            // otherwise upheap
            sched->heap_upheap(index);      // or downheap are
        else                                // must be done
            sched->heap_downheap(index);
        
        delete act_event;
    }
    
    act_event = sched->heap[index];
    use_current = TRUE;
}



     
/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNEventHIterator::print(ostream &strm) const
{
    strm << "current event: " << act_event << endl
         << "index: " << index << endl;
}

// Debug output
void CNEventHIterator::dump(ostream &strm) const
{
    strm << "CNEventHIterator { $Revision: 0.30 $ ..." << endl;
    strm << "current event: " << act_event << endl
         << "index: " << index << endl;
    strm << " }" << endl;
}

// IOStream operator <<
ostream &operator << (ostream &strm, const CNEventHIterator &obj)
{
    obj.print(strm);
    return strm;
}

ostream &operator << (ostream &strm, const CNEventHIterator *obj)
{
    if(obj)
	obj->print(strm);
    else
	strm << "(NIL)";
    return strm;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNEventHIterator
static CNClass CNEventHIterator_desc("CNEventHIterator", "$Revision: 0.30 $",
			  /*CNEventHIterator::new_object*/ NIL);

// "Type" for type checking functions
CNClassDesc CN_EVENTHITERATOR = &CNEventHIterator_desc;
