// -*- C++ -*-

#include <string.h>
#include <stdlib.h>
#include <iostream.h>

#include <CNCL/QueueFIFO.h>
#include <CNCL/EventScheduler.h>
#include <CNCL/FiboG.h>
#include <CNCL/NegExp.h>
#include <CNCL/Moments.h>

#include <CNCL/EZDQueue.h>
#include <CNCL/EZDServer.h>
#include <CNCL/EZDWindow.h>
#include <CNCL/EZDPushButton.h>
#include <CNCL/EZDText.h>

#include <CNCL/Job.h>


enum { NJOBS=100000 };

enum { JOB, TIMER_G, TIMER_S };		// CNEvent types for M/M/1 simulation


class Server : public CNEventHandler
{
private:
    CNRandom &rnd_b;	// Distribution of service time b
    CNJob *job;		// Served job
    CNQueueFIFO queue;	// CNQueue
    CNMoments t_w, t_b;	// Evaluation tau_w, tau_b
    enum { WAITING, SERVING };
    EZDQueue  ezdq;
    EZDServer ezds;
    EZDText   ezdl;
    EZDText   ezdmb;
    EZDText   ezdvb;
    bool step_flag;
    
public:
    virtual void event_handler(const CNEvent *ev);

    void print_results();
    void eval_job(CNJob *job);

    void ezd_input();
    
    Server(CNRandom &rnd)
	: rnd_b(rnd), job(NIL), t_w("tau_w"), t_b("tau_b"),
	  ezdq(100, 100, 100, 30, 30), ezds(-1, 100, 30),
	  ezdl("l", 100, 160, ""),
	  ezdmb("mb", 100, 190, ""), ezdvb("vb", 100, 210, "")
    {
	step_flag = TRUE;
	state(WAITING);
	ezds.x( ezdq.right() );
	ezdq.redraw();
	ezdq.length(0);
	ezdq.color("LightSlateGray");
	ezds.color("Green");
	ezdl.set_text_val ("Queue length:", 0);
	ezdmb.set_text_val("b mean:      ", 0);
	ezdvb.set_text_val("b variance:  ", 0);
    }
};



void Server::ezd_input()
{
    if(step_flag || EZD::test_event())
    {
	CNString in;
	in = EZD::event();
	if(strstr(in, " EXIT "))
	    exit(0);
	if(strstr(in, " STEP "))
	    step_flag = TRUE;
	if(strstr(in, " RUN "))
	    step_flag = FALSE;
    }
}



class Generator : public CNEventHandler
{
private:
    CNRandom &rnd_a;	// Distribution of arrival time a
    Server *server;	// Connected queue/server
    long n;
    
public:
    virtual void event_handler(const CNEvent *ev);

    Generator(CNRandom &rnd, Server *serv) : rnd_a(rnd), server(serv), n(0) {}
};



void Generator::event_handler(const CNEvent *)
{
    if(n == NJOBS)
	// Stop simulation
	return;

#ifdef DEBUG
    cout << "----- Generator -----" << endl;
    ev->dump(cout);
#endif    

    // Incoming event -> generate new Job
    send_now(new CNEvent(JOB, server, new CNJob));
    // CNRandom delay
    send_delay(new CNEvent(TIMER_G), rnd_a());
    n++;
}



void Server::event_handler(const CNEvent *ev)
{
#ifdef DEBUG
    cout << "----- Server -----" << endl;
    ev->dump(cout);
#endif    

    ezd_input();
    
    switch(state())
    {
    case SERVING:
	switch(ev->type())
	{
	case JOB:
	    // Incoming job, put into queue
	    {
		CNJob *job;
		job = (CNJob *)ev->object();
		job->in = now();
		queue.put(job);
		ezdq.length( queue.length()*10 );
		ezdl.set_text_val("Queue length:", queue.length());
	    }
	    break;
	    
	case TIMER_S:
	    // Timer event, service time run down
	    job->out = now();
	    // Evaluate job
	    eval_job(job);
	    delete job;
	    job = NIL;
	    // Get new job from queue
	    if(!queue.empty())
	    {
		job = (CNJob *)queue.get();
		ezdq.length( queue.length()*10 );
		ezdl.set_text_val("Queue length:", queue.length());
		job->start = now();
		// CNRandom service time
		send_delay(new CNEvent(TIMER_S), rnd_b());
		ezds.color("Red");
		state(SERVING);
	    }
	    else
	    {
		ezds.color("Green");
		state(WAITING);
	    }
	    break;
	    
	default:
	    error("mm1: ", "illegal event in state SERVING");
	    break;
	}

	break;
	
    case WAITING:
	switch(ev->type())
	{
	case JOB:
	    // Incoming job
	    job = (CNJob *)ev->object();
	    job->in    = now();
	    job->start = now();
	    // CNRandom service time
	    send_delay(new CNEvent(TIMER_S), rnd_b());
	    ezds.color("Red");
	    state(SERVING);
	    break;

	default:
	    error("mm1: ", "illegal event in state WAITING");
	    break;
	}
	
	break;
	
    default:
	error("mm1: ", "illegal state");
	break;
	
    }
}



void Server::eval_job(CNJob *job)
{
    t_w.put(job->start - job->in);
    t_b.put(job->out   - job->in);

    ezdmb.set_text_val("b mean:      ", t_b.mean());
    ezdvb.set_text_val("b variance:  ", t_b.variance());
}


void Server::print_results()
{
    cout << t_w << t_b;
}

 

main()
{
    CNRNG    *rng   = new CNFiboG;
    CNNegExp rnd_a(10, rng);
    CNNegExp rnd_b( 5, rng);
    EZDPushButton *b;
    
    EZDWindow win(400, 300);
    EZDDrawing draw;
    win.overlay(&draw);

    EZD::draw_text(0, 0, 400, 50, "center center", "M/M/1 Simulation",
		   "", "-*-times-bold-r-*-*-40-*-*-*-*-*-*-*");
    EZD::draw_text(0, 279, 220, 20, "right center", "Press button:", "", "");
    
    b = new EZDPushButton("STEP", 229, 279, 50, 20, "Step");
    b = new EZDPushButton("RUN",  289, 279, 50, 20, "Run");
    b = new EZDPushButton("EXIT", 349, 279, 50, 20, "Exit");
    
    Server         server(rnd_b);
    Generator      generator(rnd_a, &server);
    CNEventScheduler scheduler;
    
    scheduler.start(new CNEvent(TIMER_G, &generator, 0.0));

    server.print_results();
    
}
