//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: GetOpt.c,v 0.32 1996-08-07 17:59:07+02 steppler Exp $
 *
 * Class: CNGetOpt --- Interface to GNU getopt()
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

/* Partly from GNU getopt.c / getopt1.c: */

/* Getopt for GNU.
   NOTE: getopt is now part of the C library, so if you don't know what
   "Keep this file name-space clean" means, talk to roland@gnu.ai.mit.edu
   before changing it!

   Copyright (C) 1987, 88, 89, 90, 91, 92, 1993
   	Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <string.h>
#include <stdlib.h>
#include <iostream.h>

#include <CNCL/Class.h>

#include "GetOpt.h"



/*
 * Set argc, argv
 */
void CNGetOpt::set_args(int argc, char **argv)
{
    i_argc = argc;
    i_argv = argv;
}


/*
 * Set single character options
 */
void CNGetOpt::set_char_options(char *copts)
{
    i_char_options = copts;
}


/*
 * Set long options array
 */
void CNGetOpt::set_long_options(option *lopts)
{
    i_long_options = lopts;
    n_long = -1;
}


/*
 * Add one long option
 */
void CNGetOpt::add_long_option(char *lopt, ParamType pt, char copt)
{
    if(n_long == -1)
    {
	error(NIL, "Getopt: ",
	      "can't use add_long_option() after set_long_options().");
	return;
    }

    if(!i_long_options)
	i_long_options = new option[MAXLONGOPTIONS];

    if(n_long >= MAXLONGOPTIONS-1)
    {
	error(NIL, "Getopt: ", "too many long options.");
	return;
    }

    i_long_options[n_long].name    = lopt;
    i_long_options[n_long].has_arg = int(pt);
    i_long_options[n_long].flag    = NIL;
    i_long_options[n_long].val     = copt;
    n_long++;
    i_long_options[n_long].name    = NIL;
    i_long_options[n_long].has_arg = 0;
    i_long_options[n_long].flag    = NIL;
    i_long_options[n_long].val     = 0;
}



/*
 * Run getopt()
 */
int CNGetOpt::getopt()
{
    return i_long_options
	?  getopt_long(i_argc, i_argv, i_char_options,
		       i_long_options, &i_index       )
	:  getopt     (i_argc, i_argv, i_char_options);
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNGetOpt
static CNClass CNGetOpt_desc("CNGetOpt", "$Revision: 0.32 $",
			    CNGetOpt::new_object);

// "Type" for type checking functions
CNClassDesc CN_GETOPT = &CNGetOpt_desc;



/* If GETOPT_COMPAT is defined, `+' as well as `--' can introduce a
   long-named option.  Because this is not POSIX.2 compliant, it is
   being phased out.  */
/* #define GETOPT_COMPAT */

/* This version of `getopt' appears to the caller like standard Unix `getopt'
   but it behaves differently for the user, since it allows the user
   to intersperse the options with the other arguments.

   As `getopt' works, it permutes the elements of ARGV so that,
   when it is done, all the options precede everything else.  Thus
   all application programs are extended to handle flexible argument order.

   Setting the environment variable POSIXLY_CORRECT disables permutation.
   Then the behavior is completely standard.

   GNU application programs can use a third alternative mode in which
   they can distinguish the relative order of options and other arguments.  */

#define	my_index		strchr
#define	my_bcopy(src, dst, n)	memcpy ((dst), (src), (n))

/* Exchange two adjacent subsequences of ARGV.
   One subsequence is elements [first_nonopt,last_nonopt)
   which contains all the non-options that have been skipped so far.
   The other is elements [last_nonopt,optind), which contains all
   the options processed since those non-options were skipped.

   `first_nonopt' and `last_nonopt' are relocated so that they describe
   the new indices of the non-options in ARGV after they are moved.  */

void CNGetOpt::exchange (char **argv)
{
  int nonopts_size = (last_nonopt - first_nonopt) * sizeof (char *);
//  char **temp = (char **) malloc (nonopts_size);

//  char **temp = new (char *)[last_nonopt - first_nonopt];
// Broken compilers ... :-(
  typedef char *char_p;
  char **temp = new char_p [last_nonopt - first_nonopt];
  
  /* Interchange the two blocks of data in ARGV.  */

  my_bcopy ((char *) &argv[first_nonopt], (char *) temp, nonopts_size);
  my_bcopy ((char *) &argv[last_nonopt], (char *) &argv[first_nonopt],
	    (i_optind - last_nonopt) * sizeof (char *));
  my_bcopy ((char *) temp,
	    (char *) &argv[first_nonopt + i_optind - last_nonopt],
	    nonopts_size);

//  free(temp);
  delete [] temp;
  
  /* Update records for the slots the non-options now occupy.  */

  first_nonopt += (i_optind - last_nonopt);
  last_nonopt = i_optind;
}

/* Scan elements of ARGV (whose length is ARGC) for option characters
   given in OPTSTRING.

   If an element of ARGV starts with '-', and is not exactly "-" or "--",
   then it is an option element.  The characters of this element
   (aside from the initial '-') are option characters.  If `getopt'
   is called repeatedly, it returns successively each of the option characters
   from each of the option elements.

   If `getopt' finds another option character, it returns that character,
   updating `optind' and `nextchar' so that the next call to `getopt' can
   resume the scan with the following option character or ARGV-element.

   If there are no more option characters, `getopt' returns `EOF'.
   Then `optind' is the index in ARGV of the first ARGV-element
   that is not an option.  (The ARGV-elements have been permuted
   so that those that are not options now come last.)

   OPTSTRING is a string containing the legitimate option characters.
   If an option character is seen that is not listed in OPTSTRING,
   return '?' after printing an error message.  If you set `opterr' to
   zero, the error message is suppressed but we still return '?'.

   If a char in OPTSTRING is followed by a colon, that means it wants an arg,
   so the following text in the same ARGV-element, or the text of the following
   ARGV-element, is returned in `optarg'.  Two colons mean an option that
   wants an optional arg; if there is text in the current ARGV-element,
   it is returned in `optarg', otherwise `optarg' is set to zero.

   If OPTSTRING starts with `-' or `+', it requests different methods of
   handling the non-option ARGV-elements.
   See the comments about RETURN_IN_ORDER and REQUIRE_ORDER, above.

   Long-named options begin with `--' instead of `-'.
   Their names may be abbreviated as long as the abbreviation is unique
   or is an exact match for some defined option.  If they have an
   argument, it follows the option name in the same ARGV-element, separated
   from the option name by a `=', or else the in next ARGV-element.
   When `getopt' finds a long-named option, it returns 0 if that option's
   `flag' field is nonzero, the value of the option's `val' field
   if the `flag' field is zero.

   The elements of ARGV aren't really const, because we permute them.
   But we pretend they're const in the prototype to be compatible
   with other systems.

   LONGOPTS is a vector of `struct option' terminated by an
   element containing a name which is zero.

   LONGIND returns the index in LONGOPT of the long-named option found.
   It is only valid when a long-named option has been found by the most
   recent call.

   If LONG_ONLY is nonzero, '-' as well as '--' can introduce
   long-named options.  */

int CNGetOpt::_getopt_internal(int argc, char *const *argv,
			       const char *optstring,
			       const struct option *longopts, int *longind,
			       int long_only                                )
{
  int option_index;

  i_optarg = 0;

  /* Initialize the internal data when the first call is made.
     Start processing options with ARGV-element 1 (since ARGV-element 0
     is the program name); the sequence of previously skipped
     non-option ARGV-elements is empty.  */

  if (i_optind == 0)
    {
      first_nonopt = last_nonopt = i_optind = 1;

      nextchar = NULL;

      /* Determine how to handle the ordering of options and nonoptions.  */

      if (optstring[0] == '-')
	{
	  ordering = RETURN_IN_ORDER;
	  ++optstring;
	}
      else if (optstring[0] == '+')
	{
	  ordering = REQUIRE_ORDER;
	  ++optstring;
	}
      else if (getenv ("POSIXLY_CORRECT") != NULL)
	ordering = REQUIRE_ORDER;
      else
	ordering = PERMUTE;
    }

  if (nextchar == NULL || *nextchar == '\0')
    {
      if (ordering == PERMUTE)
	{
	  /* If we have just processed some options following some non-options,
	     exchange them so that the options come first.  */

	  if (first_nonopt != last_nonopt && last_nonopt != i_optind)
	    exchange ((char **) argv);
	  else if (last_nonopt != i_optind)
	    first_nonopt = i_optind;

	  /* Now skip any additional non-options
	     and extend the range of non-options previously skipped.  */

	  while (i_optind < argc
		 && (argv[i_optind][0] != '-' || argv[i_optind][1] == '\0')
#ifdef GETOPT_COMPAT
		 && (longopts == NULL
		     || argv[i_optind][0] != '+' || argv[i_optind][1] == '\0')
#endif				/* GETOPT_COMPAT */
		 )
	    i_optind++;
	  last_nonopt = i_optind;
	}

      /* Special ARGV-element `--' means premature end of options.
	 Skip it like a null option,
	 then exchange with previous non-options as if it were an option,
	 then skip everything else like a non-option.  */

      if (i_optind != argc && !strcmp (argv[i_optind], "--"))
	{
	  i_optind++;

	  if (first_nonopt != last_nonopt && last_nonopt != i_optind)
	    exchange ((char **) argv);
	  else if (first_nonopt == last_nonopt)
	    first_nonopt = i_optind;
	  last_nonopt = argc;

	  i_optind = argc;
	}

      /* If we have done all the ARGV-elements, stop the scan
	 and back over any non-options that we skipped and permuted.  */

      if (i_optind == argc)
	{
	  /* Set the next-arg-index to point at the non-options
	     that we previously skipped, so the caller will digest them.  */
	  if (first_nonopt != last_nonopt)
	    i_optind = first_nonopt;
	  return EOF;
	}

      /* If we have come to a non-option and did not permute it,
	 either stop the scan or describe it to the caller and pass it by.  */

      if ((argv[i_optind][0] != '-' || argv[i_optind][1] == '\0')
#ifdef GETOPT_COMPAT
	  && (longopts == NULL
	      || argv[i_optind][0] != '+' || argv[i_optind][1] == '\0')
#endif				/* GETOPT_COMPAT */
	  )
	{
	  if (ordering == REQUIRE_ORDER)
	    return EOF;
	  i_optarg = argv[i_optind++];
	  return 1;
	}

      /* We have found another option-ARGV-element.
	 Start decoding its characters.  */

      nextchar = (argv[i_optind] + 1
		  + (longopts != NULL && argv[i_optind][1] == '-'));
    }

  if (longopts != NULL
      && ((argv[i_optind][0] == '-'
	   && (argv[i_optind][1] == '-' || long_only))
#ifdef GETOPT_COMPAT
	  || argv[i_optind][0] == '+'
#endif				/* GETOPT_COMPAT */
	  ))
    {
      const struct option *p;
      char *s = nextchar;
      int exact = 0;
      int ambig = 0;
      const struct option *pfound = NULL;
      int indfound = 0;

      while (*s && *s != '=')
	s++;

      /* Test all options for either exact match or abbreviated matches.  */
      for (p = longopts, option_index = 0; p->name;
	   p++, option_index++)
	if (!strncmp (p->name, nextchar, s - nextchar))
	  {
	    if (((unsigned long)(s - nextchar)) == strlen (p->name))
	      {
		/* Exact match found.  */
		pfound = p;
		indfound = option_index;
		exact = 1;
		break;
	      }
	    else if (pfound == NULL)
	      {
		/* First nonexact match found.  */
		pfound = p;
		indfound = option_index;
	      }
	    else
	      /* Second nonexact match found.  */
	      ambig = 1;
	  }

      if (ambig && !exact)
	{
	  if (i_opterr)
	      cerr << argv[0] << ": option `" << argv[i_optind]
		   << "' is ambiguous" << endl;
	  nextchar += strlen (nextchar);
	  i_optind++;
	  return '?';
	}

      if (pfound != NULL)
	{
	  option_index = indfound;
	  i_optind++;
	  if (*s)
	    {
	      /* Don't test has_arg with >, because some C compilers don't
		 allow it to be used on enums.  */
	      if (pfound->has_arg)
		i_optarg = s + 1;
	      else
		{
		  if (i_opterr)
		    {
		      if (argv[i_optind - 1][1] == '-')
			/* --option */
			  cerr << argv[0] << ": option `--" << pfound->name
			       << "' doesn't allow an argument" << endl;
		      else
			/* +option or -option */
			  cerr << argv[0] << ": option `"
			       << argv[i_optind - 1][0] << pfound->name
			       << "' doesn't allow an argument" << endl;
		    }
		  nextchar += strlen (nextchar);
		  return '?';
		}
	    }
	  else if (pfound->has_arg == 1)
	    {
	      if (i_optind < argc)
		i_optarg = argv[i_optind++];
	      else
		{
		  if (i_opterr)
		      cerr << argv[0] << ": option `" << argv[i_optind - 1]
			   << "' requires an argument" << endl;
		  nextchar += strlen (nextchar);
		  return optstring[0] == ':' ? ':' : '?';
		}
	    }
	  nextchar += strlen (nextchar);
	  if (longind != NULL)
	    *longind = option_index;
	  if (pfound->flag)
	    {
	      *(pfound->flag) = pfound->val;
	      return 0;
	    }
	  return pfound->val;
	}
      /* Can't find it as a long option.  If this is not getopt_long_only,
	 or the option starts with '--' or is not a valid short
	 option, then it's an error.
	 Otherwise interpret it as a short option.  */
      if (!long_only || argv[i_optind][1] == '-'
#ifdef GETOPT_COMPAT
	  || argv[i_optind][0] == '+'
#endif				/* GETOPT_COMPAT */
	  || my_index (optstring, *nextchar) == NULL)
	{
	  if (i_opterr)
	    {
	      if (argv[i_optind][1] == '-')
		/* --option */
		  cerr << argv[0] << ": unrecognized option `"
		       << nextchar << "'" << endl;
	      else
		/* +option or -option */
		  cerr << argv[0] << ": unrecognized option `"
		       <<argv[i_optind][0] << nextchar << "'" << endl;
	    }
	  nextchar = (char *) "";
	  i_optind++;
	  return '?';
	}
    }

  /* Look at and handle the next option-character.  */

  {
    char c = *nextchar++;
    const char *temp = my_index (optstring, c);

    /* Increment `i_optind' when we start to process its last character.  */
    if (*nextchar == '\0')
      ++i_optind;

    if (temp == NULL || c == ':')
      {
	if (i_opterr)
	  {
	    /* 1003.2 specifies the format of this message.  */
	      cerr << argv[0] << ": illegal option -- " << c << endl;
	  }
	i_optopt = c;
	return '?';
      }
    if (temp[1] == ':')
      {
	if (temp[2] == ':')
	  {
	    /* This is an option that accepts an argument optionally.  */
	    if (*nextchar != '\0')
	      {
		i_optarg = nextchar;
		i_optind++;
	      }
	    else
	      i_optarg = 0;
	    nextchar = NULL;
	  }
	else
	  {
	    /* This is an option that requires an argument.  */
	    if (*nextchar != '\0')
	      {
		i_optarg = nextchar;
		/* If we end this ARGV-element by taking the rest as an arg,
		   we must advance to the next element now.  */
		i_optind++;
	      }
	    else if (i_optind == argc)
	      {
		if (i_opterr)
		  {
		    /* 1003.2 specifies the format of this message.  */
		      cerr << argv[0] << ": option requires an argument -- "
			   << c << endl;
		  }
		i_optopt = c;
		if (optstring[0] == ':')
		  c = ':';
		else
		  c = '?';
	      }
	    else
	      /* We already incremented `optind' once;
		 increment it again when taking next ARGV-elt as argument.  */
	      i_optarg = argv[i_optind++];
	    nextchar = NULL;
	  }
      }
    return c;
  }
}

int CNGetOpt::getopt(int argc, char *const *argv, const char *optstring)
{
  return _getopt_internal (argc, argv, optstring,
			   (const struct option *) 0,
			   (int *) 0,
			   0);
}

int CNGetOpt::getopt_long(int argc, char *const *argv, const char *options,
			  const struct option *long_options, int *opt_index)
{
  return _getopt_internal (argc, argv, options, long_options, opt_index, 0);
}

/* Like getopt_long, but '-' as well as '--' can indicate a long option.
   If an option that starts with '-' (not '--') doesn't match a long option,
   but does match a short option, it is parsed as a short option
   instead.  */

int CNGetOpt::getopt_long_only(int argc, char *const *argv,
			       const char *options,
			       const struct option *long_options,
			       int *opt_index)
{
  return _getopt_internal (argc, argv, options, long_options, opt_index, 1);
}
