//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_            C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_                      N E T W O R K S
 * |_        |_  |_  |_  |_        |_                            C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_                  L I B R A R Y
 *
 * $Id: Batches.c,v 0.33 1996-08-07 18:02:45+02 steppler Exp $
 *
 * Class: CNBatches --- bad old Batch Means conversion from SIC
 * Derived from SIC - Simulation In C++, use CNBatchMeans instead
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#include "Batches.h"


long CNBatches::global_intervalnumber = 10;     // number of interval



/***** Constructors **********************************************************/

CNBatches::CNBatches( double BOTTOM, double TOP, double MAX_ERR, long NOG,
                      long SOG, long NOI, short CONF,
		      const char *NAME, const char *TEXT )
    : CNStatistics(NAME, TEXT)
{

    if ( CONF != 95 && CONF != 99 ) {
    CNCL::warning("Batches.c", 
    "CNBatches::CNBatches: confidence not available, set to 95");
    CONF = 95;
    }       
    
    nog= NOG;
    remaining = sog = SOG;
    max_err = MAX_ERR;
    conf = CONF;
    noi = NOI;
    if (noi > global_intervalnumber) global_intervalnumber = noi; // olly 01/95
    bottom = BOTTOM;
    top = TOP;
    phase = ITERATE;
    act_group = 0;
    sum = 0.0;
    sqsum = 0.0;
    min_val = DBL_MAX;
    max_val = -DBL_MAX;
    nrv = 0;
    wasted_left = 0;
    wasted_right = 0;
    gm_sum = gm_sqsum = 0.0;
    gv_sum = gv_sqsum = 0.0;
    ah = 0;
    rh = rh_sum = rh_mean = 0.0;
    f = g = 0.0;
    f_sum = f_sqsum = 0.0;
    f_var = 0.0;
    f_mean = g_mean = 0.0;
    f_bayes = g_bayes = 0.0;
    fg_conf= 0.0;
    
    //generate reqired number of groups
    if (( g_chain = new group[nog]) == NIL) 
    CNCL::fatal("Batches.c", "CNBatches::CNBatches: out of memory\n") ;

    
    //generate reqired number of intervals
    if (( i_chain = new interval[noi] ) == NIL) 
    CNCL::fatal("Batches.c", "CNBatches::CNBatches: out of memory\n") ;


    // and initialize 

    for (long i=0; i<noi; i++) { //olly global_int...->noi 1/95
    (&i_chain[i])->n = 0;
    (&i_chain[i])->left = bottom + i*(top-bottom)/noi;
    (&i_chain[i])->right = (&i_chain[i])->left +
                    (top-bottom)/noi;
    }

    //generate reqired number of output intervals
    if (( i_out_chain = new interval_out[noi] ) == NIL) 
    CNCL::fatal("Batches.c", "CNBatches::CNBatches: out of memory\n") ;
    end_reached = FALSE;    // 03.12.93 (fuss)
}

//---------------->   CNBatches::variance()  <---------------  2/10/90 - sth -//
//  returns variance of handed over values                    13.08.93 fuss   //
//  ATTENTION: The PC implementation of the floating point arithmetic         //
//       doesn't work correct: y = 0.0/x isn't calculated with 0.0            //
//       therefore testing y < 1E-15 ==> y=0.0                                //
//----------------------------------------------------------------------------//
//  used functions: --                                                        //
//                                                                            //
//  parameter:                                                                //
//      - n     number of values                                              //
//      - sum   sum of values                                                 //
//      - sqsum square sum of values                                          //
//                                                                            //
//  return value:   variance of values                                        // 
//----------------------------------------------------------------------------//
double CNBatches::variance( long n, double sum, double sqsum ) const
{
    double rwert;
    rwert=(n>1) ? (sqsum - ((sum*sum)/n))/(n-1)
        : DBL_MAX;
    return ( (n > 1) 
        ? (((rwert=(sqsum - ((sum*sum)/n))/(n-1))>0.00000000000001) 
            ? rwert 
            : 0) 
        : DBL_MAX );  
}

//------------------->   CNBatches::confi()  <---------------  2/10/90 - sth -//
//  returns confidence interval                               13.08.93 fuss   //
//  tables (tvert975, tvert995) changed                     11/03/93 (olly)   //
//----------------------------------------------------------------------------//
//  used functions:         -sqrt()                                           //
//                                                                            //
//  parameter:                                                                //
//      - n     degree of freedom of inverse t-distribution                   //
//      - kz    confidence of inverse t-distribution                          //
//      - var   variance of value, which ci is calculated                     //
//                                                                            //
//  return value:   confidence interval                                       //
//                                                                            //
//----------------------------------------------------------------------------//
double CNBatches::confi( long n, double kz, double var ) const
{
    if ((n>0) && ((kz==99) || (kz==95))) {// permitted values
    if (n>101) n = 101; // only first 100 degrees of freedom 
                // are calculated exact (danach Naeherung)
    if ( kz == 95 ) { 
#define VERT975(a) a
static double    tvert975[] = { 
#include "tvert975.dat" 
}; 
        return (tvert975[n]*sqrt(var/(n+1)));   
    }
    else {  
#define VERT995(a) a
static double    tvert995[] = { 
#include "tvert995.dat" 
}; 
        return (tvert995[n]*sqrt(var/(n+1)));
        }
    }
    else return DBL_MAX;
}

//-------------------->   CNBatches::calc_groups   <------  2/10/90 - sth ---//
//  calculation of characteristic group values             13.08.93 fuss     //
//---------------------------------------------------------------------------//
//  used functions: - CNBatches::variance()                                  //
//                                                                           //
//  parameter:                                                               //
//      --                                                                   //
//                                                                           //
//  return value: --                                                         //
//---------------------------------------------------------------------------//
void CNBatches::calc_groups() 
{
    gm_sum = gm_sqsum = 0;
    gv_sum = gv_sqsum = 0;
    
    group* g;

    for( long i=0; i<act_group; i++) {
        // act_group contains number of current group
        // attention: 1st group -> act_group = 0 etc.

        g = &g_chain[i];

            // sum, square sum of group means
    gm = (g->sum)/(g->n);
    gm_sum += gm;
    gm_sqsum += gm * gm;

            // sum, square sum of group variances
    gv = ( (g->n)!=0 ) 
        ? variance( g->n, g->sum, g->sqsum ) : 0;
    gv_sum += gv;
    gv_sqsum += gv * gv;
    }
}

//---------------->   CNBatches::calc_intervals <---------  2/10/90 - sth ---//
//  calculation of characteristic interval values          13.08.93 fuss     //
//---------------------------------------------------------------------------//
//  used functions:                                                          //
//          sqrt()                                                           //
//          CNBatches::variance()                                            //
//          CNBatches::confi()                                               //
//                                                                           //
//  parameter: --                                                            //
//                                                                           //
//---------------------------------------------------------------------------//
void CNBatches::calc_intervals()
{
    long i, j;
    
    //start of evaluation
    for (j=0; j<noi; j++) {
            
        //evaluation of each interval
        for(i=0; i<act_group; i++) {
        
            //sum absolute frequency of interval hits
            //(for all groups) -> e.g.: interv. x, gr.3: ah=ah(0)+...+ah(3)
        ah = (&((&g_chain[i])->ip)[j])->n;
            rh = (double) ah/sog;
            rh_sum += rh;
            
        // new value of df  
        f = (j!=0) 
            ? ( rh + (&((&g_chain[i])->ip)[j-1])->f ) 
            : rh ;
        
        // compare to 1.0 works not correct !!!
        assert((f>=0)&&(f<=1.00001));
        
        // save new value of df
        (&((&g_chain[i])->ip)[j])->f = f;
        
        // stat evaluation for df
        f_sum += f;
        f_sqsum += f * f;
        
        } // end for i (all groups evaluated)
       
        //nowq I`m in Interval j,
        //i = "act_group" !!
        rh_mean = rh_sum/i;
    
    // calculation of df`s values
    f_mean = f_sum/i;
    f_var = variance(i, f_sum, f_sqsum);
    f_bayes = ((i>5) && (f_mean!=0)) 
        ? sqrt(f_var*(i-1)/(i*(i-5)))/f_mean 
        : DBL_MAX;
    fg_conf= confi(i-1,conf,f_var);

    // calculation of cdf`s values
    g_mean = 1.0 - f_mean;      
    g_mean = (g_mean < 0) ? -g_mean : g_mean ;
        //sonst evtl g_mean = -0.00000
    g_bayes = ((i>5) && (g_mean!=0)) ? f_mean*f_bayes/g_mean : DBL_MAX;
    
    // save the results to output interval
    (&i_out_chain[j])->rh_mean = rh_mean;
    (&i_out_chain[j])->f_mean = f_mean;
    (&i_out_chain[j])->f_bayes = f_bayes;
    (&i_out_chain[j])->g_bayes = g_bayes;
    (&i_out_chain[j])->fg_conf= fg_conf;
    
    
        //re-initialize of counters
        ah = 0;
        rh = 0.0;
        rh_sum = 0.0;
        //f = 0;
        f_sum = 0.0;
    f_sqsum = 0.0;
       
    } // end for j
   
}

//----------------->   CNBatches::print_intervals <-------  17/10/90 - sth --//
//  output of interval characteristics                      13.08.93 fuss    //
//  interval_out contains the characteristics                                //
//---------------------------------------------------------------------------//
//  used functions:  --                                                      //
//                                                                           //
//  parameter:                                                               //
//       - &strm        pointer to output stream,                            //
//              must be opened by the calling procedure                      //
//           - flag     flag=DF:  output of distribution function            //
//                              (flag=CDF:output of compl. distr. function   //
//                                                                           //
//  return value: --                                                         //
//---------------------------------------------------------------------------//
void CNBatches::print_intervals( ostream &strm, CNStatistics::Type flag ) const
{
    double lower;
    
    strm.setf( ios::scientific, ios::floatfield );
    strm.setf( ios::right );
    strm.precision(6);
    for (long j=0; j<noi; j++) {
    // confidence interval only Freiheitsgrad > 0 sinnvoll
    lower = (double) wasted_left/(sog*act_group);

        switch ( flag ) {
        case CNStatistics::DF:
            if ( !j ) {
                strm << "#F(x)          x              rel.error      ";
                strm << "rel.H          conf\n";
                strm << setw(12) << lower;
                strm << setw(15) << (&((&g_chain[0])->ip)[j])->left;
                strm << setw(15) << 0.0;
                strm << setw(15) << 0.0;
                strm << setw(15) << 0.0 << endl;
            }
            strm << setw(12) << (&i_out_chain[j])->f_mean + lower;
            strm << setw(15) << (&((&g_chain[0])->ip)[j])->right;
            strm << setw(15) << (&i_out_chain[j])->f_bayes;
            strm << setw(15) << (&i_out_chain[j])->rh_mean;
            strm << setw(15) << (&i_out_chain[j])->fg_conf << endl;

/*      strm << form("%-12lg %-10lg %-13lg %-13lg %-12le\n",
            (&i_out_chain[j])->f_mean + lower,
        (&((&g_chain[0])->ip)[j])->right,
        (&i_out_chain[j])->f_bayes,
            (&i_out_chain[j])->rh_mean,
        (&i_out_chain[j])->fg_conf);
*/            break;

        case CNStatistics::CDF:
            if ( !j ) {
                strm << "#G(x)          x              rel.error      ";
                strm << "rel.H          conf\n";
                strm << setw(12) << 1.0 - lower;
                strm << setw(15) << (&((&g_chain[0])->ip)[j])->left;
                strm << setw(15) << 0.0;
                strm << setw(15) << 0.0;
                strm << setw(15) << 0.0 << endl;
            }
            strm << setw(12) << 1.0 - (&i_out_chain[j])->f_mean - lower;
            strm << setw(15) << (&((&g_chain[0])->ip)[j])->right;
            strm << setw(15) << (&i_out_chain[j])->f_bayes;
            strm << setw(15) << (&i_out_chain[j])->rh_mean;
            strm << setw(15) << (&i_out_chain[j])->fg_conf << endl;

/*      strm << form("%-12lg %-10lg %-13lg %-13lg %-12le\n",
            1 - lower - (&i_out_chain[j])->f_mean,
            (&((&g_chain[0])->ip)[j])->right,
            (&i_out_chain[j])->g_bayes,
            (&i_out_chain[j])->rh_mean,
            (&i_out_chain[j])->fg_conf);
*/            break;

	  default:
	    break;
        }

    } // end for j
    
}
   

//------------------------>   CNBatches::put   <----------  2/10/90 - sth ---//
//                                                         13.08.93 fuss     //
//  this function sorted values to interval                                  //
//---------------------------------------------------------------------------//
//  used functions: - sqrt()                                                 //
//              - CNBatches::calc_intervals()                                //
//              - CNBatches::calc_groups()                                   //
//              - CNBatches::status()                                        //
//              - disp_open (DOS!)                                           //
//              - disp_close (DOS!)                                          //
//              - disp_pokew (DOS!)                                          //
//                                                                           //
//  parameter:                                                               //
//  - wert       einzusortierender (Zufalls-)Wert                            //
//                                                                           //
//  return value: --                                                         //
//---------------------------------------------------------------------------//
void CNBatches::put(double wert)
{ 
    // if not enough trials
    nrv++;
    if ( status() != END ) {

    assert(wert>=0); //muss das sein? 
    
               //new minimum/maximum?
    min_val = (wert < min_val) ? wert : min_val;
    max_val = (wert > max_val) ? wert : max_val;
    sum += wert;        // sum all random numbers
    sqsum += wert * wert;   // squade sum
        
    for (long i=0; i<noi; i++) {
        if ( wert > (&i_chain[i])->right ) {
        if ( i == noi-1 ) {
                // higher than (absolute) right boundary
            wasted_right++;
            break;
        }
        else 
            continue;
        }
        
        if ( wert >= (&i_chain[0])->left ) {
                // hit !
        (&i_chain[i])->n += 1;
        break;
        }
        else {
            // lower than (absolute) left boundary
        wasted_left++;
        break;
        }
    }   
        
    if ( --remaining == 0 ) {
            // save group values
        (&g_chain[act_group])->n = sog;
        (&g_chain[act_group])->sum = sum;
        (&g_chain[act_group])->sqsum = sqsum;
        for (long j=0; j<noi; j++) {
        (&((&g_chain[act_group])->ip)[j])->n = (&i_chain[j])->n;
        (&((&g_chain[act_group])->ip)[j])->left=(&i_chain[j])->left;
        (&((&g_chain[act_group])->ip)[j])->right=(&i_chain[j])->right;
        }
        
            //re-initialize of counters
        sum = 0.0;
        sqsum = 0.0;  
        remaining = sog;        
        for (long i=0; i<noi; i++) 
        (&i_chain[i])->n = 0;

            ++act_group;
            calc_groups();
           
        // Bayes-error of group means
            double calculated_gm_bayes = (act_group>5)
                ? sqrt((act_group/(act_group-5)*(gm_sqsum/(gm_sum*gm_sum)-1/act_group)))
                : DBL_MAX;
    
       //print_groups(stdout);
           //fprintf(stdout,"act: %ld..gm_sum: %lg,bayes Fehler: calc: %lg%%, vorgeg.: %lg%%\n\n",act_group,gm_sum,calculated_gm_bayes,max_err);

        if ((act_group == nog) || (calculated_gm_bayes <= max_err ) ) {
        // no more groups or Bayes-error lower than limit
        phase = END ;
        end_reached = TRUE; // 03.12.93 (fuss)
        calc_intervals();
        }
    }
    }
}


//----------------->   CNBatches::~CNBatches() <----------  17/10/90 - sth --//
//  destructor of class CNBatches.                          13.08.93 fuss    //
//---------------------------------------------------------------------------//
//  used functions:    --                                                    //
//                                                                           //
//  parameter: --                                                            //
//                                                                           //
//  return value: --                                                         //
//---------------------------------------------------------------------------//
CNBatches::~CNBatches()
{
    delete i_chain;
    delete g_chain;
}
    
//------------------->   CNBatches::print_groups  <-------  17/10/90 - sth --//
//  output of group characteristics                         13.08.93 fuss    //
//---------------------------------------------------------------------------//
//  used functions:                                                          //
//                  sqrt()                                                   //
//                  CNBatches::confi()                                       //
//                  CNBatches::variance()                                    //
//                                                                           //
//  parameter:                                                               //
//       - &strm        pointer to output stream,                            //
//              must be opened by the calling procedure                      //
//              geoeffnet worden sein !!                                     //
//                                                                           //
//  return value: --                                                         //
//---------------------------------------------------------------------------//
void CNBatches::print_groups( ostream &strm ) const
{
    // output of values beyond boundary
    strm.unsetf( ios::scientific );
    strm.setf( ios::dec, ios::floatfield );
    strm.setf( ios::left );
    strm.precision(6);

    double percent = (wasted_left) ?  (wasted_left*100/(sog*act_group))
                                  : 0.0;
    strm << "# " << wasted_left << " values < " << bottom;
    strm << " (" << percent << "%), ";

    percent = (wasted_right)?(double) wasted_right*100/(sog*act_group):0.0;
    strm << " \t\t" << wasted_right << " values > " << top;
    strm << " (" << percent << "%), " << endl;
    
                    // variance of group means
    double gm_var = variance (act_group, gm_sum, gm_sqsum);
                    // variance of group variances
    double gv_var = variance (act_group, gv_sum, gv_sqsum);
                    
                    //Bayes-error

    double gm_bayes = (act_group>5)     
    ? sqrt((act_group/(act_group-5)*(gm_sqsum/(gm_sum*gm_sum)-1./act_group)))
        : DBL_MAX;

    double gv_bayes = (act_group>7) 
    ? sqrt(2.0/(act_group-7)) 
    : DBL_MAX;
                    
                // confidence intervals
                // confidence is defined by user
    double gm_conf = confi(act_group-1,conf,gm_var);
    double gv_conf = confi(act_group-1,conf,gv_var);
    

    double help = (act_group) ? gm_sum/act_group : DBL_MAX;
    strm << "#mean    : " << setw(11) << help << " +/- " << setw(11) << gm_conf;
    strm << " (" << conf << "%) (rel.error: +";
    strm << gm_bayes << ")" << endl;
    
    help = (act_group) ? gv_sum/act_group : DBL_MAX;
    strm << "#variance: " << setw(11) << help << " +/- " << setw(11) << gv_conf;
    strm << " (" << conf << "%) (rel.error: +";
    strm << gv_bayes << ")" << endl;
}


    
//------------>   CNBatches::print_group_results <---------  2/10/90 - sth --//
// output of group results                                  13.08.92 fuss    //             //
//---------------------------------------------------------------------------//
//  used functions:                                                          //
//                            CNBatches::calc_groups()                       //
//                            CNBatches::print_groups()                      //
//                                                                           //
//  parameter: --                                                            //
//                                                                           //
//---------------------------------------------------------------------------//
void CNBatches::print_group_results( ostream &strm )
{
    strm.unsetf( ios::scientific );
    strm.setf(ios::floatfield, ios::fixed);
    strm << endl;
    strm << "-----------------------------------------------------------------------------" << endl;
    strm << "       batch -- means  --  output" << endl;
    strm << "-----------------------------------------------------------------------------" << endl;
    strm << "\nsize of groups: " << sog << endl;
    strm << "\tnumber of groups: " << nog;
    strm << " evaluated incl. group no. " << act_group << endl;
    strm << "number of intervals: " << noi << endl;
    strm << "max. rel. error: " << max_err << endl;
    strm << "name: " << name << endl;
    strm << "text: " << text << endl;
    strm.precision(6);
    strm << endl << "min. value: " << setw(9) << min_val;
    strm << ",   max. value: " << setw(9) << max_val << endl;
    
    //output of group characteristics
    calc_groups();
    print_groups( strm );   
}



/***** Default I/O member function for CNCL classes **************************/

// Normal output

//------------------->   CNBatches::print  <--------------  17/10/90 - sth --//
//                                                          13.08.93 fuss    //
// output of evaluation results to output stream, opened by the user         //
// Benutzer anzugeben ist.                                                   //
//---------------------------------------------------------------------------//
//      used functions:                              //
//                            CNBatches::calc_groups()                       //
//                            CNBatches::print_groups()                      //
//                            CNBatches::print_intervals()                   //
//                                       //
//  parameter:                               //
//      - &strm               pointer to output stream                       //
//      - type                DF bzw. CDF fuer F(x) bzw. G(x)                //
//                                                                           //
//      return value: dummy !                                                //
//---------------------------------------------------------------------------//
void CNBatches::print( CNStatistics::Type type, ostream &strm ) const
{
    // Magic Zeile
    strm << "#BATCHMEANS RESULT (THIS IS A MAGIC LINE)\n";
    strm << "#------------------------------------------------------------------------ " ;
    strm << endl;
    strm << "#Name: " << name << endl;
    strm << "#Text: " << text << endl;
    strm << "#conf .........: " << conf;
    strm << " \tev_groups ...: " << act_group;
    strm << " \tmax_err ........: " << max_err << endl;
    strm << "#size of groups: " << sog;
    strm << " \tno. of groups: " << nog;
    strm << " \tno. of intervals: " << noi << endl;
    strm << "#min. value ...: ";
    strm << setiosflags(ios::scientific) << setw(12) << setprecision(3);
    strm << min_val << ", \tmax. value: " << setw(12) << max_val << endl;
    //output of group characteristics
    //calc_groups();
    print_groups( strm );   
    
    if ( noi > 1 ) {
    strm << "#------------------------------------------------------------------------ " ;
    strm << endl;
        print_intervals( strm, type );  
    }
}


void CNBatches::reset() {
    error( "CNBatches::reset(): Not yet implemented !" );
    }
    
void CNBatches::print( ostream &strm ) const
    {
        print( CNStatistics::DF, strm );
    }

// Debug output
void CNBatches::dump(ostream &strm) const
{
    strm << "CNBatches { $Revision: 0.33 $ ..."
     << " }" << endl;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNBatches
static CNClass CNBatches_desc("CNBatches", "$Revision: 0.33 $",
                            CNBatches::new_object);

// "Type" for type checking functions
CNClassDesc CN_BATCHES = &CNBatches_desc;

