//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_                C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_                          N E T W O R K S
 * |_        |_  |_  |_  |_        |_                                C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_                      L I B R A R Y
 *
 * $Id: Batches.h,v 0.33 1996-08-07 18:02:47+02 steppler Exp $
 *
 * Class: CNBatches --- bad old Batch Means conversion from SIC
 * Derived from SIC - Simulation In C++, use CNBatchMeans instead
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#ifndef __CNBatches_h
#define __CNBatches_h


#include <CNCL/Class.h>
#include <CNCL/Param.h>
#include <CNCL/CNCL.h>

#include <CNCL/Statistics.h>            // Base class

#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include <string.h>
#include <fstream.h>
#include <limits.h>
#include <float.h>
#include <iomanip.h>


extern CNClassDesc CN_BATCHES;  // Class CNBatches description


/*
 * The class CNBatches
 */

class CNBatches : public CNStatistics
{
#ifdef CNCL_NESTED_NEEDS_PUBLIC
  public:
#endif
    static long global_intervalnumber;	// number of intervals
    
  public:

    /***** Constructors ******************************************************/  public:
    CNBatches( double BOTTOM=0.0, double TOP=100.0, double MAX_ERR=0.2, 
               long NOG=10, long SOG=1000,
               long NOI=10, short CONF=95, 
               const char* NAME = NIL, const char* TEXT = NIL);
    CNBatches(CNParam *) {}                // CNParam constructor
    
    /***** Destructors ******************************************************/
    ~CNBatches();
    
    /***** Member functions **************************************************/
  private:
#ifdef CNCL_NESTED_NEEDS_PUBLIC
  public:
#endif
    /*********************  class  interval  ***************** 2/10/90 sth ***/
    /*                                                        13/08/93 fuss  */
    /*      class containing the statistical values of an interval           */
    /*                                                                       */
    /*************************************************************************/
    struct interval 
    {
        long        n;                      // number of trials in the interval
        double      left;                   // left boundary of interval                                                            
        double      right;                  // right boundary of interval                                                           
        double      f;                      // value of distribution function
                                            // at right interval boundary,
                                            // necessary for evaluation
        // constructor
        interval()                          // 17/10/90 - sth
        {
            n = -1;
            f = 0;
        }
    };

    /*****************   class interval_out  **************** 17/10/90 sth ***/
    /*                                                        13/08/93 fuss  */
    /*      class used by output of intervals, contains                      */
    /*      statistical results                                              */
    /*                                                                       */
    /*************************************************************************/
    struct interval_out 
    {
        // values, collected for the output
        double      rh_mean;                // mean relative hit frequency
        double      f_mean;                 // mean of df
        double      f_bayes, g_bayes;       // rel. bayes-error of df/compl. df 
        double      fg_conf;                // confidence interval
                                            // of f and g
        // constructor
        interval_out()                      // 17/10/90 - sth
        {
            rh_mean = DBL_MAX;
            f_mean = DBL_MAX;
            f_bayes = DBL_MAX;
            g_bayes = DBL_MAX;
            fg_conf= DBL_MAX;
        }
    };


    /*********************  class group  ********************* 2/10/90 sth ***/
    /*                                                        13/08/93 fuss  */
    /*      class contains statistical group values                          */
    /*                                                                       */
    /*************************************************************************/
    struct group 
    {
        long        n;                      // number of trials
        double      sum;                    // sum of trials
        double      sqsum;                  // square sum of trials
        interval*   ip;                     // pointer to intervals of a group
        // constructor
        group()                             // 17/10/90 - sth
        {
            n=-1;
            sum=sqsum=0.0;
            //
            //benoetigte Intervalle erzeugen
            if (( ip = new interval[CNBatches::global_intervalnumber] ) == NIL) 
                CNCL::fatal("Batches.c", "group::group: out of memory");
        }
    };

  private:
    long        nog;                    // number of batches
    long        sog;                    // size of batches
    long        noi;                    // number of intervals
    double      bottom;                 // absolute lower/upper boundary
    double      top;                    // of intervals
    double      max_err;                // max. rel. error of group means
    short       conf;                   // confidence probability
        
    // characteristic group values
    CNBatches::group  *g_chain;                // anchor of group list
    long        act_group;              // current group
    long        remaining;              // number of still needed trials 
                                        // per group
    double      sum;                    // sum of a group's trials
    double      sqsum;                  // sqyare sum of a group's trials
    double      gm,gv;                  // group mean, group variance
    double      gm_sum,gm_sqsum;        // random value sum resp.
                                        // square random value sum of
                                        // group means
    double      gv_sum,gv_sqsum;        // random value sum resp.
                                        // square random value sum of
                                        // group variances
        
    // interval characteristics 
    CNBatches::interval  *i_chain;                // anchor of interval list 
                                        // (first element)
    CNBatches::interval_out * i_out_chain;    // anchor of list for 
                                        // interval output
                                        // (first element)
    long        wasted_left;            // wasted random numbers
    long        wasted_right;           // numbers > maximum (1.Serie) resp.
                                        // < minimum (1.Serie)
    long        ah;                     // absolute hit frequency
    double      rh,rh_mean;             // relative hit frequency
                                        // and mean
    double      rh_sum;                 // sum of all rel. frequencies
    double      f,g;                    // df. and compl. df
    double      f_sum;                  // sum of all df-values
                                        // in interval
    double      f_sqsum;                // square sum ...
    double      f_mean;                 // mean ...
    double      f_var;                  // variance ...
    double      f_bayes;                // rel. bayes-error ...
 
    double      g_mean;                 // mean of compl. df.
    double      g_bayes;                // rel. bayes-error ...

    double      fg_conf;                // confidenceinterval of  
                                        // f and g (identical) 
                                        // confidence probability,
                                        // defined by user
    
    Phase       phase;                  // state of evaluation 
    double      min_val,max_val;        // minimal/maximaize value
                                        // of collected random numbers
    bool        end_reached;            // true: end of evaluation reached
                                        // 03.12.93 (fuss)
    unsigned 
        long    nrv;                    // number of trials 03.12.93 (fuss)

    double      confi(long, double, double) const;
                                        // returns confidence interval
    double      variance(long, double, double) const;
                                        // returns variance
    
    void        calc_groups() ;         // evaluation function of 
                                        // characteristic group values
                                        // output function of 
                                        // characteristic group values
    void        print_groups(ostream &strm = cout) const;
                                        // output of group results      
    void        print_group_results( ostream &strm = cout);
    void        calc_intervals();       // evaluation function of
                                        // characteristic interval values
    void        print_intervals(ostream &strm = cout, Type = DF) const; 
                                        // output function of  
                                        // characteristic interval values
  public:    
    virtual double         mean() const 
                { return (act_group) ? (gm_sum/act_group) : DBL_MAX ; }
    virtual double variance() const
                { return (act_group) ? (gv_sum/act_group) : DBL_MAX ; }
    double                sigma() { 
        // returns (deviation/sqrt(2)) of group means!
        // can be plotted as interval at curve point
        calc_groups();
        return ( sqrt(variance(act_group,gm_sum,gm_sqsum)/2) );
            }
    double                bayes_err() { 
        calc_groups();
        return (act_group>5) 
        ? sqrt((act_group/(act_group-5)*(gm_sqsum/(gm_sum*gm_sum)-1./act_group)))
        : 0.0 ; }
    double                rd_conf() { 
        calc_groups();
        double vari =  variance (act_group, gm_sum, gm_sqsum);
        return(confi(act_group-1,conf,vari));
        }
        
    virtual void          put( double );
    virtual unsigned long trials() const { return nrv; }
    virtual double        min()    const { return min_val; }
    virtual double        max()    const { return max_val; }

    virtual bool          end()    const { return end_reached; } // 03.12.93 
    virtual Phase         status() const { return phase; }       //  (fuss)
    virtual void          reset();
    void                  print( Type type = CNStatistics::DF, 
                                ostream &strm = cout )  const;


    /***** Member functions required by CNCL *********************************/
    virtual CNClassDesc class_desc() const      // CNClass description
    {
        return CN_BATCHES;
    };
            
    virtual bool is_a(CNClassDesc desc) const   // Type checking
    {
        return desc == CN_BATCHES ? TRUE : CNStatistics::is_a(desc);
    };
        
    static CNBatches *cast_from_object(CNObject *obj) // Safe type cast
    {
#   ifdef NO_TYPE_CHECK
        return (CNBatches *)obj;
#   else
        return (CNBatches *)( obj->is_a(CN_BATCHES)
               ? obj : fatal_type(obj->class_desc(), CN_BATCHES) );
#   endif
    }
    
    static CNObject *new_object(CNParam *param = NIL) // Object creation
    { return param ? new CNBatches(param) : new CNBatches; }
    
    // Print/debug output
    virtual void print(ostream &strm = cout) const;
    virtual void dump (ostream &strm = cout) const;
};

#endif /**__CNBatches_h**/
