//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: DLRE.c,v 0.35 1996-08-07 18:02:51+02 steppler Exp $
 *
 * Class: CNDLRE --- Discrete LRE (LRE III), base class
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#include <CNCL/Class.h>
#include <stdlib.h>
#include <math.h>

#include "DLRE.h"


CNDLRE::CNDLRE(double * xvalues, long level, double eerror, 
	       double pre_first, const char* aName, const char* aDescription, 
	       unsigned long mmax_nrv)
    : CNStatistics(aName, aDescription)
{
    reason = NO_REASON;
    if( level > 1 ) 
	x = new result[level+1];// one element more because of get_index
    else
	CNCL::fatal("CNDLRE::CNDLRE:", " No. of levels <= 1 !\n");
  
    if( xvalues == NIL )    // fill x[].x with default-values 
    {                     // 0 ... level-1
	for( long i = 0; i<=level; i++) 
        {   
	    x[i].x = i; x[i].h = 0; x[i].c = 0;
	    //printf( "%d %f\n", i, x[i].x);
        }
	index_min = 0;
	xmin      = 0.0; 
	index_max = level;
	xmax      = double(level - 1);
	int_size  = 1.0;
	equi_dist = TRUE;
    }
    else 
    {
	long i = 0;
	while( ((xvalues+i) != NIL) && (i<level) )
        {
	    x[i].x = xvalues[i]; x[i].h = 0; x[i].c = 0;
	    i++;
        }
	x[level].h = x[level].c = 0;
	index_min = 0;
	xmin      = (double) x[0].x; 
	index_max = level;
	xmax      = (double) x[level-1].x;
	int_size  = 0.0;
	equi_dist = FALSE;
    }
  
    if( pre_first < xmin ) pre_index = index_min;
    else if( pre_first > xmax ) pre_index = index_max-1;
    else 
    {
	pre_index = 0;
	while( pre_index<level ) 
	    if( x[pre_index].x == pre_first ) break;
	    else pre_index++;
    }
  
    reler_max = eerror;
    nrv = 0;
    base = 1.0;
    pre_rv = pre_first;     // state of beginning
    cur_level_index = index_min;
    rv_sum        = 0.0;
    rv_square_sum = 0.0;
    h  = 0;
    n0 = 2.0/eerror/eerror > 1000 ? long( 2.0/eerror/eerror ) : 1000;
    max_nrv = mmax_nrv;
    wasted_right = wasted_left = 0;
    min_value = DBL_MAX;
    max_value = -DBL_MAX;
    phase = INITIALIZE;
    end_reached = FALSE; 
}


CNDLRE::CNDLRE(double xxmin, double xxmax, double iint_size, double eerror, 
	       double pre_first, const char* aName, const char* aDescription, 
	       unsigned long mmax_nrv)
    : CNStatistics(aName, aDescription)
{
    reason = NO_REASON;
    if( iint_size <= 0 ) 
	CNCL::fatal("CNDLRE::CNDLRE:", " interval width <= 0!\n");
    if( xxmin >= xxmax )
    {
        CNCL::warning("CNDLRE::CNDLRE:", "X_min >= X_max; exchanged!\n");
	double help = xxmin; xxmin = xxmax; xxmax = help;
    }
    long level = CNabs((long)((xxmax - xxmin) / iint_size)) + 1;
    if( level <= 1 ) 
	CNCL::fatal("CNDLRE::CNDLRE:", " No. of levels <= 1 !\n");
  
    x = new result[level+1];
    for( int i = 0; i<=level; i++) 
    {   
	x[i].x = (i*iint_size)+xxmin; x[i].h = 0; x[i].c = 0;
	//printf( "%d %f\n", i, x[i].x);
    }
    index_min       = 0; 
    xmin            = xxmin; 
    index_max       = (long) (level);
    xmax            = xxmax;
    int_size        = iint_size;
  
    if( pre_first < xmin ) pre_index = index_min;
    else if( pre_first > xmax ) pre_index = index_max;
    else 
    {
	pre_index= 0;
	while( pre_index<level ) 
	    if( x[pre_index].x == pre_first ) break;
	    else pre_index++;
    }
  
    reler_max = eerror;
    nrv = 0;
    base = 1.0;
    pre_rv = pre_first;     // state of beginning
    cur_level_index = index_min;
    rv_sum        = 0.0;
    rv_square_sum = 0.0;
    h  = 0;
    n0 = 2.0/eerror/eerror > 1000 ? long( 2.0/eerror/eerror ) : 1000;
    max_nrv = mmax_nrv;
    wasted_right = wasted_left = 0;
    min_value = DBL_MAX;
    max_value = -DBL_MAX;
    phase = INITIALIZE;
    end_reached = FALSE;
    equi_dist = TRUE;
}


CNDLRE::~CNDLRE()
{
    delete [] x;
}



void CNDLRE::reset()
{
    nrv = 0;
    h = 0;

    rv_sum        = 0.0;
    rv_square_sum = 0.0;
    wasted_right = wasted_left = 0;
    min_value = DBL_MAX;
    max_value = -DBL_MAX;

    phase = INITIALIZE;
    end_reached = FALSE;
  
    for (long i=index_min; i<index_max; i++)
    {
	x[ i ].h = 0;
	x[ i ].c = 0;
    }
}



long CNDLRE::get_index( double& cur_rv )
{
    if( cur_rv < xmin )       return LOWER;
    else if ( cur_rv > xmax ) return GREATER;
    else if( equi_dist ) 
    {
	long index; //= long( (cur_rv-xmin)/(xmax-xmin)*(index_max-1) + 0.1);

	// truncate value
	cur_rv = double(long(rint(cur_rv / int_size))) * int_size;
	
	index = long(index_min + (cur_rv - xmin) / int_size); 
	return (cur_rv == x[index].x) ? index : (long)NO_INDEX;
    }
    else 
    {
	// search the correct index (you know a better algorithm ? - Send it !)
	result * start = x;
	result * end   = x+index_max;
	result * lauf;
	register int step;
	while( (step = (end - start)) >= 1)
	{
	    lauf = start + (step >> 1);  // `>>1' equals `/2' but is faster
	    if( lauf->x == cur_rv )    
		return (step>>1) + (start-x); 
	    else if( lauf->x > cur_rv ) end = lauf;
	    else if( lauf->x < cur_rv ) start = lauf;
	}
	return NO_INDEX;
	//return long(cur_rv - index_min - xmin);
    } 
}



double CNDLRE::p( double xt )
{
    long index = get_index( xt );
    return (nrv > 0 && index >= 0) ? x[index].h / double(nrv) : 0.0;
}


double CNDLRE::mean() const
{
    return (nrv>0)? double(rv_sum)/nrv: DBL_MAX;
}


double CNDLRE::variance() const
{
    return (nrv>1)? 
	double(rv_square_sum-mean()*mean()*nrv)/(nrv - 1.0):0.0;
}



/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNDLRE::print(ostream &strm) const
{
    strm << "..." << endl;
}

// Debug output
void CNDLRE::dump(ostream &strm) const
{
    strm << "CNDLRE { $Revision: 0.35 $ ..."
	 << " }" << endl;
}

// IOStream operator <<
ostream &operator << (ostream &strm, const CNDLRE &obj)
{
    obj.print(strm);
    return strm;
}

ostream &operator << (ostream &strm, const CNDLRE *obj)
{
    if(obj)
	obj->print(strm);
    else
	strm << "(NIL)";
    return strm;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNDLRE
static CNClass CNDLRE_desc("CNDLRE", "$Revision: 0.35 $", 0);

// "Type" for type checking functions
CNClassDesc CN_DLRE = &CNDLRE_desc;
