//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: DLRE.h,v 0.35 1996-08-07 18:02:53+02 steppler Exp $
 *
 * Class: CNDLRE --- Discrete LRE (LRE III), base class 
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#ifndef __CNDLRE_h
#define __CNDLRE_h


#include <CNCL/Statistics.h>		// Base class

#include <limits.h>
#include <float.h>
#include <math.h>
#include <iomanip.h>
#include <CNCL/minmax.h>

extern CNClassDesc CN_DLRE;	// Class CNDLRE description


/*
 * The class CNDLRE
 */

class CNDLRE : public CNStatistics
{
protected:
    enum {NO_INDEX=-3, GREATER=-2, LOWER=-1};    // returned by get_index
    enum Reason {OK, MIN, LAST, NO_REASON };    // reason for status = END
    
    struct result 
    {
	double x;
	long h;
	long c;
    };
  
public:
    struct resultline 
    { 
	double vf;        // F, G or P value (depends on derived class)
	double x;         // x value
	double relerr;    // relative error 
	double rho;       // local correlation
	double sigrho;    // relative error of correlation
	unsigned long nx; // absolute number of hits of x
    };

public:	/***** Constructors ******************************************/
    // constructor for non-equi-distant x-values
    CNDLRE(double*, long, double, double = 0.0, const char* = NIL,
	   const char* = NIL, unsigned long = ULONG_MAX);
    // constructor for equi-distant x-values
    CNDLRE(double, double, double, double, double = 0.0, 
	   const char* = NIL, const char* = NIL, 
	   unsigned long = ULONG_MAX);
    CNDLRE(CNParam *) {}
    // destructor
    ~CNDLRE();
  
protected:
    CNDLRE() {}
  
public:	/***** Public interface **************************************/
    virtual void put( double ) = 0;
  
    virtual double mean()     const;
    virtual double variance() const;
    virtual unsigned long trials() const { return nrv; }
    virtual double min()    const { return min_value; }
    virtual double max()    const { return max_value; }
    virtual bool   end()    const { return end_reached; }
    virtual Phase  status() const { return phase; }
    virtual void   reset();
    
    void set_base( double b ) { base = b; } // factor for cond. prob.
    virtual void change_error( double ne ) { reler_max = ne; } // change error
    
    virtual double cur_x_lev() { return x[ cur_level_index ].x; } 
    virtual double p( double xt ); // returns the prob. of xt 
    
    virtual long min_index() const { return index_min; }
    virtual long max_index() const { return index_max-1; }
    
    virtual const resultline *get_result( long ) = 0;

protected:
    long get_index( double& ); // get index of current x
  
private:	/***** Internal private members ******************************/
  
protected:
    result * x;         // array of results
    resultline line;    // temporary struct holding a single result-line 

    double        reler_max;      // max. relative error
    unsigned long nrv;            // number of trials
    unsigned long max_nrv;        // maximal number of trials
    double        min_value;      // minimal value of trials
    double        max_value;      // maximal value of trials
    long          wasted_left;    // values lower minimum
    long          wasted_right;   // values greater maximum
  
    double rv_sum;        // necess. for mean
    double rv_square_sum; // and variance
  
    long h;  // run time control (rtc) counter
    long n0; // trials without rtc
  
    long   xoffset;   // offset of x-values (step 1)
    double xmin;      // minimum of x
    double xmax;      // maximum of x
    long   index_min; // begin of array; normaly 0
    long   index_max; // end of array 
    bool   equi_dist; // 
    double int_size;  // distance between x values
  
    long   cur_index;       // index of current x-value
    double pre_rv;          // predecessor
    long   pre_index;       // index of predecessor
    double base;            // needed for conditional probabilties
    Phase  phase;           // state of DLREF
    bool   end_reached;     // end of evaluation reached ?
    Reason reason;          // reason for end reached
    long   cur_level_index; // index of current x-level

public:	/***** Member functions required by CNCL *********************/
    virtual CNClassDesc class_desc() const	// CNClass description
    { return CN_DLRE; }
  
    virtual bool is_a(CNClassDesc desc) const	// Type checking
    { return desc == CN_DLRE ? TRUE : CNStatistics::is_a(desc); }
  
    static CNDLRE *cast_from_object(CNObject *obj) // Safe type cast
    {
#   ifdef NO_TYPE_CHECK
	return (CNDLRE *)obj;
#   else
	return (CNDLRE *)( !obj || obj->is_a(CN_DLRE)
			   ? obj : fatal_type(obj->class_desc(), CN_DLRE) );
#   endif
    }
  
  
    // Print/debug output
    virtual void print(ostream &strm = cout) const;
    virtual void dump (ostream &strm = cout) const;
};


ostream &operator << (ostream &strm, const CNDLRE &obj);
ostream &operator << (ostream &strm, const CNDLRE *obj);

#endif /**__CNDLRE_h**/
