//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_            C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_                      N E T W O R K S
 * |_        |_  |_  |_  |_        |_                            C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_                  L I B R A R Y
 *
 * $Id: LRE.c,v 0.35 1996-08-07 18:03:07+02 steppler Exp $
 *
 * Class: CNLRE --- Abstract LRE base class
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#include "LRE.h"

#include <iomanip.h>


CNLRE::CNLRE(double min, double max, double reler, int levn,
	     Scale scale, int maxsort, const char* n, const char* t)
    : CNStatistics(n, t)
{
    phase           = CNStatistics::INITIALIZE;
    end_reached     = FALSE;                     // 03.12.93 (fuss)
    minlev          = min;
    maxlev          = max;
    relerror        = reler;
    maxreler        = reler * reler;
    nsec            = levn;
    fl_log          = scale;

    if ( maxsort == 0 )
	maxsortfeldgroesse = LONG_MAX;
    else
	maxsortfeldgroesse = maxsort;

    if ( fl_log == LOG )
    {
	levfac = (double) pow (
	    ((minlev*0.95) / (double)maxlev), (1.0 / (double)nsec));
    }
    else
    {
	levfac = (maxlev - minlev *0.95) / (double)nsec;
    }
    nrv             = 0;
    rv_sum          = 0.0;
    rv_square_sum   = 0.0;
    fl_init         = 1;
    min_value       = DBL_MAX;
    max_value       = -DBL_MAX;

    pfirst          = new result[2*nsec];
    pr              = pfirst;
    pr->sortmsize   = 0;
    pr->transcount  = 0;
    pr->nxvalue     = 0;

    sortmergeend    = sortmerge = new sm;
    long a;
    n0 = long(CNMAX((long)((1.0-maxlev)/(maxlev*maxreler)),(long)1000));
    a  = long(CNMAX((int)((1.0-minlev)/(maxreler)),(int) n0));
    a  = long(CNMIN(a,(int)((double)maxsortfeldgroesse/1.1)));
    n0 = long(CNMIN(n0,(long)a));

    sortquickend    = sortquick =  new sm[(int)((double)a*1.1) + 1];
    sortfeldgroesse = (int)((double)a*1.1);
    base            = 1.0;             // fuss (06.07.93)
    min_ind = max_ind = 0;
}

CNLRE::~CNLRE()
{
    if( !fl_init) sortquick--;
    delete [] sortquick;
    delete [] pfirst;
    delete sortmerge;
}

void CNLRE::reset ()
{
    error( "CNLRE::reset: Not yet implemented !" );
}
    

/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNLRE::print( CNStatistics::Type type, ostream &strm) const
{
    long a,r;
    result *ppr = pr;
    double rho,d,d1;
    double sigrho, v;                   // fuss (02.02.93)
    double rf,af;
    double d_pre=0, rho_pre=0, x=0.0;
    long nx;
    double nf = (double)nrv;
    
    if(ppr != NULL)
    {
//  if ( phase != CNStatistics::END)    // fuss (01.07.93)
//  make_level();
//  phase = CNStatistics::ITERATE;
	ppr = pr - 1;
	strm << setprecision(7);
	strm.setf(ios::left, ios::fixed);
	strm.setf(ios::dec);
	strm << "#LRE RESULT (THIS IS A MAGIC LINE)" << endl;
	strm << "#---------------------------------------------------------------------------" << endl;
	strm << "#   This LRE algortihm was written by Dr. M Richter, 1992"
	     << endl;
        strm << "#---------------------------------------------------------------------------" << endl;
        strm << "#Name: " << name << endl;
        strm << "#Text: " << text << endl;
	strm << "#Default values: max.rel. error = " << setw(6)
	     << relerror*100;
	if( type==CNStatistics::DF )
	{
	    strm << "% \tF_max = " << setw(4) << maxlev;
	    strm << " F_min = " << setw(4) << minlev << endl;
	}
	else
	{
	    strm << "% \tG_max = " << setw(4) << maxlev;
	    strm << " G_min = " << setw(4) << minlev << endl;
	}
	strm << "#                No. of levels = " << nsec;
        if (fl_log == LOG ) strm << ", scale = log. " << endl;
        else strm << ", scale = lin. " << endl;
	strm << "#trials: " << nrv;
	strm << "\tmean: " << mean() << "\tvariance: " << variance() << endl;
	strm << "#---------------------------------------------------------------------------" << endl;
	if( type==CNStatistics::DF )
	    strm << "#F(x)           x               rel.error";
	else
	    strm << "#G(x)           x               rel.error";
	strm << "       rho             sigrho" << endl;
	a = ppr->transcount;
	r = ppr->sortmsize;
	while(ppr > pfirst)
	{
	    rf = (double)r;
	    af = (double)a;
	    nx = ppr->nxvalue;
	    x  = ppr->xlevlim;
	    if (nf < 1000.0 || rf < 100.0 || af < 10.0 || ( rf -af) < 10.0 )
	    {
		strm << "#";
		if( (nx > 1) && ((r-nx) != 0) )     //  fuss (11.05.93)
		{
		    strm << setiosflags(ios::scientific) ;
		    strm << "  " << (double)(r-nx)*base/nf;
		    strm << x << endl;
		} 
		else                    // fuss (11.05.93)
		{
		    strm << setiosflags(ios::scientific) ; 
		    strm << "  " << rf/nf*base;
		    strm << x << endl;
		} 
	    }
	    else
	    {
		rho = 1.0 - ((af/rf)/(1.0 - rf/nf));
		d1 = ((1 + rho) / (1-rho)) * ((1-rf/nf)/rf);
		d = sqrt(d1);
		v = nf - rf;
		sigrho = sqrt( af*(((1 - (af/rf)) / (rf*rf)) +
				   ((1 - (af/v)) / (v*v) )));
		if(nx > 1)
		{                                                      
		    // fuss (11.05.93)
		    strm << setw(12) << setiosflags(ios::scientific);
		    strm << (double)(r-nx)*base/nf;
		    strm << "  " << setw(14) << x;       
		    // fuss (16.05.93)
		    strm << setw(16) << d_pre;
		    strm << setw(16) << rho_pre << endl;
		}
		// fuss (11.05.93)
		strm << setw(12) << setiosflags(ios::scientific)
		     << (double)rf*base/nf;
		strm << "  " << setw(14) << x;
		strm << setw(16) << d << setw(16) << rho;
		strm << setw(16) << sigrho << endl;
		rho_pre = rho;
		d_pre = d;
	    }
	    ppr--;
	    a += ppr->transcount;
	    r += ppr->sortmsize;
	}
    }
}

// Debug output
void CNLRE::dump(ostream &strm) const
{
    if ( pr == NIL)
	error("CNLRE::dump: Nothing in memory");
    else
    {
	strm << "sortqick= " << sortquick ;
	strm << setiosflags(ios::scientific);
	strm << "x       = " << pr->xlevlim ;
	strm << "n       = " << setiosflags(ios::fixed) << setw(6) << nrv;
	strm << "r       = " << pr->sortmsize;
	strm << "a       = " << pr->transcount;
	strm << "nxvalue = " << pr->nxvalue << endl;
    }
}




void CNLRE::make_level()
{

    maxreler = 1.0;
    while ( phase != CNStatistics::END)
	next_level();
}

void CNLRE::next_level()
{
    double reler;
    result *presult;
    
    if (fl_init) /* bei der Initalisierung korretur vornehmen */
    {
	sortquick++;
	nrv--;  /* erster Wert wird weggeschmissen */
	pr->sortmsize--;
	reler = 0.0;    /* auf alle Faelle neues Level */
    }
    else
    {
	reler = compreler (nrv,pr) ;
    }
    
    if (reler <=  maxreler )
    {
	if ( FX(pr) > minlev  && ( (FX(pr) > levfac) || (int)fl_log ))
	    /* noch ein Level */
        {
	    /* zu wenig platz fuer levels allokiert */
	    if ( pr - pfirst >= 2 * nsec)
		error("CNLRE::next_level: too much levels");

	    max_ind++; // olly
	    presult = pr+1;
	    presult->transcount = 0;
	    presult->nxvalue = 0;
	    if (fl_init) /* erstes Level */
            {
		fl_init = 0;
		phase = CNStatistics::ITERATE;
		presult->sortmsize = (int) FIRSTLEVEL();
            }
	    else if (fl_log == LOG) /* log Massstab */
            {
		presult->sortmsize = (int) LOGLEVEL();
            }
	    else
            {
		presult->sortmsize = (int) LINLEVEL();
            }
	    /************************************************/
	    /* umsortieren und ermittlung des neuen xwertes */
	    /************************************************/
            //echoresult(pr,nrv);  //  DEBUG 
	    neusort(presult);
            //echoresult(presult,nrv);  //  DEBUG 
	    
	    n0 = NEWN0(presult);
	    /*Sortierfeld nicht leer daher weitermachen */
	    if ( phase != CNStatistics::END)
            {
		pr->transcount -= presult->transcount;
		pr->sortmsize -= presult->sortmsize;
            }
	    else
            {
		presult->xlevlim = end_lev();
            }
	    pr = presult;
        }
	else
        {  /* kleinstes Level unterschritten */
	    /* zu wenig platz fuer levels allokiert */
	    if ( pr - pfirst >= 2 * nsec)
		error("CNLRE::next_level: too much levels") ;
	    pr++;
	    pr->xlevlim = end_lev();
	    pr->transcount = 0;
	    pr->nxvalue = 0;
	    phase = CNStatistics::END;
	    end_reached = TRUE; // 03.12.93 (fuss)
        }
    }
    else    /* relativer Fehler nicht unterschritten */
    {
	n0 = NEWN0(pr);
    }
}


double CNLRE::compreler (long nrv, result *pr)
{
    const double nf = ((double)nrv);
    const double rf = ((double)pr->sortmsize);
    const double af = ((double)pr->transcount);

    double rn,ar;
    
    if (nf < 1000.0 || rf < 100.0 || af < 10.0 || ( rf -af) < 10.0 )
	return (1.0);

    rn = rf/nf;
    ar = af/rf;
    return( ( 2.0 *(1 - rn) / ar - 1.0 ) * ( (1.0 - rn) / rf ));
}

void CNLRE::schaufelum()
{
    
    sm *source;
    sm *dest;
    sm *end;
    sm *start;
    double savereler;
    
    if ( sortfeldgroesse ==  maxsortfeldgroesse )
    {
	savereler = maxreler;
	maxreler = 1.0;
	next_level();
	maxreler = savereler;
	return;
    }
    sortfeldgroesse = (int)((double) sortfeldgroesse * 1.25);
    if ( sortfeldgroesse > maxsortfeldgroesse )
	sortfeldgroesse = maxsortfeldgroesse;
    
    dest = new sm[sortfeldgroesse + 1];
    dest++;
    start = dest; 
    source = sortquick;
    end = sortquickend;
    int test_i =0;
    while ( source != end)
    {
	*dest++ = *source++;
	test_i++;
    }
    delete --sortquick; /* erster Wert wurde weggeschmissen */
    sortquick = start;
    sortquickend = dest;
}

double CNLRE::mean() const
{
    return (nrv>0)? double(rv_sum)/nrv: DBL_MAX;
}

double CNLRE::variance() const
{
    return (nrv>1)? 
	double(rv_square_sum-mean()*mean()*nrv)/(nrv - 1.0):0.0;
}


const struct CNLRE::resultline *CNLRE::get_result( long index )
{
    if ((index < min_index()) || (index > max_index()))
    {
	warning("CNLRE::get_result():", " index out of range.");
	return NIL;
    }
    
    result *ppr = pr;
    long a = ppr->transcount;
    long r = ppr->sortmsize;
    long i = max_index();
    while (i-- > index) // we're coming from upper end of array here
    {
	ppr--;
	a += ppr->transcount;
	r += ppr->sortmsize;
    }
    double nf = double( nrv ), af = double( a ), rf = double( r );
    double F = rf / nf;
    if (nrv < 1000 || r < 100 || a < 10 || (r-a) < 10 )
    {
	line.rho    = 0.0;
	line.sigrho = 0.0;
	line.relerr = 0.0;
    }
    else
    {
	double rho = ((F == 1.0) || (rf == 0.0))
	    ? 0.0 : 1.0 - (af/rf) / (1.0 - F);
	double vf = nf - rf;
	line.sigrho = ((rf == 0.0) || (vf == 0.0)) ? 0.0
	    : sqrt( af * ((1 - af/rf) / (rf*rf) + (1 - af/vf) / (vf*vf)) );
	line.relerr = ((rf == 0.0) || (rho == 1.0)) ? 0.0
	    : sqrt( (1.0 - rf/nf)/rf * (1.0 + rho)/(1.0 - rho) );
	line.rho = rho;
    }
    line.vf = F * base;
    line.nx = ppr->nxvalue;
    line.x  = ppr->xlevlim;

    return &line;
}


/***** CNCL stuff for type information and exemplar objects ******************/

// Describing object for class CNLRE
static CNClass CNLRE_desc("CNLRE", "$Revision: 0.35 $", NIL);

// "Type" for type checking functions
CNClassDesc CN_LRE = &CNLRE_desc;
