//   -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: Moments.c,v 0.34 1996-08-07 18:03:15+02 steppler Exp $
 *
 * CNClass: CNMoments --- Simple stat. evaluation: mean, variance, skewness etc.
 *
 * Derived from SIC - Simulation In C++
 *
 *****************************************************************************
 * Copyright (C) 1992-1996   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@comnets.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#include "Moments.h"

#include <math.h>
#include <limits.h>
#include <float.h>

CNMoments::CNMoments(const char* aName, const char* aDescription)
    : CNStatistics(aName, aDescription),
      sum(0), wsum(0), sqsum(0), cbsum(0), n(0), 
      min_value(DBL_MAX), max_value(-DBL_MAX)
{}

void CNMoments::put(double x_i, double w_i)
{
    double square = w_i * x_i * x_i;
    sum   += w_i * x_i;
    wsum  += w_i;
    sqsum += square;
    cbsum += x_i * square;
    if (x_i > max_value)
	max_value = x_i;
    if (x_i < min_value)
	min_value = x_i;
    n++;
}

void CNMoments::reset()
{
    n = 0;
    sum = wsum = sqsum = cbsum = 0;
    min_value = DBL_MAX;
    max_value = -DBL_MAX;
}

double CNMoments::mean() const
{
    return (n != 0 && wsum != 0.0) ? (sum / wsum) : 0.0 ;
}

double CNMoments::variance() const
{
    return (wsum != 0.0) ?
	(sqsum / wsum - (sum * sum) / (wsum * wsum)) : 0.0 ;
}

double CNMoments::relative_variance() const
{
    return (sum == 0.0) ?
	0.0 : variance() * wsum * wsum / (sum * sum) ;
}

double CNMoments::M_2() const
{
    return (wsum != 0.0) ? sqsum / wsum : 0.0 ;
}

double CNMoments::M_3() const
{
    return (wsum != 0.0) ? cbsum / wsum : 0.0 ; 
}

double CNMoments::Z_3() const
{
    double m_1 = mean();

    return M_3() - 3 * m_1 * M_2() + 2 * m_1 * m_1 * m_1;
}

double CNMoments::skewness() const
{
    double var = variance();

    return (var == 0.0) ? 0.0 : Z_3() / (var * sqrt(var));
}

double CNMoments::deviation() const
{
    return sqrt(variance());
}

double CNMoments::relative_deviation() const
{
    double m_1 = mean();

    return (m_1 == 0.0) ? 0.0 : sqrt(variance()) / mean();
}

/***** Default I/O member function for CNCL classes **************************/

// CNNormal output
void CNMoments::print(ostream &strm) const
{
    strm << "Evaluation " << name << " (CNMoments)" << endl;
    strm << "Description        : " << text                 << endl;
    strm << "Trials             : " << n                    << endl;
    strm << "Mean               : " << mean()               << endl;
    strm << "Variance           : " << variance()           << endl;
    strm << "Relative variance  : " << relative_variance()  << endl;
    strm << "Deviation          : " << deviation()          << endl;
    strm << "Relative deviation : " << relative_deviation() << endl;
    strm << "Skewness           : " << skewness()           << endl;
    strm << "Minimum            : " << min()                << endl;
    strm << "Maximum            : " << max()                << endl;
}


// Debug output
void CNMoments::dump(ostream &strm) const
{
    strm << "CNMoments { " << endl;
    strm << "name = " << name << endl;
    strm << "text = " << text << endl;
    strm << "sum = " << sum << "  sqsum = " << sqsum
	 << "  cbsum = " << cbsum << endl;
    strm << "n = " << n << "  min_value = " << min_value
	 << "  max_value = " << max_value << endl;
    strm << "Evaluation " << name << " (CNMoments)" << endl;
    strm << "Trials             : " << n                    << endl;
    strm << "Mean               : " << mean()               << endl;
    strm << "Variance           : " << variance()           << endl;
    strm << "Relative variance  : " << relative_variance()  << endl;
    strm << "Deviation          : " << deviation()          << endl;
    strm << "Relative deviation : " << relative_deviation() << endl;
    strm << "Skewness           : " << skewness()           << endl;
    strm << "Minimum            : " << min()                << endl;
    strm << "Maximum            : " << max()                << endl;
    strm << " }" << endl;
}

/***** CNCL stuff for type information and exemplar objects ******************/

// Describing object for class CNMoments
static CNClass CNMoments_desc("CNMoments", "$Revision: 0.34 $",
			      CNMoments::new_object);

// "Type" for type checking functions
CNClassDesc CN_MOMENTS = &CNMoments_desc;






