#include <iostream.h>

#include <qlayout.h>

#include <kconfig.h>
#include <kapp.h>

#include "TConnectionDlg.h"

#include "TStoreType.h"
#include "TEditConnectionDlg.h"
#include "TDataListBoxDlg.h"

TConnectionWindow::TConnectionWindow(QWidget *parent, const char *name): QWindow(parent, name) {
  initMetaObject();

  fRootObject = NULL;
  fActiveObjectTree =  NULL;

  // Right hand frame
  QFrame *frame = new QFrame(this, "Frame");
  frame->setFrameStyle(QFrame::Box | QFrame::Sunken);

  listBoxConnections = new TDataListBox(frame, "List Connections");
  listBoxConnections->setMinimumSize(100,100);

  btnAdd = new QPushButton("Add", frame, "Btn Add");
  btnEdit = new QPushButton("Edit", frame, "Btn Edit");
  btnDel = new QPushButton("Del", frame, "Btn Del");
  btnNew = new QPushButton("New", frame, "Btn New");

  btnAdd->adjustSize();
  btnEdit->adjustSize();
  btnDel->adjustSize();
  btnNew->adjustSize();

  btnEdit->setFixedSize(btnEdit->width()+5, btnEdit->height());
  btnAdd->setFixedSize(btnEdit->width()+5, btnEdit->height());
  btnDel->setFixedSize(btnEdit->width()+5, btnEdit->height());
  btnNew->setFixedSize(btnEdit->width()+5, btnEdit->height());

  QVBoxLayout *frameLayout = new QVBoxLayout(frame, 5);
  QHBoxLayout *frameBtnLayout = new QHBoxLayout();

  frameLayout->addWidget(listBoxConnections, 1);
  frameLayout->addLayout(frameBtnLayout);
  frameLayout->addStrut(btnEdit->width()*4 + 20 + 25);

  frameBtnLayout->addStretch(1);
  frameBtnLayout->addWidget(btnAdd);
  frameBtnLayout->addWidget(btnEdit);
  frameBtnLayout->addWidget(btnDel);
  frameBtnLayout->addSpacing(20);
  frameBtnLayout->addWidget(btnNew);
  frameBtnLayout->addStretch(1);

  frameLayout->activate();



  comboObjList = new QComboBox(this, "Object ComboBox");
  comboObjList->adjustSize();
  comboObjList->setFixedHeight(comboObjList->height());
  
  listBoxSignals = new TDataListBox(this, "List available Signals");

  listBoxSignals->setMinimumSize(100,100);  

  QGridLayout *grid = new QGridLayout(this, 2, 2, 5);
  grid->setRowStretch(1, 1);

  grid->addWidget(comboObjList, 0,0, AlignLeft);
  grid->addWidget(listBoxSignals, 1, 0);
  grid->addWidget(frame, 1, 1);

  grid->activate();


  connect(comboObjList, SIGNAL(activated(const char *)), 
	  SLOT(comboActivated(const char *)));
  connect(btnNew, SIGNAL(clicked()), SLOT(newClicked()));
  connect(btnAdd, SIGNAL(clicked()), SLOT(addClicked()));
  connect(btnDel, SIGNAL(clicked()), SLOT(delClicked()));
  connect(btnEdit, SIGNAL(clicked()), SLOT(editClicked()));
  connect(listBoxSignals, SIGNAL(selected(int)), SLOT(signalDblClick(int)));
  connect(listBoxSignals, SIGNAL(highlighted(int)), SLOT(updateConnections(int)));
  connect(listBoxConnections, SIGNAL(highlighted(int)), SLOT(connectionHighlighted(int)));

  //cout << "Load the Connection Window geometry" << endl;
  KConfig *config = KApplication::getKApplication()->getConfig();
  config->setGroup( "WindosPos" );
  if (config) {
    QString str;
    str = config->readEntry("ConnectionWindow");
    if (!str.isEmpty() && str!="") {
      int a,b,c,d;
      if (4 == sscanf(str.data(), "%d %d %d %d", &a, &b, &c, &d))
	setGeometry(a,b,c,d);
    }
  }
}



TConnectionWindow::~TConnectionWindow() {
	//cout << "Delete the Connection Window" << endl;
  KConfig *config = KApplication::getKApplication()->getConfig();
  config->setGroup( "WindosPos" );
    
  QString str;
  str.sprintf("%d %d %d %d", x(), y(), width(), height());
  config->writeEntry("ConnectionWindow", str);
  config->sync();
}



void TConnectionWindow::setActive(void) {
  TStoreType *tree = fRootObject->find(sender());

  activeObject(tree);
}


void TConnectionWindow::activeObject(TStoreType *w) {
	if (w && fActiveObjectTree!=w) {
		updateObjectList(w);
    
		// Remove all of the connections to the current Object Tree
		if (fActiveObjectTree) {
			disconnect(this, SIGNAL(newSignal(const char *, const char*)),
					   fActiveObjectTree->layoutWindow(), 
					   SLOT(addSignalCode(const char *, const char *)));
			disconnect(this, SIGNAL(newSlot(const char *, const char*)),
					   fActiveObjectTree->layoutWindow(), 
					   SLOT(addSlotCode(const char *, const char *)));
		}
		fActiveObjectTree = w;
    
		// Add all of the connections to the new Object Tree
		connect(this, SIGNAL(newSignal(const char *, const char*)),
				fActiveObjectTree->layoutWindow(), 
				SLOT(addSignalCode(const char *, const char *)));
		connect(this, SIGNAL(newSlot(const char *, const char*)),
				fActiveObjectTree->layoutWindow(), 
				SLOT(addSlotCode(const char *, const char *)));
    
		// delete all the current properties
		listBoxSignals->clear();
    
		// Add the required properties for this class
		QString str;
		if (fRootObject == w) {
			TSignalList *list = (TSignalList*)fRootObject->findData("CreatedSignals");
			if (list) {
				TSignalListIt sit(*list);
				for ( ; sit.current() ; ++sit) {
					str.sprintf("%s ( %s )", (const char *)sit.current()->name, 
								(const char*)sit.current()->args);
					listBoxSignals->insertItem(str, sit.current());
				}
			}
		}
		if (w->getSignals()) {
			TSignalListIt it(*w->getSignals());
			for (it.toFirst() ; it.current() ; ++it ) {
				str.sprintf("%s ( %s )", (const char *)it.current()->name, 
							(const char*)it.current()->args);
				listBoxSignals->insertItem(str, it.current());
			}
		}
		fActiveObjectTree = w;
    
		if (listBoxSignals->count())
			updateConnections();
    
		// set the current combo item to be the new object
		str = str.sprintf("%s:%s", w->object()->name(), 
						  (const char *)w->storeTypeName());
		for (int i=0 ; i<comboObjList->count() ; i++) {
			if (comboObjList->text(i) == str) {
				comboObjList->setCurrentItem(i);
				break;
			}
		}
	}
}
  


void TConnectionWindow::signalDblClick(int i) {
	//cout << "Double Click!!!" << endl;
	QString currSignal = listBoxSignals->text(i);

	updateConnections(i);
	if (0==listBoxConnections->count()) {
		//cout << "There are no connections for signal: " << currSignal << endl;

		// Look for the Connection List for this signal
		TSignal *signal = fActiveObjectTree->getSignals()->find(currSignal);
		if (signal) {
			// we have found the correct signal

			// Create a new slot for this connection
			QString slotName;
			slotName.sprintf("%s%s", fActiveObjectTree->object()->name(),
							 (const char *)signal->name);
			TSlot *ss = new TSlot(slotName, (const char *)signal->args);
			ss->created = TRUE;

			TSlotList *slist = (TSlotList*)fRootObject->findData("CreatedSlots");
			if (!slist) {
				slist = new TSlotList;
				fRootObject->addData("CreatedSlots", slist);
			}
			slist->append(ss);
			emit(newSlot(slotName, signal->args));

			// Create the Connection
			TConnection *con = new TConnection();
			con->dest = fRootObject;
			con->connection = ss;
			addConnectionToCurrent(signal, con);
		}
		updateConnections(i);
	}
}

void TConnectionWindow::updateConnections(int) {
  updateConnections();
}



void TConnectionWindow::updateConnections() {
  //cout << "updateConnections" << endl;
  //cout << "listboxSignals->currentItem() = " << listBoxSignals->currentItem() << endl;

  listBoxConnections->clear();

  // Make sure that we have a signal selected
  if (-1==listBoxSignals->currentItem())
    return;


  QString currSignal = listBoxSignals->text(listBoxSignals->currentItem());
  
  TLinkedConnections *cons = fActiveObjectTree->connections()->find(currSignal);
  if (cons) {
    TConnectionListIt cit(cons->connections);
    for ( ; cit.current() ; ++cit ) {
      QString foo;
      foo.sprintf ("%s.%s ( %s )", cit.current()->dest->object()->name(),
		   (const char *)cit.current()->connection->name,
		   (const char *)cit.current()->connection->args);
      listBoxConnections->insertItem(foo, cit.current());
    }
  }
  if (listBoxConnections->count())
    listBoxConnections->setCurrentItem(0);

  connectionHighlighted(listBoxConnections->currentItem());
}





void TConnectionWindow::addNewObject(TStoreType *tree) {
	QString str;

	//cout << "addNewObject: " << tree->object()->name() << endl;
	if (!updateObjectList(tree)) {
		str = str.sprintf("%s:%s", tree->object()->name(), tree->storeTypeName().data() );

		// Add the object in sorted order
		//cout << "TConnectionDlg:: Adding " << str << " to combo list." << endl;
		for (int i=0 ; i<comboObjList->count() ; i++) {
			if (str < comboObjList->text(i)) {
				comboObjList->insertItem(str, i);
				return;
			}
		}
		comboObjList->insertItem(str, -1);
	}
}




void TConnectionWindow::updateObject(TStoreType *tree, QString oldName) {
  QString str;
  QString old;

  if (tree==NULL)
    return;
  
  old = old.sprintf("%s:%s", (const char *)oldName, 
		    tree->storeTypeName().data());
  str = str.sprintf("%s:%s", tree->object()->name(), 
		    tree->storeTypeName().data());

  //cout << "TConnectionDlg:: Updating combo item from: " << old << " to " << str << endl;
  for (int i=0 ; i<comboObjList->count() ; i++) {
    if (comboObjList->text(i) == old) {
      comboObjList->removeItem(i);
      break;
    }
  }
  
  for (int i=0 ; i<comboObjList->count() ; i++) {
    if (str < comboObjList->text(i)) {
      comboObjList->insertItem(str, i);
      comboObjList->setCurrentItem(i);
      return;
    }
  }
  comboObjList->insertItem(str, -1);
  comboObjList->setCurrentItem(comboObjList->count());
}




void TConnectionWindow::comboActivated(const char *text) {
  QString lClass, lName;
  
  if (!fActiveObjectTree)
    return;
  
  // Find the class and name
  QString str = text;

  int pos = str.find(":");
  if (pos != -1) {
    lName = str.mid(0, pos);
    lClass = str.mid(pos+1, str.length());
    if (lName == "(null)")
      lName = "";
    
    //cout << "comboActivated looking for : " << lClass << " " << lName << endl;
    
    TStoreType *ot = fRootObject;
    while (ot) {
      if ((ot->storeTypeName() == lClass) &&
	  (ot->object()->name() == lName))
	break;
      ot = ot->next();
    }

    if (ot) {
		//cout << "Found an object matching" << endl;
      emit( sigObjectChanged(ot->object()));
    }
  }
}




bool TConnectionWindow::updateObjectList(TStoreType *tree) {
	if (!tree)
		return FALSE;

	TStoreType *ptr = tree->root();

	//cout << "updateObjectList: " << tree->object()->name() << endl;
  
	if (!fRootObject || (ptr != fRootObject)) {
		QString str;

		//cout << "updateObjectList: repopulate" << endl;
		comboObjList->clear();
		fRootObject=ptr;
		for ( ; ptr ; ptr=ptr->next() )
			addNewObject(ptr);

		str = str.sprintf("%s:%s", tree->object()->name(), 
						  tree->storeTypeName().data());

		for (int i=0 ; i<comboObjList->count() ; i++) 
			if (comboObjList->text(i) == str) 
				comboObjList->setCurrentItem(i);
		return TRUE;
	}
	else
		return FALSE;
}




void TConnectionWindow::addSlotsToDlg(TDataListBoxDlg *dlg, QString args, TStoreType *ptr, TSlotList *list) {
  if (list && dlg) {
    TSlotListIt slotIt(*list);
    for ( ; slotIt.current() ; ++slotIt ) {
      if (slotIt.current()->args == args) {
	TConnection *con = new TConnection();
	QString str;
	
	str.sprintf("Slot: %s.%s ( %s )",
		    ptr->object()->name(),
		    (const char *)slotIt.current()->name,
		    (const char *)slotIt.current()->args);
	con->dest = ptr;
	con->connection = slotIt.current();
	dlg->insertItem(str, con);
      }
    }
  }
}

void TConnectionWindow::addSignalsToDlg(TDataListBoxDlg *dlg, QString args, TStoreType *ptr, TSignalList *list) {
  if (list && dlg) {
    TSignalListIt sigIt(*list);
    for ( ; sigIt.current() ; ++sigIt ) {
      if (sigIt.current()->args == args) {
	TConnection *con = new TConnection();
	QString str;
	
	str.sprintf("Signal: %s.%s ( %s )", 
		    ptr->object()->name(),
		    (const char *)sigIt.current()->name,
		    (const char *)sigIt.current()->args);
	
	con->dest = ptr;
	con->connection = sigIt.current();
	dlg->insertItem(str, con);
      }
    }
  }
  else
    cerr << "TConnectionDlg: Either List or Dlg is missing" << endl;
}





void TConnectionWindow::addClicked() {
  int id = listBoxSignals->currentItem();
  if (-1==id)
    return;

  TSignal *signal = (TSignal*) listBoxSignals->data(id);
  QString args = signal->args;


  // Create a new list dialog
  TDataListBoxDlg *listDlg = new TDataListBoxDlg();

  TStoreType *ptr = fRootObject;
  
  // For the root object, there may also be a selection of
  // signals and slots which the user has created.
  addSignalsToDlg(listDlg, args, ptr, 
		  (TSignalList*)ptr->findData("CreatedSignals"));
  addSlotsToDlg(listDlg, args, ptr, 
		(TSlotList*)ptr->findData("CreatedSlots"));

  // For every object, add all of the signals and slots
  for ( ; ptr ; ptr = ptr->next()) {
    addSignalsToDlg(listDlg, args, ptr, ptr->getSignals());
    addSlotsToDlg(listDlg, args, ptr, ptr->getSlots());
  }


  if (listDlg->exec()) {
    // Create the connection and add it to the Signals list.
    //cout << "ok Clicked" << endl;
    TConnection *c = (TConnection*)listDlg->data();
    TConnection *con = new TConnection();
    con->dest = c->dest;
    con->connection = c->connection;

    addConnectionToCurrent(signal, con);
  }

  for (uint i=0 ; i<listDlg->count() ; i++)
    delete listDlg->data(i);
  delete listDlg;
}


void TConnectionWindow::addConnectionToCurrent(TSignal *signal, TConnection *con) {
	TLinkedConnectionsListIt cit(*fActiveObjectTree->connections());
	for ( ; cit.current() ; ++cit) {
		if (cit.current()->source==signal) {
			//cout << "Found a present connection list" << endl;
			cit.current()->connections.inSort(con);
			break;
		}
	}
	if (!cit.current()) {
		//cout << "creating a new Connection list" << endl;
		TLinkedConnections *lc = new TLinkedConnections;
		fActiveObjectTree->connections()->append(lc);
    
		lc->source = signal;
		lc->connections.inSort(con);
	}
	updateConnections();
}





void TConnectionWindow::delClicked() {
}


void TConnectionWindow::editClicked() {
  int i = listBoxConnections->currentItem();
  if (-1==i)
    return;

  // we can only do all this if we have a _created_ Signal/Slot
  TConnection *con = (TConnection*)listBoxConnections->data(i);
  
  TEditConnectionDlg *dlg = new TEditConnectionDlg(con->connection);
  if (dlg->exec()) {
    con->connection->name = dlg->name();
    con->connection->args = dlg->args();
    if (ctSlot==dlg->type()) {
      TSlot *s = (TSlot*)con->connection;
      s->privacy = dlg->privacy();
      s->isVirtual = dlg->isVirtual();
    }
  }
}


void TConnectionWindow::newClicked() {
  if (!fRootObject)
    return;


  TEditConnectionDlg *dlg = new TEditConnectionDlg;
  
  if (dlg->exec()) {
    switch (dlg->type()) {
    case ctSlot: {
      TSlot *ss = new TSlot(dlg->name(), dlg->args(), 
			    dlg->privacy(), dlg->isVirtual());
      ss->created = TRUE;

      TSlotList *slist = (TSlotList*)fRootObject->findData("CreatedSlots");
      if (!slist) {
	slist = new TSlotList;
	fRootObject->addData("CreatedSlots", slist);
      }
      slist->append(ss);
      emit(newSlot(dlg->name(), dlg->args()));
    }
    break;
    case ctSignal: {
      TSignal *ss = new TSignal(dlg->name(), dlg->args());
      ss->created = TRUE;

      TSignalList *slist = (TSignalList*)fRootObject->findData("CreatedSignals");
      if (!slist) {
	slist = new TSignalList;
	fRootObject->addData("CreatedSignals", slist);
      }
      slist->append(ss);
      emit(newSignal(dlg->name(), dlg->args()));
      break;
    }
    }
  }    
  delete dlg;
}

  
void TConnectionWindow::connectionHighlighted(int id) {
  //cout << "Connection Highlighted: " << id << endl;

  if (-1==id)
    btnEdit->setEnabled(false);
  else {
    TConnection *con = (TConnection*)listBoxConnections->data(id);

    btnEdit->setEnabled(con->connection->created);
  }
}
