/* pios.cc
 * Written by Joshua Rowe
 * Definitions for the streamable class library
 */

#include	"pios.hh"
#include	<string.h>

pstream::pstream(pstreambuf *abuf, int aflags)
{
  buf		= abuf;
  curid		= 1;
  flags	= aflags;
  streamed	= 0;
  err		= 0;
}

pstream::~pstream()
{
  close();
}

void	pstream::addstreamed(pstreamable *obj)
{
  streamed	= new pstreamed(curid++, streamed, pvoid(obj));
}

void	pstream::close()
{
  pstreamed *	p	= streamed;
  pstreamed *	q	= p;
  for(; p != 0; p = q)
    {
      q	= p->next;
      delete	p;
    }
  if (buf != 0)
    {
      if (flags & xxkill)
	{
	  delete	buf;
	  buf		= 0;
	}
      else if (flags & xxclose)
	{
	  buf->close();
	  error(buf->error());
	}
    }
  else
    error(peNoBuffer);
  curid		= 1;
  streamed	= 0;
  err		= 0;
}

void	pstream::error(int aerror)
{
  err	= aerror;
}

int	pstream::error()
{
  return	err;
}

int	pstream::lookup(pvoid q)
{
  pstreamed *	p	= streamed;
  for (; p != 0; p = p->next)
    if (p->p == q)
      break;
  if (p)
    return       	p->id;	// p = 0 if not found, otherwise
				// it points to the pstreamreg
  return	0;
}

pstreamed *	pstream::lookup(int aid)
{
  pstreamed *	p	= streamed;
  for (; p != 0; p = p->next)
    if (p->id == aid)
      break;
  return	p;
}

int	pstream::operator!()
{
  return	err;
}

opstream::opstream(pstreambuf *abuf, int aflags)
:	pstream(abuf, aflags)
{
}

void	opstream::writebyte(char c)
{
  writebytes((pvoid) &c, sizeof(c));
}

void	opstream::writebytes(pvoid d, int l)
{
  if (error())
    return;
  if (buf != 0)
    buf->write(d, l);
  else
    error(peNoBuffer);
  error(buf->error());
}

void	opstream::writeint(int i)
{
  writebytes(&i, sizeof(i));
}

void	opstream::writeobj(pstreamable * obj)
{
  int	index	= 0;
  if (obj == 0)
    writebyte(pstream::ptNULL);
  else if ((index = lookup(pvoid(obj))) != 0)
    {
      writebyte(pstream::ptIndex);
      writeint(index);
    }
  else
    {
      writebyte(pstream::ptObject);
				// Write the object Prefix
      writebyte('[');
      writestring(obj->streamableName());
				// Write the object data
      addstreamed(obj);
      obj->write(*this);
				// Write the object suffix
      writebyte(']');
    }
}

void	opstream::writestring(char *s)
{
  int	l	= strlen(s);
  writeint(l);
  writebytes(s, l);
}

opstream &	operator <<(opstream & os, char c)
{
  os.writebyte(c);
  return	os;
}

opstream &	operator <<(opstream & os, char * s)
{
  os.writestring(s);
  return	os;
}

opstream &	operator <<(opstream & os, short c)
{
  os.writebytes(&c, sizeof(c));
  return	os;
}

opstream &	operator <<(opstream & os, int c)
{
  os.writebytes(&c, sizeof(c));
  return	os;
}

opstream &	operator <<(opstream & os, long c)
{
  os.writebytes(&c, sizeof(c));
  return	os;
}

opstream &	operator <<(opstream & os, float c)
{
  os.writebytes(&c, sizeof(c));
  return	os;
}

opstream &	operator <<(opstream & os, double c)
{
  os.writebytes(&c, sizeof(c));
  return	os;
}

opstream &	operator <<(opstream & os, pstreamable & x)
{
  os.writeobj(&x);
  return	os;
}

opstream &	operator <<(opstream & os, pstreamable * x)
{
  os.writeobj(x);
  return	os;
}

ipstream::ipstream(pstreambuf *abuf, int aflags)
:	pstream(abuf, aflags)
{
}

char	ipstream::readbyte()
{
  char		c;
  readbytes(&c, sizeof(c));
  return	c;
}

pvoid	ipstream::readbytes(pvoid d, int l)
{
  if (error())
    return	0;
  pvoid	x	= 0;
  if (buf != 0)
    x	= buf->read(d, l);
  else
    error(peNoBuffer);
  return	x;
}

int	ipstream::readint()
{
  int		i;
  readbytes(&i, sizeof(i));
  return	i;
}

pvoid	ipstream::readobj(pstreamreg *d, pstreamable *m)
{
  if (error())
    return	0;
  if (m == 0)
    m	= d->builder();
  streamed	= new pstreamed(curid++, streamed, ((char *)m - d->delta));
  pvoid	q	= m->read(*this);
  readbyte();
  if (error())
    return	0;
  return	q;
}

char *	ipstream::readstring()
{
  int		l	= readint();
  char *	s	= new char[l+1];
  readbytes(s, l);
  s[l]		= 0;
  if (error())
    {
      delete	s;
      s		= 0;
    }
  return	s;
}

ipstream &	operator >>(ipstream & is, char &c)
{
  c	= is.readbyte();
  return	is;
}

ipstream &	operator >>(ipstream & is, char * &s)
{
  s	= is.readstring();
  return	is;
}

ipstream &	operator >>(ipstream & is, short &c)
{
  is.readbytes(&c, sizeof(c));
  return	is;
}

ipstream &	operator >>(ipstream & is, int &c)
{
  is.readbytes(&c, sizeof(c));
  return	is;
}

ipstream &	operator >>(ipstream & is, long &c)
{
  is.readbytes(&c, sizeof(c));
  return	is;
}

ipstream &	operator >>(ipstream & is, float &c)
{
  is.readbytes(&c, sizeof(c));
  return	is;
}

ipstream &	operator >>(ipstream & is, double &c)
{
  is.readbytes(&c, sizeof(c));
  return	is;
}

ipstream &	operator >>(ipstream & is, pstreamable & x)
{
  is.readbyte();
  is.readbyte();
  char *	c	= is.readstring();
  pstreamreg *	d	= pstreamreg::findclass(c);
  delete	c;
  is.readobj(d, &x);
  return	is;
}

ipstream &	operator >>(ipstream & is, pvoid &x)
{
  char		c	= is.readbyte();
  switch (c)
    {
    default:
      break;
    case pstream::ptNULL:
      x		= 0;
      return	is;
      break;
    case pstream::ptIndex:
      x		= pvoid(is.lookup(is.readint())->p);
      return	is;
      break;
    case pstream::ptObject:
      is.readbyte();
      char *		d	= is.readstring();
      pstreamreg *	r	= pstreamreg::findclass(d);
      delete	d;
      x	= is.readobj(r, 0);
      break;
    }
  return	is;
}

pstreamed::pstreamed(int aid, pstreamed * anext, pvoid ap)
{
  id	= aid;
  next	= anext;
  p	= ap;
}

pstreamable::pstreamable()
{
}

pstreamable::pstreamable(pstreamableInit)
{
}

pstreamreg *	pstreamreged	= 0;

pstreamreg::pstreamreg(char * aname, BUILDER abuilder, int adelta)
{
  name		= aname;
  builder	= abuilder;
  delta		= adelta;
  next		= pstreamreged;
  pstreamreged	= this;
}

pstreamreg *	pstreamreg::findclass(char * aname)
{
  pstreamreg *	p	= pstreamreged;
  for (; p != 0; p = p->next)
    if (strcmp(aname, p->name) == 0)
      break;
  return	p;		// p = 0 if not found, otherwise
				// it points to the pstreamreg
}

pstreambuf::pstreambuf(int aflags)
{
  flags	= aflags;
}

pstreambuf::~pstreambuf()
{
  if (flags & pstream::xxclose)
    close();
}

void	pstreambuf::close()
{
}

int	pstreambuf::error()
{
  return	err;
}

void	pstreambuf::error(int aerror)
{
  err	= aerror;
}

