//===============================================================
// vCanvasPaneDC - a basic canvas for drawing
//
// Copyright (C) 1995,1996  Bruce E. Wampler
//
// This file is part of the V C++ GUI Framework, and is covered
// under the terms of the GNU Library General Public License,
// Version 2. This library has NO WARRANTY. See the source file
// vapp.cxx for more complete information about license terms.
//===============================================================

extern "C"
{
#include <math.h>
}

#include <v/vcpdc.h>

#include <v/vapp.h>		// need access to the app
#include <v/vcanvas.h>		// our own canvas widget
#include <v/vicon.h>		// for icons

#define _XWindow XtWindow(_drawWidget)

//-----------------------------------------------------------------------
// Define some X bitmaps for patterned fill
//
#define ldiag_width 16
#define ldiag_height 16
static char ldiag_bits[] = {
   0x01, 0x01, 0x02, 0x02, 0x04, 0x04, 0x08, 0x08, 0x10, 0x10, 0x20, 0x20,
   0x40, 0x40, 0x80, 0x80, 0x01, 0x01, 0x02, 0x02, 0x04, 0x04, 0x08, 0x08,
   0x10, 0x10, 0x20, 0x20, 0x40, 0x40, 0x80, 0x80};
   static Pixmap ldiagPM = 0;

#define cdiag_width 16
#define cdiag_height 16
static char cdiag_bits[] = {
   0x81, 0x81, 0x42, 0x42, 0x24, 0x24, 0x18, 0x18, 0x18, 0x18, 0x24, 0x24,
   0x42, 0x42, 0x81, 0x81, 0x81, 0x81, 0x42, 0x42, 0x24, 0x24, 0x18, 0x18,
   0x18, 0x18, 0x24, 0x24, 0x42, 0x42, 0x81, 0x81};
   static Pixmap cdiagPM = 0;

#define cross_width 15
#define cross_height 15
static char cross_bits[] = {
   0x84, 0x10, 0x84, 0x10, 0xff, 0x7f, 0x84, 0x10, 0x84, 0x10, 0x84, 0x10,
   0x84, 0x10, 0xff, 0x7f, 0x84, 0x10, 0x84, 0x10, 0x84, 0x10, 0x84, 0x10,
   0xff, 0x7f, 0x84, 0x10, 0x84, 0x10};
   static Pixmap crossPM = 0;

#define rdiag_width 16
#define rdiag_height 16
  static char rdiag_bits[] = {
   0x80, 0x80, 0x40, 0x40, 0x20, 0x20, 0x10, 0x10, 0x08, 0x08, 0x04, 0x04,
   0x02, 0x02, 0x01, 0x01, 0x80, 0x80, 0x40, 0x40, 0x20, 0x20, 0x10, 0x10,
   0x08, 0x08, 0x04, 0x04, 0x02, 0x02, 0x01, 0x01};
   static Pixmap rdiagPM = 0;

#define horiz_width 15
#define horiz_height 15
static char horiz_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0xff, 0x7f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0xff, 0x7f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0x7f, 0x00, 0x00, 0x00, 0x00};
   static Pixmap horizPM = 0;

#define verti_width 15
#define verti_height 15
static char verti_bits[] = {
   0x84, 0x10, 0x84, 0x10, 0x84, 0x10, 0x84, 0x10, 0x84, 0x10, 0x84, 0x10,
   0x84, 0x10, 0x84, 0x10, 0x84, 0x10, 0x84, 0x10, 0x84, 0x10, 0x84, 0x10,
   0x84, 0x10, 0x84, 0x10, 0x84, 0x10};
   static Pixmap vertiPM = 0;

  vFont vCanvasPaneDC::_GCFont;		// the font actually in the GC

//================>>> vCanvasPaneDC::vCanvasPaneDC <<<========================
  vCanvasPaneDC::vCanvasPaneDC(vCanvasPane* parentPane)
  {
    XColor fg;
    XColor bg;

    SysDebug(Constructor,"vCanvasPaneDC::vCanvasPaneDC() constructor\n")

    _parentPane = parentPane;

    _drawWidget = _parentPane->DrawingWindow();	// drawing to this widget

    _GCFont =
    _font = theApp->GetDefaultFont();	// get the default font

    _GC = makeGC(_drawWidget);		// make a normal GC to use
    _XorGC = makeXorGC(_drawWidget);	// make a GC to use

    _pen.SetPenToPixel(_canvasFG);
    _brush.SetBrushToPixel(_canvasFG);

    SetPen(_pen);
    SetBrush(_brush);

    _XDisplay = theApp->display();
//    _XWindow = XtWindow(_drawWidget);
  }

//================>>> vCanvasPaneDC::~vCanvasPaneDC <<<========================
  vCanvasPaneDC::~vCanvasPaneDC()
  {
    XtReleaseGC(_drawWidget,_GC);	// Release the GCs we allocated
    XtReleaseGC(_drawWidget,_XorGC);

    SysDebug(Destructor,"vCanvasPaneDC::~vCanvasPaneDC() destructor\n")
  }

//=====================>>> vCanvasPaneDC::Clear <<<==========================
  void vCanvasPaneDC::Clear(void)
  {
    XClearArea(_XDisplay, _XWindow, 0, 0, 
	_parentPane->GetWidth(), _parentPane->GetHeight(), 0);
  }

//==================>>> vCanvasPaneDC::ClearRect <<<==========================
  void vCanvasPaneDC::ClearRect(int x, int y, int width, int height)
  {
    // Clear a rectangluar area starting at left,top of width and height

    if (height == 0 || width == 0)	// ignore 0 h/w
	return;

    int xx = Scale(x+_tx);
    int yy = Scale(y+_ty);
    int h = Scale(height);
    int w = Scale(width);

    if (w < 0)		// convert to something we like
      {
	w = -w;		// make a positive width
	xx = xx - w;	// translate x
      }
    if (h < 0)
      {
	h = -h;		// positive h
	yy = yy - h;	// translate x
      }

    XClearArea(_XDisplay, _XWindow, xx, yy, w, h, 0);
  }

#ifdef DRAWARC
//====================>>> vCanvasPaneDC::DrawArc <<<==========================
  void vCanvasPaneDC::DrawArc(int xx1, int yy1, int xx2, int yy2, 
    int xxc, int yyc)
  {
    double dx = xx1 - xxc;
    double dy = yy1 - yyc;
    double radius = sqrt(dx * dx + dy * dy);
    int r = (int) radius;

const double PIx2 = 6.283185;

    double radius1, radius2;

    if (xx1 == xx2 && yy1 == yy2)
      {
	radius1 = 0.0;
	radius2 = 360.0;
      }
    else if (radius == 0.0)
	radius1 = radius2 = 0.0;
    else
      {
	if (xx1 - xxc == 0)
	  {
	    if (yy1 - yyc < 0)
		radius1 = 90.0;
	    else
		radius1 = -90.0;
	  }
	else
	    radius1 = -atan2 ((double) (yy1 - yyc), (double) (xx1 - xxc)) * 360.0 / PIx2;

	if (xx2 - xxc == 0)
	  {
	    if (yy2 - yyc < 0)
		radius2 = 90.0;
	    else
		radius2 = -90.0;
	  }
	else
	    radius2 = -atan2 ((double) (yy2 - yyc), (double) (xx2 - xxc)) * 360.0 / PIx2;
      }
    radius1 *= 64.0;
    radius2 *= 64.0;
    int alpha1 = (int) radius1;
    int alpha2 = (int) (radius2 - radius1);
    while (alpha2 <= 0)
	alpha2 += 360 * 64;

    while (alpha2 > 360 * 64)
	alpha2 -= 360 * 64;

    if (_brush.brushStyle != vTransparent)
       {
	 SetGCtoBrush();
	 XFillArc(_XDisplay, _XWindow, _GC, 
	    xxc - r, yyc - r, 2 * r, 2 * r, alpha1, alpha2);
       }

    if (_pen.penStyle != vTransparent)
      {
	SetGCtoPen();
	XDrawArc(_XDisplay, _XWindow, _GC,
	    xxc - r, yyc - r, 2 * r, 2 * r, alpha1, alpha2);
      }
  }
#endif

//====================>>> vCanvasPaneDC::DrawEllipse <<<==========================
  void vCanvasPaneDC::DrawEllipse(int x, int y, int width, int height)
  {
    static const int angle = (360 * 64);
    
    if (height == 0 || width == 0)	// ignore 0 h/w
	return;

    int xx = Scale(x+_tx);
    int yy = Scale(y+_ty);
    int w = Scale(width);
    int h = Scale(height);

    if (w < 0)		// convert to something we like
      {
	w = -w;		// make a positive width
	xx = xx - w;	// translate x
      }
    if (h < 0)
      {
	h = -h;		// positive h
	yy = yy - h;	// translate x
      }

    if (_brush.brushStyle != vTransparent)
      {
	SetGCtoBrush();
	XFillArc(_XDisplay, _XWindow, _GC, xx, yy, w, h, 0, angle);
      }

    if (_pen.penStyle != vTransparent)
      {
	SetGCtoPen();
	XDrawArc(_XDisplay, _XWindow, _GC, xx, yy, w, h, 0, angle);
      }
  }

//====================>>> vCanvasPaneDC::DrawIcon <<<==========================
  void vCanvasPaneDC::DrawIcon(int x, int y, vIcon& icon)
  {

    int xx = Scale(x+_tx);
    int yy = Scale(y+_ty);

    Pixmap iconPM = icon.GetXPM(0);		// may already be mapped

    if (iconPM == 0)
      {
	DrawRectangle(x,y,icon.width,icon.height);
      }
    else if (icon.depth == 1)		// monochrome bitmap
      {
	XCopyPlane (_XDisplay, iconPM, _XWindow, _GC,
            0, 0, icon.width, icon.height, xx, yy, 1);
      }
    else
      {
	XCopyArea(_XDisplay, iconPM, _XWindow, _GC,
            0, 0, icon.width, icon.height, xx, yy);
      }
  }

//====================>>> vCanvasPaneDC::DrawLine <<<==========================
  void vCanvasPaneDC::DrawLine(int x, int y, int xend, int yend)
  {
    // Draw a line from x,y to xend,yend

    int xx = Scale(x+_tx);
    int yy = Scale(y+_ty);
    int xe = Scale(xend+_tx);
    int ye = Scale(yend+_ty);

    SetGCtoPen();

    if (xx < 0) xx = 0;
    if (yy < 0) yy = 0;
    if (xe < 0) xe = 0;
    if (ye < 0) ye = 0;		// X hates negative coords

    XDrawLine(_XDisplay, _XWindow, _GC, xx, yy, xe, ye);
  }

//==================>>> vCanvasPaneDC::DrawPoint <<<======================
  void vCanvasPaneDC::DrawPoint(int x, int y)
  {

    int xx = Scale(x+_tx);
    int yy = Scale(y+_ty);

    if (xx < 0) xx = 0;
    if (yy < 0) yy = 0;

    SetGCtoPen();

    XDrawPoint(_XDisplay, _XWindow, _GC, xx, yy);
  }

//==================>>> vCanvasPaneDC::DrawPolygon <<<======================
  void vCanvasPaneDC::DrawPolygon(int n, vPoint points[], int fillStyle)
  {
    // draw a complete polygon (starting point specified twice!)


    XPoint* xpoints;

    if (_hasScale || _tx != 0 || _ty != 0) // If we have to scale, then we need to copy
      {
	xpoints = new XPoint[n];	// allocate space for copy
	for (int i = 0; i < n; i++)
	  {
	    xpoints[i].x = ((points[i].x+_tx) * _Mult) / _Div;	// scale
	    xpoints[i].y = ((points[i].y+_ty) * _Mult) / _Div;
	  }
      }
    else
      {
	xpoints = (XPoint*)points;	// just use original values
      }


    if (_brush.brushStyle != vTransparent)
      {
	SetGCtoBrush();
	XSetFillRule (_XDisplay, _GC,	// how to fill the polygon
	    (fillStyle == vAlternate) ? EvenOddRule : WindingRule);
						// xpoints1
	XFillPolygon (_XDisplay, _XWindow, _GC, xpoints, n-1, Complex, 0);
	XSetFillRule (_XDisplay, _GC, EvenOddRule);	// default mode
      }

    if (_pen.penStyle != vTransparent)
      {
	SetGCtoPen();
					// xpoints2
	XDrawLines (_XDisplay, _XWindow, _GC, xpoints, n, 0);
      }

    if (_hasScale)
	delete[]xpoints;
}

//==================>>> vCanvasPaneDC::DrawRectangle <<<======================
  void vCanvasPaneDC::DrawRectangle(int x, int y, int width, int height)
  {

    if (height == 0 || width == 0)	// ignore 0 h/w
	return;

    int xx = Scale(x+_tx);
    int yy = Scale(y+_ty);
    int w = Scale(width);
    int h = Scale(height);

    if (w < 0)		// convert to something we like
      {
	w = -w;		// make a positive width
	xx = xx - w;	// translate x
      }
    if (h < 0)
      {
	h = -h;		// positive h
	yy = yy - h;	// translate x
      }

    if (_brush.brushStyle != vTransparent)
      {
	SetGCtoBrush();
	XFillRectangle(_XDisplay, _XWindow, _GC, xx, yy, w, h);
      }

    if (_pen.penStyle != vTransparent)
      {
	SetGCtoPen();
	XDrawRectangle(_XDisplay, _XWindow, _GC, xx, yy, w, h);
      }
  }

//================>>> vCanvasPaneDC::DrawRoundedRectangle <<<===================
  void vCanvasPaneDC::DrawRoundedRectangle(int x, int y,
	int width, int height, int radius)
  {
    if (height == 0 || width == 0)	// ignore 0 h/w
	return;

    int xx = Scale(x+_tx);
    int yy = Scale(y+_ty);
    int w = Scale(width);
    int h = Scale(height);
    int r;

    if (w < 0)		// convert to something we like
      {
	w = -w;		// make a positive width
	xx = xx - w;	// translate x
      }
    if (h < 0)
      {
	h = -h;		// positive h
	yy = yy - h;	// translate x
      }

    if (radius < 0)
      {
	// Negative radius means divide average of hight and width
	// by this
	r = ((w+h)/(-2 * radius));
      }
    else
	r = Scale(radius);

    int cw = r + r;

    if (_brush.brushStyle != vTransparent)
      {
	SetGCtoBrush();
	// Draw three rectangles: the vertical, and then the sides

	XFillRectangle(_XDisplay, _XWindow, _GC, // full height in middle
		xx+r, yy, w-cw, h);
	XFillRectangle(_XDisplay, _XWindow, _GC, // left vertical box
		xx, yy+r, r, h-cw);
	XFillRectangle(_XDisplay, _XWindow, _GC, // right vertical box
		xx+w-r, yy+r, r, h-cw);

	XFillArc(_XDisplay, _XWindow, _GC, // upper left
		xx,yy,cw,cw, 90*64, 90*64);
	XFillArc(_XDisplay, _XWindow, _GC, // upper right
		xx+w-cw, yy, cw, cw, 0, 90*64);
	XFillArc(_XDisplay, _XWindow, _GC, // bottom left
		xx, yy+h-cw, cw, cw, 180 * 64, 90 * 64);
	XFillArc(_XDisplay, _XWindow, _GC, // bottom right
		xx+w-cw, yy+h-cw,cw,cw, 270 * 64, 90 * 64);
      }

    if (_pen.penStyle != vTransparent)
      {
	SetGCtoPen();
	XDrawLine(_XDisplay, _XWindow, _GC, xx+r, yy, xx+w-r, yy);
	XDrawLine(_XDisplay, _XWindow, _GC, xx+r, yy+h, xx+w-r, yy+h);
	XDrawLine(_XDisplay, _XWindow, _GC, xx, yy+r, xx, yy+h-r);
	XDrawLine(_XDisplay, _XWindow, _GC, xx+w, yy+r, xx+w, yy+h-r);
	XDrawArc(_XDisplay, _XWindow, _GC, // upper left
		xx,yy,cw,cw, 90*64, 90*64);
	XDrawArc(_XDisplay, _XWindow, _GC, // upper right
		xx+w-cw, yy, cw, cw, 0, 90*64);
	XDrawArc(_XDisplay, _XWindow, _GC, // bottom left
		xx, yy+h-cw, cw, cw, 180 * 64, 90 * 64);
	XDrawArc(_XDisplay, _XWindow, _GC, // bottom right
		xx+w-cw, yy+h-cw,cw,cw, 270 * 64, 90 * 64);
      }
  }

//==================>>> vCanvasPaneDC::DrawRubberEllipse <<<===================
  void vCanvasPaneDC::DrawRubberEllipse(int x, int y, int width, int height)
  {
    if (height == 0 || width == 0)	// ignore 0 h/w
	return;

    int xx = Scale(x+_tx);
    int yy = Scale(y+_ty);
    int w = Scale(width);
    int h = Scale(height);

    if (w < 0)		// convert to something we like
      {
	w = -w;		// make a positive width
	xx = xx - w;	// translate x
      }
    if (h < 0)
      {
	h = -h;		// positive h
	yy = yy - h;	// translate x
      }

    static const int angle = (360 * 64);

    SetPenStyle(_XorGC);
    XDrawArc(_XDisplay, _XWindow, _XorGC, xx, yy, w, h, 0, angle);
  }

//==================>>> vCanvasPaneDC::DrawRubberLine <<<======================
  void vCanvasPaneDC::DrawRubberLine(int x, int y, int xend, int yend)
  {
    // Draw a rubber-band line from x,y to xend,yend. Redrawing
    // over the same with will erase it.

    int xx = Scale(x+_tx);
    int yy = Scale(y+_ty);
    int xe = Scale(xend+_tx);
    int ye = Scale(yend+_ty);
    if (xx < 0) xx = 0;
    if (yy < 0) yy = 0;
    if (xe < 0) xe = 0;
    if (ye < 0) ye = 0;		// X hates negative coords

    SetPenStyle(_XorGC);
    XDrawLine(_XDisplay, _XWindow, _XorGC, xx, yy, xe, ye);
  }

//==================>>> vCanvasPaneDC::DrawRubberPoint <<<======================
  void vCanvasPaneDC::DrawRubberPoint(int x, int y)
  {
    int xx = Scale(x+_tx);
    int yy = Scale(y+_ty);

    if (xx < 0) xx = 0;
    if (yy < 0) yy = 0;

    SetPenStyle(_XorGC);
    XDrawPoint(_XDisplay, _XWindow, _GC, xx, yy);
  }

//==================>>> vCanvasPaneDC::DrawRubberRectangle <<<==================
  void vCanvasPaneDC::DrawRubberRectangle(int x, int y, int width, int height)
  {
    if (height == 0 || width == 0)	// ignore 0 h/w
	return;

    int xx = Scale(x+_tx);
    int yy = Scale(y+_ty);
    int w = Scale(width);
    int h = Scale(height);

    if (w < 0)		// convert to something we like
      {
	w = -w;		// make a positive width
	xx = xx - w;	// translate x
      }
    if (h < 0)
      {
	h = -h;		// positive h
	yy = yy - h;	// translate x
      }

    SetPenStyle(_XorGC);
    XDrawRectangle(_XDisplay, _XWindow, _XorGC, xx, yy, w, h);
  }

//=====================>>> vCanvasPaneDC::DrawAttrText <<<==========================
  void vCanvasPaneDC::DrawAttrText(int x, int y, char* text, const ChrAttr attr)
  {
    // Draw text with attributes at given x, y.

    // The fonts may not be equal because we are using a shared GC,
    // and another window might change the font in the GC.  We
    // need to check the current font and change to our font if
    // it is not the same

    static int mapColor[] =
      {
	vC_Black, vC_Blue, vC_Green, vC_Cyan,
        vC_Red, vC_Magenta, vC_Yellow, vC_White,
        vC_DarkGray, vC_DimBlue, vC_DimGreen, vC_DimCyan,
	vC_DimRed, vC_DimMagenta, vC_DimYellow, vC_MedGray
      };
    int xx = Scale(x+_tx);
    int yy = Scale(y+_ty);

    if (_GCFont.GetXFont() != _font.GetXFont())
      {
	// Change the font in the server
	XSetFont( theApp->display(), _GC, 
	    _font.GetXFont()->fid);
	_GCFont = _font;	// update current font
      }

    Pixel attrColor = _pen.penColor.pixel();	// Default
    if (attr & 0xF0)		// a color attribute
	attrColor = vStdColors[ mapColor[((attr & 0xF0) >> 4)] ].pixel();
//	attrColor = vStdColors[vC_Blue].pixel();

    if (attr & ChReverse)
      {
	XSetForeground(_XDisplay, _GC, _canvasBG);
	XSetBackground(_XDisplay, _GC, attrColor);
      }
    else
      {
	XSetForeground(_XDisplay, _GC, attrColor);
	XSetBackground(_XDisplay, _GC, _canvasBG);
      }

    int len = strlen(text);
    XDrawImageString(_XDisplay, _XWindow,
	_GC, xx, yy, text, len);

    // X can't do underlined, so we have to do it ourselves
    if (_font.GetUnderlined())
      {
	int width = XTextWidth(_font.GetXFont(),text,len);
	XDrawLine(_XDisplay, _XWindow, _GC, xx, yy+1, xx+width, yy+1);
      }

    if (attr)		// & ChReverse)
      {
	XSetForeground(_XDisplay, _GC, _pen.penColor.pixel());
	XSetBackground(_XDisplay, _GC, _canvasBG);
      }

  }

//=====================>>> vCanvasPaneDC::DrawText <<<==========================
  void vCanvasPaneDC::DrawText(int x, int y, char* text)
  {
    // simple draw text at given x, y

    // The fonts may not be equal because we are using a shared GC,
    // and another window might change the font in the GC.  We
    // need to check the current font and change to our font if
    // it is not the same

    int xx = Scale(x+_tx);
    int yy = Scale(y+_ty);

    if (_GCFont.GetXFont() != _font.GetXFont())
      {
	// Change the font in the server
	XSetFont( theApp->display(), _GC, 
	    _font.GetXFont()->fid);
	_GCFont = _font;	// update current font
      }

    XSetForeground(_XDisplay, _GC, _pen.penColor.pixel());
    XSetBackground(_XDisplay, _GC, _canvasBG);

    int len = strlen(text);
    XDrawImageString(_XDisplay, _XWindow,
	_GC, xx, yy, text, len);

    // X can't do underlined, so we have to do it ourselves
    if (_font.GetUnderlined())
      {
	int width = XTextWidth(_font.GetXFont(),text,len);
	XDrawLine(_XDisplay, _XWindow, _GC, xx, yy+1, xx+width, yy+1);
      }

  }

//=====================>>> vCanvasPaneDC::makeGC <<<=========================
  GC vCanvasPaneDC::makeGC(Widget w)
  {
    // This one gets a default GC to used for the base canvas

    GC gc;		// local copy
    XGCValues values;	// to setup values
    Pixel fg, bg;

    int n = 0;

    // set colors according to type of canvas

    _canvasFG = theApp->Xfg();
    _canvasBG = theApp->Xbg();

    XtVaSetValues(w,			// reset fg/bg
	XtNforeground, _canvasFG,
	XtNbackground, _canvasBG,
	NULL);

    values.line_style = LineSolid;	// only solid lines for now
    values.line_width = 1;		// width 1
    values.fill_style = FillSolid;
    values.function = GXcopy;
    values.foreground = _canvasFG;
    values.background = _canvasBG;

    // This is a SHARED GC, so anything that changes must be
    // reset on a per window basis before drawing anything.

    gc = XtAllocateGC(w, 0,
	GCForeground | GCBackground | GCFunction | GCLineStyle | 
	  GCLineWidth | GCFillStyle ,
	&values,
	GCForeground | GCBackground | GCFunction | GCLineStyle | // changable
	  GCLineWidth | GCFillStyle | GCFont,
	0);

    // set the font

    // Be sure the font is loaded
    _font.LoadFont();

    XSetFont(theApp->display(), gc, _font.GetXFont()->fid);

    return gc;
  }

//=====================>>> vCanvasPaneDC::makeXorGC <<<==========================
  GC vCanvasPaneDC::makeXorGC(Widget w)
  {
    // This one gets a default GC to used for the base canvas

    GC gc;		// local copy
    XGCValues values;	// to setup values

    int n = 0;

    // set colors according to type of canvas

    values.line_width = 1;		// width 1
    values.function = GXxor;

    if (_canvasFG == 0)
	values.foreground = ~0;		// force black
    else
	values.foreground = _canvasFG;

    gc = XtAllocateGC(w, 0,
	GCForeground | GCFunction | GCLineWidth,
	&values,
	GCLineWidth,			// Width changes
	0);

    return gc;
  }

//================>>> vCanvasPaneDC::SetBackground <<<==========================
  void vCanvasPaneDC::SetBackground(vColor& color)
  {
    if (!color.IsMapped())	// Map the color?
	color.MapColor();
    
    _canvasBG = color.pixel();		// retrieve X pixel value
    XtVaSetValues(_drawWidget,			// reset fg/bg
	XtNbackground, _canvasBG,
	NULL);
  }

//=====================>>> vCanvasPaneDC::SetBrush <<<============================
 void vCanvasPaneDC::SetBrush(vBrush& brush)
  {
    _brush = brush;
    if (!_brush.brushColor.IsMapped())	// Map the color?
	_brush.brushColor.MapColor();
  }

//=====================>>> vCanvasPaneDC::SetGCtoBrush <<<========================
  void vCanvasPaneDC::SetGCtoBrush()
  {

    Pixmap my_stipple = 0;

    switch (_brush.brushStyle)
      {
	case vSolid:
	    XSetFillStyle(_XDisplay,_GC,FillSolid);
	    break;

	case vTransparent:
	    XSetFillStyle(_XDisplay,_GC,FillSolid);
	    break;

	case vHorizontalHatch:
	  {
	    if (!horizPM)
		horizPM = XCreateBitmapFromData (_XDisplay,
			      RootWindow (_XDisplay, DefaultScreen (_XDisplay)),
				     horiz_bits, horiz_width, horiz_height);
	    XSetFillStyle(_XDisplay,_GC,FillStippled);	// we will set pixmaps
	    XSetStipple (_XDisplay, _GC, horizPM);
	    break;
	  }

	case vVerticleHatch:
	  {
	    if (!vertiPM)
		vertiPM = XCreateBitmapFromData (_XDisplay,
			      RootWindow (_XDisplay, DefaultScreen (_XDisplay)),
				     verti_bits, verti_width, verti_height);
	    XSetFillStyle(_XDisplay,_GC,FillStippled);	// we will set pixmaps
	    XSetStipple (_XDisplay, _GC, vertiPM);
	    break;
	  }

	case vLeftDiagonalHatch:
	  {
	    if (!ldiagPM)
		ldiagPM = XCreateBitmapFromData (_XDisplay,
			      RootWindow (_XDisplay, DefaultScreen (_XDisplay)),
				     ldiag_bits, ldiag_width, ldiag_height);
	    XSetFillStyle(_XDisplay,_GC,FillStippled);	// we will set pixmaps
	    XSetStipple (_XDisplay, _GC, ldiagPM);
	    break;
	  }

	case vRightDiagonalHatch:
	  {
	    if (!rdiagPM)
		rdiagPM = XCreateBitmapFromData (_XDisplay,
			      RootWindow (_XDisplay, DefaultScreen (_XDisplay)),
				     rdiag_bits, rdiag_width, rdiag_height);
	    XSetFillStyle(_XDisplay,_GC,FillStippled);	// we will set pixmaps
	    XSetStipple (_XDisplay, _GC, rdiagPM);
	    break;
	  }

	case vCrossHatch:
	  {
	    if (!crossPM)
		crossPM = XCreateBitmapFromData (_XDisplay,
			      RootWindow (_XDisplay, DefaultScreen (_XDisplay)),
				     cross_bits, cross_width, cross_height);
	    XSetFillStyle(_XDisplay,_GC,FillStippled);	// we will set pixmaps
	    XSetStipple (_XDisplay, _GC, crossPM);
	    break;
	  }

	case vDiagonalCrossHatch:
	  {
	    if (!cdiagPM)
		cdiagPM = XCreateBitmapFromData (_XDisplay,
			      RootWindow (_XDisplay, DefaultScreen (_XDisplay)),
				     cdiag_bits, cdiag_width, cdiag_height);
	    XSetFillStyle(_XDisplay,_GC,FillStippled);	// we will set pixmaps
	    XSetStipple (_XDisplay, _GC, cdiagPM);
	    break;
	  }

	default:
	    XSetFillStyle(_XDisplay,_GC,FillSolid);
	    break;
      }

    XSetForeground(_XDisplay, _GC, _brush.brushColor.pixel());

  }


//=====================>>> vCanvasPaneDC::SetPenStyle <<<========================
  void vCanvasPaneDC::SetPenStyle(GC gc)
  {
    static char dot[] = {2, 5};
    static char dash[] = {4, 4};
    static char dashdot[] = {6, 6, 2, 6};

    int ps;
    switch (_pen.penStyle)		// find a suitable X line style
      {
	case vSolid:
	case vTransparent:
	    ps = LineSolid;
	    break;

	case vDash:
	    ps = LineOnOffDash;
	    XSetDashes(_XDisplay, gc, 0, dash, 2);
	    break;

	case vDot:
	    ps = LineOnOffDash;
	    XSetDashes(_XDisplay, gc, 0, dot, 2);
	    break;

	case vDashDot:
	    ps = LineOnOffDash;
	    XSetDashes(_XDisplay, gc, 0, dashdot, 4);
	    break;

	default:
	    ps = LineSolid;
	    break;
      }

    XSetLineAttributes(_XDisplay, gc, _pen.penWidth, ps, CapButt, JoinMiter);
  }

//=====================>>> vCanvasPaneDC::SetGCtoPen <<<========================
  void vCanvasPaneDC::SetGCtoPen()
  {

    XSetForeground(_XDisplay, _GC, _pen.penColor.pixel());
    XSetBackground(_XDisplay, _GC, _canvasBG);

    SetPenStyle(_GC);

    XSetFillStyle(_XDisplay,_GC,FillSolid);

  }

//=====================>>> vCanvasPaneDC::SetPen <<<============================
 void vCanvasPaneDC::SetPen(vPen& pen)
  {
    _pen = pen;
    if (!_pen.penColor.IsMapped())	// Map the color?
	_pen.penColor.MapColor();
  }

//======================>>> vCanvasPaneDC::SetFont <<<===========================
  void vCanvasPaneDC::SetFont(vFont& vf)
  {
    // Change the font associated with this window.

    _font = vf;

    // First, make sure the font is loaded

    _GCFont = _font;		// we have changed the font in
					// the shared GC
    _font.LoadFont();

    XSetFont( theApp->display(), _GC, 
	_font.GetXFont()->fid);

    _parentPane->FontChanged(_font);		// We have changed the font,
					// so we may need to do some things
  }

//====================>>> vCanvasPaneDC::TextHeight <<<=============================
  int vCanvasPaneDC::TextHeight(int& asc, int& des)
  {
    // Return total height of this font. V will use total height, which
    // is most often made up of ascent + descent.  This is too much
    // detail for the kind of apps V will support.

    return _font.XFontH(asc, des);
  }

//========================>>> vCanvasPaneDC::TextWidth <<<==========================
  int vCanvasPaneDC::TextWidth(char* str)
  {
    return _font.XTextW(str);
  }
