/*
 * addon.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "addon.h"
#include "button.h"

static char daysinmonth[] =
{
	31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
};

static char monthstr[][10] = 
{
	"January  ", "February ", "March    ", "April    ",
	"May      ", "June     ", "July     ", "August   ",
	"September", "October  ", "November ", "December "
};

static unsigned char part[][25] =
{
	{	/* 0 */
		0xc9, 0xcd, 0xcd, 0xcd, 0xbb, 0xba, 0x20, 0x20, 0x20, 0xba,
		0xba, 0x20, 0x20, 0x20, 0xba, 0xba, 0x20, 0x20, 0x20, 0xba,
		0xc8, 0xcd, 0xcd, 0xcd, 0xbc
	},
	{	/* 1 */
		0x20, 0x20, 0xd2, 0x20, 0x20, 0x20, 0x20, 0xba, 0x20, 0x20,
		0x20, 0x20, 0xba, 0x20, 0x20, 0x20, 0x20, 0xba, 0x20, 0x20,
		0x20, 0x20, 0xd0, 0x20, 0x20
	},
	{	/* 2 */
		0xcd, 0xcd, 0xcd, 0xcd, 0xbb, 0x20, 0x20, 0x20, 0x20, 0xba,
		0xc9, 0xcd, 0xcd, 0xcd, 0xbc, 0xba, 0x20, 0x20, 0x20, 0x20,
		0xc8, 0xcd, 0xcd, 0xcd, 0xcd
	},
	{	/* 3 */
		0xcd, 0xcd, 0xcd, 0xcd, 0xbb, 0x20, 0x20, 0x20, 0x20, 0xba,
		0x20, 0x20, 0xcd, 0xcd, 0xb9, 0x20, 0x20, 0x20, 0x20, 0xba,
		0xcd, 0xcd, 0xcd, 0xcd, 0xbc
	},
	{	/* 4 */
		0xd2, 0x20, 0x20, 0x20, 0xd2, 0xba, 0x20, 0x20, 0x20, 0xba,
		0xc8, 0xcd, 0xcd, 0xcd, 0xb9, 0x20, 0x20, 0x20, 0x20, 0xba,
		0x20, 0x20, 0x20, 0x20, 0xd0
	},
	{	/* 5 */
		0xc9, 0xcd, 0xcd, 0xcd, 0xcd, 0xba, 0x20, 0x20, 0x20, 0x20,
		0xc8, 0xcd, 0xcd, 0xcd, 0xbb, 0x20, 0x20, 0x20, 0x20, 0xba,
		0xcd, 0xcd, 0xcd, 0xcd, 0xbc
	},
	{	/* 6 */
		0xc9, 0xcd, 0xcd, 0xcd, 0xcd, 0xba, 0x20, 0x20, 0x20, 0x20,
		0xcc, 0xcd, 0xcd, 0xcd, 0xbb, 0xba, 0x20, 0x20, 0x20, 0xba,
		0xc8, 0xcd, 0xcd, 0xcd, 0xbc
	},
	{	/* 7 */
		0xcd, 0xcd, 0xcd, 0xcd, 0xbb, 0x20, 0x20, 0x20, 0x20, 0xba,
		0x20, 0x20, 0x20, 0x20, 0xba, 0x20, 0x20, 0x20, 0x20, 0xba,
		0x20, 0x20, 0x20, 0x20, 0xd0
	},
	{	/* 8 */
		0xc9, 0xcd, 0xcd, 0xcd, 0xbb, 0xba, 0x20, 0x20, 0x20, 0xba,
		0xcc, 0xcd, 0xcd, 0xcd, 0xb9, 0xba, 0x20, 0x20, 0x20, 0xba,
		0xc8, 0xcd, 0xcd, 0xcd, 0xbc
	},
	{	/* 9 */
		0xc9, 0xcd, 0xcd, 0xcd, 0xbb, 0xba, 0x20, 0x20, 0x20, 0xba,
		0xc8, 0xcd, 0xcd, 0xcd, 0xb9, 0x20, 0x20, 0x20, 0x20, 0xba,
		0xcd, 0xcd, 0xcd, 0xcd, 0xbc
	}
};

/*
 * Calendar constructor.
 */

Calendar::Calendar(int x, int y): Window(Rect(x, y, x + 26, y + 12),
	"Calendar", Window::initFrame)
{
	DEBUG("new Calendar at %p\n", this);
	flags = WF_CLOSE | WF_MOVE;
	insert(calview = new CalendarView(3, 1));
	insert(new Button(Rect(3, 9, 12, 11), "~P~rev", CM_BACKWARD));
	insert(new Button(Rect(14, 9, 23, 11), "~N~ext", CM_FORWARD));
}

/*
 * Handles the Calendar's events.
 */

void Calendar::handleEvent(Event &event)
{
	Window::handleEvent(event);
	switch (event.what)
	{
	case EV_COMMAND:
		switch (event.command)
		{
		case CM_BACKWARD:
			calview->moveDelta(-1);
			clearEvent(event);
			break;
		case CM_FORWARD:
			calview->moveDelta(1);
			clearEvent(event);
		}
		break;
	case EV_KEYDOWN:
		switch (event.keycode)
		{
		case KC_ARROWDOWN:
		case KC_ARROWLEFT:
			calview->moveDelta(-1);
			clearEvent(event);
			break;
		case KC_ARROWRIGHT:
		case KC_ARROWUP:
			calview->moveDelta(1);
			clearEvent(event);
		}
	}
}

/*
 * CalendarView constructor.
 */

CalendarView::CalendarView(int x, int y): View(Rect(x, y, x + 21, y + 8))
{
	DEBUG("new CalendarView at %p\n", this);
	time_t now = time(NULL);
	tm *broken = localtime(&now);

	month = broken->tm_mon + 1;
	year = broken->tm_year + 1900;
}

/*
 * Returns the day of the week.
 */

int CalendarView::dayOfWeek(int day, int month, int year)
{
	int century, dw, yr;

	if (month < 3)
	{
		month += 10;
		year--;
	}
	else month -= 2;
	century = year / 100;
	yr = year % 100;
	dw = (((26 * month - 2) / 10) + day + yr + (yr / 4) + (century / 4) -
		(2 * century)) % 7;
	if (dw < 0) dw += 7;
	return dw;
}

/*
 * Draws the CalendarView on the screen.
 */

void CalendarView::draw()
{
	DrawBuf buf;
	Video *where = buf + 2 * size.x;
	char str[MAX_WIDTH];
	int col = getColor(C_CALENDARVIEW);
	int day, dayof, days, i, j;

	dayof = dayOfWeek(1, month, year);
	days = daysinmonth[month - 1] + ((year % 4 == 0) && (month == 2));
	day = 1 - dayof;
	sprintf(str, "%s   %4d     Su Mo Tu We Th Fr Sa ",
		monthstr[month - 1], year);
	moveStr(buf, str, col);
	for (i = 0; i < 7; i++)
	{
		for (j = 0; j < 7; j++)
		{
			if (day < 1 || day > days)
			{
				moveStr(where + 3 * j, "   ", col);
			}
			else
			{
				sprintf(str, "%2d ", day);
				moveStr(where + 3 * j, str, col);
			}
			day++;
		}
		where += size.x;
	}
	writeBuf(0, 0, size.x, size.y, buf);
}

/*
 * Moves backward or forward the calendar.
 */

void CalendarView::moveDelta(int delta)
{
	if (delta < 0 && --month < 1)
	{
		month =	12;
		year--;
	}
	else if (delta > 0 && ++month > 12)
	{
		month = 1;
		year++;
	}
	drawView();
}

/*
 * Clock constructor.
 */

Clock::Clock(int x, int y): Window(Rect(x, y, x + 43, y + 7), "Clock",
	Window::initFrame)
{
	DEBUG("new Clock at %p\n", this);
	flags = WF_CLOSE | WF_MOVE;
	insert(new ClockView(2, 1));
}

/*
 * ClockView constructor.
 */

ClockView::ClockView(int x, int y): View(Rect(x, y, x + 39, y + 5))
{
	time_t now = time(NULL);
	tm *broken = localtime(&now);

	hour = broken->tm_hour;
	min = broken->tm_min;
	sec = broken->tm_sec;
}

/*
 * Draws the ClockView on the screen.
 */

void ClockView::draw()
{
	DrawBuf buf;
	int col = getColor(C_CLOCKVIEW);

	moveStr(buf, 
		"                                       "
		"            *             *            "
		"                                       "
		"            *             *            "
		"                                       ", col);
	writeSegment(hour / 10, buf, col);
	writeSegment(hour % 10, buf + 6, col);
	writeSegment(min / 10, buf + 14, col);
	writeSegment(min % 10, buf + 20, col);
	writeSegment(sec / 10, buf + 28, col);
	writeSegment(sec % 10, buf + 34, col);
	writeBuf(0, 0, size.x, size.y, buf);
}

/*
 * Handles the ClockView's events.
 */

void ClockView::handleEvent(Event &event)
{
	View::handleEvent(event);
	switch (event.what)
	{
	case EV_BROADCAST:
		switch (event.command)
		{
		case CM_WAKEUP:
			hour = event.hour;
			min = event.min;
			sec = event.sec;
			drawView();
		}
	}
}

/*
 * Writes one digit in the buffer.
 */

void ClockView::writeSegment(int value, Video *where, int col)
{
	int i;
	unsigned char *from = part[value];

	for (i = 0; i < 7; i++)
	{
		moveBuf(where, from, col, 5);
		from += 5;
		where += size.x;
	}
}
