/*
 * button.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <unistd.h>
#include "button.h"

/*
 * Button constructor.
 */

Button::Button(Rect bounds, char *aname, int acommand, int aflags):
	View(bounds)
{
	DEBUG("new Button at %p\n", this);
	options |= OF_FIRSTCLICK | OF_POSTPROCESS | OF_SELECTABLE;
	command = acommand;
	flags = aflags;
	name = aname;
}

/*
 * Draws the Button on the screen.
 */

void Button::draw()
{
	drawState(0);
}

/*
 * Draws all possible states of the Button.
 */

void Button::drawState(int pressed)
{
	DrawBuf buf;
	Video *where = buf + (size.x - cStrLen(name)) / 2;
	int col_back = getColor(C_BUTTON);
	int col_name;

	if (state & SF_ACTIVE)
	{
		if (state & SF_FOCUSED) col_name = getColor(C_BUTTON + 3);
		else
		{
			col_name = flags & BF_DEFAULT ?
				getColor(C_BUTTON + 1) :
				getColor(C_BUTTON + 4);
		}
	}
	else col_name = getColor(C_BUTTON + 2);
	moveChar(buf, ' ', col_name, size.x);
	if (pressed)
	{
		moveStr(buf, " ", col_back);
		moveCStr(where + 1, name, col_name);
		moveChar(buf + size.x, ' ', col_back, size.x);
	}
	else
	{
		moveCStr(where, name, col_name);
		moveStr(buf + size.x - 1, "\xdc", col_back);
		moveStr(buf + size.x, " ", col_back);
		moveChar(buf + size.x + 1, 0xdf, col_back, size.x - 1);
	}
	writeBuf(0, 0, size.x, size.y, buf);
}

/*
 * Handles the Button's events.
 */

void Button::handleEvent(Event &event)
{
	int pressed = 0;

	View::handleEvent(event);
	switch (event.what)
	{
	case EV_BROADCAST:
		switch (event.command)
		{
		case CM_DEFAULT:
			pressed = (flags & BF_DEFAULT) != 0;
			break;
		case CM_COMMANDSETCHANGED:
			setState(SF_ACTIVE, commandEnabled(command));
		}
		break;
	case EV_KEYDOWN:
		switch (owner->getPhase())
		{
		case PH_FOCUSED:
			switch (event.keycode)
			{
			case ' ':
			case KC_ENTER:
				pressed = 1;
				break;
			case KC_ARROWLEFT:
			case KC_ARROWRIGHT:
				event.keycode = event.keycode ==
					KC_ARROWLEFT ? KC_METATAB : KC_TAB;
				event.what = EV_KEYDOWN;
				putEvent(event);
				clearEvent(event);
				break;
			default:
				pressed = toupper2(event.keycode) ==
					hotKey(name);
			}
			break;
		case PH_POSTPROCESS:
			pressed = event.keycode == MK_META(hotKey(name));
		}
		break;
	case EV_MOUSEDOWN:
		do
		{
			Point pnt;

			makeLocal(event.where, pnt);
			switch (event.what)
			{
			case EV_MOUSEDOWN:
				if (pnt.y == 0)
				{
					pressed = 1;
					drawState(pressed);
				}
				break;
			case EV_MOUSEMOVE:
				if (pressed && (!localPointInside(pnt) ||
					pnt.y != 0))
				{
					pressed = 0;
					drawState(pressed);
				}
			}
			getEvent(event);
 		}
		while (event.what != EV_MOUSEUP);
		drawState(0);
		clearEvent(event);
	}
	if (pressed)
	{
		clearEvent(event);
		if (state & SF_ACTIVE)
		{
			if (flags & BF_BROADCAST)
			{
				sendMessage(owner, EV_BROADCAST, command);
			}
			else
			{
				Event make;

				make.what = EV_COMMAND;
				make.command = command;
				putEvent(make);
			}
		}
	}
}
