/*
 * dialogs.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <dirent.h>
#include <stdio.h>
#include <sys/stat.h>
#include <unistd.h>
#include "button.h"
#include "dialogs.h"
#include "inpline.h"
#include "label.h"
#include "listbox.h"
#include "scrbar.h"
#include "stattxt.h"

/*
 * FileDialog constructor.
 */

FileDialog::FileDialog(char *title): Dialog(Rect(0, 0, 48, 21), title)
{
	DEBUG("new FileDialog at %p\n", this);
	ScrollBar *hs, *vs;

	insert(dir = new StaticText(Rect(2, 2, 46, 3), ""));
	insert(name = new InputLine(Rect(2, 5, 23, 6), ""));
	insert(new Label(Rect(2, 4, 6, 5), "~N~ame", name));
	insert(new Button(Rect(25, 5, 34, 7), "O~K~", CM_OK, BF_DEFAULT));
	insert(new Button(Rect(37, 5, 46, 7), "~C~ancel", CM_CANCEL));
	insert(hs = new ScrollBar(Rect(2, 18, 22, 19)));
	insert(vs = new ScrollBar(Rect(22, 8, 23, 18)));
	insert(dirs = new ListBox(Rect(2, 8, 22, 18), hs, vs, new StrList));
	insert(new Label(Rect(2, 7, 11, 8), "~D~irectory", dirs));
	insert(hs = new ScrollBar(Rect(25, 18, 45, 19)));
	insert(vs = new ScrollBar(Rect(45, 8, 46, 18)));
	insert(names = new ListBox(Rect(25, 8, 45, 18), hs, vs, new StrList));
	insert(new Label(Rect(25, 7, 29, 8), "~F~ile", names));
	focusNext(1);
	readDirectory("");
}

/*
 * Returns the size of the FileDialog's data.
 */

int FileDialog::dataSize()
{
	return sizeof(FileDialogData);
}

/*
 * Gets the FileDialog's data.
 */

void FileDialog::getData(void *buf)
{
	FileDialogData *fdd = (FileDialogData *) buf;

	if (name->getString()[0] == '\0')
	{
		/* no name ? */

		fdd->file[0] = '\0';
	}
	else if (strcmp(dir->getString(), "/") == 0)
	{
		/* it is the root directory ? */

		sprintf(fdd->file, "/%s", name->getString());
	}
	else sprintf(fdd->file, "%s/%s", dir->getString(), name->getString());
}

/*
 * Handles the FileDialog's events.
 */

void FileDialog::handleEvent(Event &event)
{
	/* grab enter or space keys if they happen in the directory ListBox */

	if (event.what == EV_KEYDOWN && (event.keycode == KC_ENTER ||
		event.keycode == ' ') && current == dirs)
	{
		readDirectory(dirs->getString());
		clearEvent(event);
	}
	Dialog::handleEvent(event);
	switch (event.what)
	{
	case EV_BROADCAST:
		switch (event.command)
		{
		case CM_LISTVIEWERCHANGED:
			if (event.sender == names) name->setString(
				names->getString());
			break;
		case CM_LISTVIEWERDOUBLE:
			if (event.sender == dirs) readDirectory(
				dirs->getString());
			else if (event.sender == names) endModal(CM_OK);
		}
	}
}

/*
 * Reads the directory from the fs.
 */

void FileDialog::readDirectory(char *move)
{
	DIR *dfd;
	char old[PATH_MAX];

	getcwd(old, PATH_MAX);
	chdir(dir->getString());
	chdir(move);
	if ((dfd = opendir(".")) != NULL)
	{
		char newdir[PATH_MAX];
		dirent *dp;

		getcwd(newdir, PATH_MAX);
		dir->setString(newdir);
		dirs->clearList();
		names->clearList();
		while ((dp = readdir(dfd)) != NULL)
		{
			struct stat stbuf;

			if (strcmp(dp->d_name, ".") != 0 && stat(dp->d_name,
				&stbuf) != -1)
			{
				switch (stbuf.st_mode & S_IFMT)
				{
				case S_IFDIR:
					dirs->getList()->addAlphabetical(
						dp->d_name);
					break;
				case S_IFREG:
					names->getList()->addAlphabetical(
						dp->d_name);
				}
			}
		}
		dirs->updateScreen();
		names->updateScreen();
	}
	chdir(old);
}

/*
 * Sets the FileDialog's data.
 */

void FileDialog::setData(void *buf)
{
	FileDialogData *ffd = (FileDialogData *) buf;
	char curdir[PATH_MAX];
	char curname[PATH_MAX];

	expandPath(ffd->file, curdir, curname);
	dir->setString(curdir);
	name->setString(curname);
	readDirectory("");
}
