/*
 * listview.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <string.h>
#include "listview.h"
#include "scrbar.h"

/*
 * ListViewer constructor.
 */

ListViewer::ListViewer(Rect bounds, ScrollBar *ahs, ScrollBar *avs):
	View(bounds)
{
	DEBUG("new ListViewer at %p\n", this);
	options = OF_FIRSTCLICK | OF_SELECTABLE;
	data.current = -1;
	left = home = 0;
	if ((hs = ahs) != NULL) hs->setStep(size.x, 1);
	if ((vs = avs) != NULL) vs->setStep(size.y, 1);
}

/*
 * Returns the size of the ListViewer's data.
 */

int ListViewer::dataSize()
{
	return sizeof(ListViewerData);
}

/*
 * Draws the ListViewer.
 */

void ListViewer::draw()
{
	DrawBuf buf;
	Video *where = buf;
	int col_current, col_normal;
	int item = home;
	int max = MIN(size.y, getLength() - home);

	if (state & SF_ACTIVE)
	{
		col_normal = getColor(C_LISTVIEWER + 3);
		col_current = state & SF_FOCUSED ? getColor(C_LISTVIEWER) :
			getColor(C_LISTVIEWER + 1);
	}
	else col_current = col_normal = getColor(C_LISTVIEWER + 2);
	moveChar(buf, ' ', col_normal, size.x * size.y);
	while (max-- > 0)
	{
		char *str = getText(item);
		int col = item++ == data.current ? col_current : col_normal;
		int x = left;

		/* skip the first invisible characters */

		while (*str != '\0' && x-- > 0) str++;

		/* write the visible characters */

		for (x = 0; *str != '\0' && x < size.x; where++, x++)
		{
			where->split.code = *str++;
			where->split.attr = col;
		}

		/* write the trailing spaces */

		for (; x < size.x; where++, x++)
		{
			where->split.code = ' ';
			where->split.attr = col;
		}
	}
	writeBuf(0, 0, size.x, size.y, buf);
}

/*
 * Gets the ListViewers's data.
 */

void ListViewer::getData(void *buf)
{
	ListViewerData *lvd = (ListViewerData *) buf;

	*lvd = data;
}

/*
 * Returns the index of the current item.
 */

int ListViewer::getValue()
{
	return data.current;
}

/*
 * Handles the ListViewer's events.
 */

void ListViewer::handleEvent(Event &event)
{
	View::handleEvent(event);
	switch (event.what)
	{
	case EV_BROADCAST:
		switch (event.command)
		{
		case CM_SCROLLBARCHANGED:
			if (event.sender == hs && hs != NULL)
				left = hs->getValue();
			else if (event.sender == vs && vs != NULL)
				home = vs->getValue();
			else break;
			drawView(); 
		}
		break;
	case EV_KEYDOWN:
		switch (event.keycode)
		{
		case KC_HOME:
			moveTo(left, 0);
			break;
		case KC_END:
			moveTo(left, getLength() - 1);
			break;
		case KC_ARROWUP:
			moveTo(left, data.current - 1);
			break;
		case KC_ARROWLEFT:
			moveTo(left - 1, data.current);
			break;
		case KC_ARROWRIGHT:
			moveTo(left + 1, data.current);
			break;
		case KC_ARROWDOWN:
			moveTo(left, data.current + 1);
			break;
		case KC_PAGEUP:
			moveTo(left, data.current - size.y);
			break;
		case KC_PAGEDOWN:
			moveTo(left, data.current + size.y);
			break;
		default:
			return;
		}
		sendMessage(owner, EV_BROADCAST, CM_LISTVIEWERCHANGED);
		clearEvent(event);
		break;
	case EV_MOUSEDOUBLE:
	case EV_MOUSEDOWN:
		makeLocal(event.where, event.where);
		moveTo(left, home + event.where.y);
		sendMessage(owner, EV_BROADCAST,
			event.what == EV_MOUSEDOUBLE ? CM_LISTVIEWERDOUBLE :
			CM_LISTVIEWERCHANGED);
		clearEvent(event);
	}
}

/*
 * It is called when the user selects an item.
 */

void ListViewer::moveTo(int offset, int current)
{
	int len = getLength();

	offset = MAX(0, MIN(offset, getMaxStrLength() - size.x));
	if (len <= 0) current = -1;
	else
	{
		current = MAX(0, MIN(current, len - 1));
		home = MAX(current - size.y + 1, MIN(home, current));
	}
	if (left != offset || data.current != current)
	{
		if (hs != NULL) hs->setValue(left = offset);
		if (vs != NULL) vs->setValue(home);
		data.current = current;
		drawView();
	}
}

/*
 * Sets the ListViewers's data.
 */

void ListViewer::setData(void *buf)
{
	ListViewerData *lvd = (ListViewerData *) buf;

	data = *lvd;
	drawView();
}

/*
 * Sets the ListViewer's state.
 */

void ListViewer::setState(int flag, int enable)
{
	View::setState(flag, enable);
	switch (flag)
	{
	case SF_ACTIVE:
		if (hs != NULL) hs->setState(flag, enable);
		if (vs != NULL) vs->setState(flag, enable);
	}
}

/*
 * Updates the scrollbar limits and draws itself.
 */

void ListViewer::updateScreen()
{
	int current = data.current;

	if (hs != NULL) hs->setRange(0, MAX(0, getMaxStrLength() - size.x));
	if (vs != NULL) vs->setRange(0, MAX(0, getLength() - size.y));
	moveTo(left, data.current);
	if (data.current != current) sendMessage(owner, EV_BROADCAST,
		CM_LISTVIEWERCHANGED);
}
