/*
 * scrbar.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include "scrbar.h"

static unsigned char hchars[] =
{
	0x11, 0x10, 0xb1, 0xfe, 0xb2
};

static unsigned char vchars[] =
{
	0x1e, 0x1f, 0xb1, 0xfe, 0xb2
};

/*
 * ScrollBar constructor.
 */

ScrollBar::ScrollBar(Rect bounds): View(bounds)
{
	DEBUG("new ScrollBar at %p\n", this);
	if (size.x == 1)
	{
		growmode = GF_GROWLOX | GF_GROWHIX | GF_GROWHIY;
		codes = vchars;
	}
	else
	{
		growmode = GF_GROWLOY | GF_GROWHIX | GF_GROWHIY;
		codes = hchars;
	}
	pgstp = stp = 1;
	min = max = val = 0;
}

/*
 * Draws the ScrollBar on the screen.
 */

void ScrollBar::draw()
{
	drawPos(getPos());
}

/*
 * Draws the ScrollBar with the indicator at the specified point.
 */

void ScrollBar::drawPos(int pos)
{
	DrawBuf buf;
	int col = state & SF_ACTIVE ? getColor(C_SCROLLBAR + 1) :
		getColor(C_SCROLLBAR);
	int dim = MAX(size.x, size.y);

	moveChar(buf, codes[0], col, 1);
	if (dim > 2)
	{
		if (min == max) moveChar(buf + 1, codes[4], col, dim - 2);
		else
		{
			moveChar(buf + 1, codes[2], col, dim - 2);
			moveChar(buf + pos, codes[3], col, 1);
		}
	}
	moveChar(buf + dim - 1, codes[1], col, 1);
	writeBuf(0, 0, size.x, size.y, buf);
}

/*
 * Returns the part of the object under the specified point.
 */

int ScrollBar::getPartCode(Point pnt)
{
	if (localPointInside(pnt))
	{
		int mark = size.x == 1 ? pnt.y : pnt.x;
		int pos = getPos();

		if (mark < 1) return SB_LITTLEDOWN;
		if (mark < pos) return SB_BIGDOWN;
		if (mark == pos) return SB_INDICATOR;
		if (mark < MAX(size.x, size.y) - 1) return SB_BIGUP;
		return SB_LITTLEUP;
	}
	return SB_OUTSIDE;
}

/*
 * Returns the position of the indicator.
 */

int ScrollBar::getPos()
{
	return min == max ? 1 : (int) ((double) (val - min) *
		(MAX(size.x, size.y) - 3) / (max - min) + 1.5);
}

/*
 * Returns the value of the ScrollBar.
 */

int ScrollBar::getValue()
{
	return val;
}

/*
 * Handles the ScrollBar's events.
 */

void ScrollBar::handleEvent(Event &event)
{
	Point mouse;
	int part;

	View::handleEvent(event);
	switch (event.what)
	{
	case EV_KEYDOWN:
		part = SB_INDICATOR;
		if (size.x == 1)
		{
			switch (event.keycode)
			{
			case KC_ARROWUP:
				part = SB_LITTLEDOWN;
				break;
			case KC_ARROWDOWN:
				part = SB_LITTLEUP;
			}
		}
		else
		{
			switch (event.keycode)
			{
			case KC_ARROWLEFT:
				part = SB_LITTLEDOWN;
				break;
			case KC_ARROWRIGHT:
				part = SB_LITTLEUP;
			}
		}
		setValue(val + scrollStep(part));
		sendMessage(owner, EV_BROADCAST, CM_SCROLLBARCHANGED);
		break;
	case EV_MOUSEDOWN:
		makeLocal(event.where, mouse);
		if ((part = getPartCode(mouse)) == SB_INDICATOR)
		{
			int dim = MAX(size.x, size.y);
			int oldpos, originalpos;

			oldpos = originalpos = getPos();
			do
			{
				getEvent(event);
				if (event.what == EV_MOUSEMOVE)
				{
					int tmp;

					makeLocal(event.where, mouse);
					if (localPointInside(mouse))
					{
						tmp = size.x == 1 ? mouse.y :
							mouse.x;
						tmp = RANGE(tmp, 1, dim - 2);
					}
					else tmp = originalpos;
					if (tmp != oldpos)
					{
						oldpos = tmp;
						setValue((int) ((double)
							(tmp - 1) *
							(max - min) /
							(dim - 3) + min +
							0.5));
						sendMessage(owner,
							EV_BROADCAST,
							CM_SCROLLBARCHANGED);
					}
				}
			}
			while (event.what != EV_MOUSEUP);
		}
		else do
		{
			makeLocal(event.where, mouse);
			if (getPartCode(mouse) == part)
			{
				setValue(val + scrollStep(part));
				sendMessage(owner, EV_BROADCAST,
					CM_SCROLLBARCHANGED);
			}
			do
			{
				getEvent(event);
			}
			while (event.what != EV_MOUSEAUTO &&
				event.what != EV_MOUSEUP);
		}
		while (event.what != EV_MOUSEUP);
		clearEvent(event);
	}
}

/*
 * Returns the size of the step of the specified part.
 */

int ScrollBar::scrollStep(int part)
{
	switch (part)
	{
	case SB_BIGDOWN:
		return -pgstp;
	case SB_BIGUP:
		return pgstp;
	case SB_LITTLEDOWN:
		return -stp;
	case SB_LITTLEUP:
		return stp;
	}
	return 0;
}

/*
 * Sets all the ScrollBar's parameters.
 */

void ScrollBar::setParams(int aval, int amin, int amax, int apgstp, int astp)
{
	if (amax < amin) amax = amin;
	if (aval < amin) aval = amin;
	if (aval > amax) aval = amax;
	val = aval;
	min = amin;
	max = amax;
	pgstp = apgstp;
	stp = astp;
	drawView();
}

/*
 * Sets minumum and maximum limits.
 */

void ScrollBar::setRange(int amin, int amax)
{
	setParams(val, amin, amax, pgstp, stp);
}

/*
 * Sets steps.
 */

void ScrollBar::setStep(int apgstp, int astp)
{
	setParams(val, min, max, apgstp, astp);
}

/*
 * Sets the current value.
 */

void ScrollBar::setValue(int aval)
{
	setParams(aval, min, max, pgstp, stp);
}
