/*
 * system.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <ctype.h>
#include <fcntl.h>
#include <iostream.h>
#include <limits.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <sys/kd.h>
#include <sys/time.h>
#include <unistd.h>
#include "venus.h"

extern "C"
{
#include <gpm.h>
};

#define DELAY_MOUSEAUTO_FIRST	400	/* delay before first EV_MOUSEAUTO */
#define DELAY_MOUSEAUTO_NEXT	50	/* delay between next EV_MOUSEAUTO */
#define DELAY_SIGALRM		50	/* delay between SIGALRM signals */
#define DELAY_WAKEUP		500	/* delay between CM_WAKEUP */

#define QUEUE_LEN		32

#define TERMINFO_MAGIC		0432	/* terminfo magic number */

struct node_s			/* a node in the key tree */
{
	node_s *child;		/* points to the child */
	node_s *sibling;	/* points to the next character */
	short value;		/* key value */
	char code;		/* character code */
};

struct termheader_s		/* header in terminfo file */
{
	short magic;		/* magic number */
	short name_size;	/* size of the name section */
	short bool_cnt;		/* number of boolean flags */
	short num_cnt;		/* number of numeric flags */
	short str_cnt;		/* number of strings */
	short table_size;	/* size of the string table */
};

/* assembly linkage */

extern "C"
{
	Video *screen;		/* pointer to Program buffer */
	int mouse_x, mouse_y;	/* mouse coordinate */
	int screen_width;	/* screen extension */
	int screen_height;
	int shadow_attr;	/* shadow data */
	int shadow_x;
	int shadow_y;
	void mousedraw(int show);
	void setcuraddr(int x, int y);
	void writerow(int dst, Video *src, int len);
}
int venusopt;			/* runtime options */

/* lookup table to translate characters from pc set to standard ascii */

static unsigned char pctoascii[] =
{
	" OOooooooooo!!!*><|!!O_|^V><--^V !\"#$%&'()*+,-./0123456789:;<=>?"
	"@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~d"
	"cueaaaaceeeiiiaaeafooouuyOUcLYPfaiounN--?--//!<>:%%|{+++++I+'++."
	"`++}-+++`.+++=+++++++++++'.|-||-abipeouyooooooEn=+><()-=o..Vn2X "
};

static Event *in;		/* message queue system */
static Event *out;
static Event queue[QUEUE_LEN];
static Video *stdoutbuf;	/* buffer used when operate remotely */
static fd_set set;		/* used for select() */
static int flag_mouseauto;	/* counters set in signal handler */
static int flag_repaint;
static int flag_wakeup;
static int length;		/* number of events in the queue */
static int ms_fd;		/* mouse file descriptor */
static int ms_oldbuttons;	/* mouse button status */
static int timer_mouseauto;	/* timers used to generate events */
static int timer_wakeup;
static int out_fd;		/* output descriptor */
static node_s *kb_root;		/* key string structure */
static termios oldtermios;	/* old console properties */
static termios newtermios;	/* current console properties */

/*
 * Hides or shows the mouse pointer.
 */

void mousedraw(int show)
{
	if (ms_fd != -1)
	{
		int addr = screen_width * mouse_y + mouse_x;

		if (venusopt & VO_REMOTEMODE)
		{
			char out[1024];
			int ch = screen[addr].split.code, len;

			if (venusopt & VO_USE7BIT) ch = pctoascii[ch];
			if (show) len = sprintf(out,
				"\0337"		/* save cursor position */
				"\033[%d;%dH"	/* move cursor */
				"\033[7m"	/* enter reverse mode */
				"%c"		/* draw character */
				"\033[m"	/* exit reverse mode */
				"\0338",	/* restore cursor position */
				mouse_y + 1, mouse_x + 1, ch);
			else len = sprintf(out,
				"\0337"		/* save cursor position */
				"\033[%d;%dH"	/* move cursor */
				"%c"		/* draw character */
				"\0338",	/* restore cursor position */
				mouse_y + 1, mouse_x + 1, ch);
			write(out_fd, out, len);
		}
		else	/* console mode */
		{
			Video cell = screen[addr];

			if (show) cell.split.attr ^= 0x7f;
			lseek(out_fd, addr * sizeof(Video) + 4, SEEK_SET);
			write(out_fd, &cell, sizeof(cell));
		}
	}
}

/*
 * Moves the cursor to another place.
 */

void setcuraddr(int x, int y)
{
	if (venusopt & VO_REMOTEMODE)
	{ 
		char out[1024];

		write(out_fd, out, sprintf(out, "\033[%d;%dH", y + 1, x + 1));
	}
	else	/* console mode */
	{
		unsigned char where[2];

		where[0] = x;
		where[1] = y;
		lseek(out_fd, 2, SEEK_SET);
		write(out_fd, where, sizeof(where));
	}
}

/*
 * Draws a line of text on the screen.
 */

void writerow(int dst, Video *src, int len)
{
	if (venusopt & VO_REMOTEMODE)
	{
		Video *old = stdoutbuf + dst;
		Video *old_right = old + len - 1;
		Video *src_right = src + len - 1;

		/* remove unchanged characters from left to right */

		while (len > 0 && old->pair == src->pair)
		{
			dst++;
			len--;
			old++;
			src++;
		}

		/* remove unchanged characters from right to left */

		while (len > 0 && old_right->pair == src_right->pair)
		{
			len--;
			old_right--;
			src_right--;
		}

		/* write only middle changed characters */

		if (len > 0)
		{
			char out[1024];
			int col = -1, i;

			i = sprintf(out,
				"\0337"		/* save cursor position */
				"\033[%d;%dH",	/* move cursor */
				dst / screen_width + 1,
				dst % screen_width + 1);
			while (len-- > 0)
			{
				int ch;

				*old++ = *src;

				/* change color ? */

				if (!(venusopt & VO_NOCOLOR) &&	(col == -1 ||
					col != src->split.attr))
				{
					/* set foreground color */

					col = src->split.attr;
					i += sprintf(out + i, "\033[3%dm",
						col & 0x7);
				}
				ch = src++->split.code;
				if (venusopt & VO_USE7BIT) ch = pctoascii[ch];
				out[i++] = ch;
			}
			out[i++] = '\033';
			out[i++] = '8';		/* restore cursor position */
			write(out_fd, out, i);
		}
	}
	else	/* console mode */
	{
		lseek(out_fd, dst * sizeof(Video) + 4, SEEK_SET);
		write(out_fd, src, len * sizeof(Video));
	}
}

/*
 * Gets an event from the queue.
 */

void System::getEvent(Event &event)
{
	event.what = EV_NONE;
	if (length > 0)		/* handles pending events */
	{
		length--;
		event = *out;
		if (++out == &queue[QUEUE_LEN]) out = &queue[0];
	}
	else if (flag_mouseauto > 0)
	{
		flag_mouseauto--;
		event.what = EV_MOUSEAUTO;
		event.buttons = ms_oldbuttons;
	}
	else if (flag_repaint > 0)
	{
		flag_repaint--;
		event.what = EV_COMMAND;
		event.command = CM_REPAINT;
		if (venusopt & VO_REMOTEMODE)
		{
			/* discard the buffer so it redraws all the screen */

			memset(stdoutbuf, 0, screen_width * screen_height *
				sizeof(Video));
		}
	}
	else if (flag_wakeup > 0)
	{
		time_t now = time(NULL);
		tm *broken = localtime(&now);

		flag_wakeup--;
		event.what = EV_BROADCAST;
		event.command = CM_WAKEUP;
		event.hour = broken->tm_hour;
		event.min = broken->tm_min;
		event.sec = broken->tm_sec;
	}
	else
	{
		fd_set ready = set;

		/*
		 * suspend until there is a signal or some data in file
		 * descriptors
		 */
		if (select(FD_SETSIZE, &ready, NULL, NULL, NULL) > 0)
		{
			if (FD_ISSET(STDIN_FILENO, &ready))
			{
				unsigned char buf[256];

				kbHandle(buf, read(STDIN_FILENO, buf,
					sizeof(buf)));
			}
			if (ms_fd != -1 && FD_ISSET(ms_fd, &ready))
			{
				Gpm_Event m_ev;

				Gpm_GetEvent(&m_ev);
				msHandle(&m_ev);
			}
		}
	}
}

/*
 * Puts an event in the queue.  If the queue is full the event will be
 * discarded.
 */

void System::putEvent(Event &event)
{
	if (event.what == EV_COMMAND && event.command == CM_BEEP)
	{
		/*
		 * Generates a tone.
		 *
		 * high word = clock ticks
		 * low word = counter value (1193180 / frequency)
		 */
		ioctl(STDIN_FILENO, KDMKTONE, 0x005004a9);
	}
	else if (length < QUEUE_LEN)
	{
		length++;
		*in = event;
		if (++in == &queue[QUEUE_LEN]) in = &queue[0];
	}
}

/*
 * Inserts a key string in the tree.
 */

void System::kbAddKey(char *str, short value)
{
	node_s *tag;

	/* don't allow strings like \Ec or \E7 */

	if (str[0] == '\x1b' && isalnum(str[1])) return;

	/* the root not exists ? */

	if (kb_root == NULL) kb_root = kbCreateNode(*str);
	tag = kb_root;

	while (*str != '\0')
	{
		node_s *prev = NULL;

		/* checks if the entry exists */

		while (tag != NULL && tag->code != *str)
		{
			prev = tag;
			tag = tag->sibling;
		}

		/* entry not exists, so creates one */

		if (tag == NULL)
		{
			prev->sibling = tag = kbCreateNode(*str);
		}

		/* one character is scanned */

		if (*++str != '\0')
		{
			/* if entry has childs goes to the first of them */

			if (tag->child != NULL) tag = tag->child;
			else do
			{
				/* otherwise creates the childs */

				tag->child = kbCreateNode(*str);
				tag = tag->child;
			}
			while (*++str != '\0');
		}
	}
	tag->value = value;
}

/*
 * Builds the tree by inserting all the key strings.
 */

void System::kbBuildTree(short *offset, char *table, int count)
{
	int i;

	cerr << __FILE__": terminfo table has " << count << " entries\n";
	kb_root = NULL;
	for (i = 0; i < count; i++)
	{
		if (offset[i] >= 0) kbAddKey(table + offset[i], i);
	}
}

/*
 * Creates a node and returns its address.
 */

node_s *System::kbCreateNode(char code)
{
	node_s *node = new node_s;

	node->child = node->sibling = NULL;
	node->code = code;
	node->value = -1;
	return node;
}

/*
 * Gets characters from the console, checks if they are escape sequences
 * or meta characters and puts the appropriate event in the queue.
 */

void System::kbHandle(unsigned char *buf, int len)
{
	Event event;
	int pos = 0;

	event.what = EV_KEYDOWN;

	/* loop until the end of the buffer */

	while (pos < len)
	{
		int chk = pos;
		int ok = 0;
		node_s *prev = NULL;
		node_s *tag = kb_root;

		/* if it can be an escape sequence starts the search */

		while (chk < len && tag != NULL)
		{
			/* finds the character in the sibling list */

			while (tag != NULL && tag->code != buf[chk])
			{
				tag = tag->sibling;
			}

			/* if the entry exists checks in its childs */

			if (tag != NULL)
			{
				/* if it can be a valid key remember it */

				chk++;
				if (tag->value >= 0)
				{
					prev = tag;
					ok = chk;
				}
				tag = tag->child;
			}
		}

		/* is it an escape key ? */

		if (prev != NULL && prev->value >= 0)
		{
			event.keycode = MK_ESC(prev->value);
			pos = ok;
		}
		else if (buf[pos] == 0x1b && len - pos >= 2)
		{
			/* is it a meta key ? */

			event.keycode = MK_META(toupper2(buf[++pos]));
			pos++;
		}
		else event.keycode = buf[pos++];
		putEvent(event);
	}
}

/*
 * Recursive function to destroy a node and its childs.
 */

void System::kbKillTree(node_s *base)
{
	if (base != NULL)
	{
		node_s *next = base->sibling;

		kbKillTree(base->child);
		delete base;
		base = next;
	}
}

/*
 * Reads the terminfo compiled file.
 */

int System::kbLoadTerm()
{
	FILE *fs;
	char *dir, *term, path[PATH_MAX];
	termheader_s h;

	if ((dir = getenv("TERMINFO")) == NULL)
	{
		dir = "/usr/lib/terminfo";
	}
	if ((term = getenv("TERM")) == NULL)
	{
		cerr << __FILE__": environment variable TERM not set\n";
		return -1;
	}
	sprintf(path, "%s/%c/%s", dir, term[0], term);

	/* open the file */

	if ((fs = fopen(path, "r")) == NULL)
	{
		cerr << __FILE__": unable to open terminfo " << path << "\n";
		return -1;
	}
	cerr << __FILE__": read terminfo " << path << "\n";

	/* read the header */

	if (fread(&h, sizeof(h), 1, fs) != 1)
	{
		fclose(fs);
		cerr << __FILE__": unable to read terminfo " << path << "\n";
		return -1;
	}

	/* check file type */

	if (h.magic == TERMINFO_MAGIC)
	{
		int skip_bytes = h.name_size + h.bool_cnt;

		/* align to an even word boundary */

		if ((skip_bytes % 2) != 0) skip_bytes++;
		skip_bytes += sizeof(short) * h.num_cnt;

		/* move to offset table */

		if (fseek(fs, skip_bytes, SEEK_CUR) == 0)
		{
			short *off = new short [h.str_cnt];

			/* read offset table */

			if (fread(off, sizeof(short) * h.str_cnt, 1, fs) == 1)
			{
				char *table = new char [h.table_size];

				/* read string table */

				if (fread(table, h.table_size, 1, fs) == 1)
				{
					fclose(fs);
					kbBuildTree(off, table, h.str_cnt);
					delete off, table;
					return 0;
				}
				delete table;
			}
			delete off;
		}
	}
	fclose(fs);
	cerr << __FILE__": bad terminfo " << path << "\n";
	return -1;
}

/*
 * Mouse handler called by Gpm_Getchar().
 */

void System::msHandle(Gpm_Event *m_ev)
{
	Event event;

	event.where.x = RANGE(m_ev->x, 0, screen_width - 1);
	event.where.y = RANGE(m_ev->y, 0, screen_height - 1);
	if (m_ev->type & GPM_DOUBLE && m_ev->type & GPM_UP)
	{
		event.buttons = ms_oldbuttons;
		if (m_ev->buttons & GPM_B_LEFT)
		{
			event.what = EV_MOUSEDOUBLE;
			putEvent(event);
		}
		if (m_ev->buttons & GPM_B_RIGHT)
		{
			event.what = EV_MOUSEDOUBLE_R;
			putEvent(event);
		}
	}
	if (m_ev->type & (GPM_DRAG | GPM_MOVE))
	{
		timer_mouseauto = -1;
		event.buttons = ms_oldbuttons = m_ev->buttons;
		mousedraw(0);
		mouse_x = event.where.x;
		mouse_y = event.where.y;
		mousedraw(1);
		event.what = EV_MOUSEMOVE;
		putEvent(event);
	}
	if (m_ev->type & GPM_DOWN)
	{
		int trigger = m_ev->buttons & ~ms_oldbuttons;

		timer_mouseauto = DELAY_MOUSEAUTO_FIRST;
		event.buttons = ms_oldbuttons = m_ev->buttons;
		if (trigger & GPM_B_LEFT)
		{
			event.what = EV_MOUSEDOWN;
			putEvent(event);
		}
		if (trigger & GPM_B_RIGHT)
		{
			event.what = EV_MOUSEDOWN_R;
			putEvent(event);
		}
	}
	if (m_ev->type & GPM_UP)
	{
		timer_mouseauto = -1;
		event.buttons = ms_oldbuttons &= ~m_ev->buttons;
		if (m_ev->buttons & GPM_B_LEFT)
		{
			event.what = EV_MOUSEUP;
			putEvent(event);
		}
		if (m_ev->buttons & GPM_B_RIGHT)
		{
			event.what = EV_MOUSEUP_R;
			putEvent(event);
		}
	}
}

/*
 * General signal handler.
 */

void System::sigHandler(int signo)
{
	switch (signo)
	{
	case SIGALRM:
		/*
		 * called every DELAY_SIGALRM ms
		 */
		signal(SIGALRM, sigHandler);
		if (timer_mouseauto >= 0)
		{
			if ((timer_mouseauto -= DELAY_SIGALRM) <= 0)
			{
				flag_mouseauto++;
				timer_mouseauto = DELAY_MOUSEAUTO_NEXT;
			}
		}
		if ((timer_wakeup -= DELAY_SIGALRM) <= 0)
		{
			flag_wakeup++;
			timer_wakeup = DELAY_WAKEUP;
		}
		break;
	case SIGCONT:
		/*
		 * called when the user restart the process after a ctrl-z
		 */
		signal(SIGCONT, sigHandler);
		signal(SIGTSTP, sigHandler);
		flag_repaint++;
		tcsetattr(STDIN_FILENO, TCSAFLUSH, &newtermios);
		cerr << __FILE__": continuing process\n";
		break;
	case SIGTSTP:
		/*
		 * called when the user presses ctrl-z
		 */
		write(STDOUT_FILENO, "\033[H\033[J", 6); /* clear screen */
		cerr << __FILE__": process stopped, type 'fg' to resume\n";
		tcsetattr(STDIN_FILENO, TCSAFLUSH, &oldtermios);
		signal(SIGTSTP, SIG_DFL);	/* use default handler */
		raise(SIGTSTP);		/* stop the process */
	}
}

/*
 * System destructor.
 */

System::~System()
{
	write(STDOUT_FILENO, "\033[H\033[J", 6);	/* clear screen */
	kbKillTree(kb_root);
	tcsetattr(STDIN_FILENO, TCSAFLUSH, &oldtermios);
	if (ms_fd != -1) Gpm_Close();
	if (out_fd != STDOUT_FILENO) close(out_fd);
	if (stdoutbuf != NULL) delete stdoutbuf;
	cerr << __FILE__": terminated\n";
}

/*
 * System constructor.
 */

System::System()
{
	Gpm_Connect conn;
	char *term;
	int console_no = -1;
	struct itimerval timer;

	/* load terminfo */

	if (kbLoadTerm() < 0) exit(EXIT_FAILURE);

	/* other stuff */

	flag_mouseauto = flag_repaint = flag_wakeup = 0;
	in = out = &queue[0];
	length = 0;
	shadow_attr = 0x08;
	shadow_x = 2;
	shadow_y = 1;
	timer_mouseauto = -1;
	timer_wakeup = DELAY_WAKEUP;
	venusopt = 0;

	/* mouse stuff */

	conn.eventMask = ~0;		/* I want all events */
	conn.defaultMask = 0;		/* no default treatment */
	conn.maxMod = ~0;
	conn.minMod = 0;
	gpm_zerobased = 1;		/* coordinates start from zero */
	if ((ms_fd = Gpm_Open(&conn, 0)) == -1)
	{
		winsize win;

		cerr << __FILE__": no Gpm, running without mouse\n";
		ioctl(STDIN_FILENO, TIOCGWINSZ, &win);
		if (win.ws_col > 0 && win.ws_row > 0)
		{
			screen_width = win.ws_col;
			screen_height = win.ws_row;
		}
		else
		{
			cerr << __FILE__": unable to detect screen size\n";
			screen_width = 80;
			screen_height = 25;
		}
	}
	else
	{
		cerr << __FILE__": Gpm server version is " <<
			Gpm_GetServerVersion(NULL) << "\n";
		screen_width = gpm_mx + 1;
		screen_height = gpm_my + 1;
	}
	cerr << __FILE__": screen is " << screen_width << "x" <<
		screen_height << "\n";
	mouse_x = mouse_y = ms_oldbuttons = 0;

	/* input stuff */

	tcgetattr(STDIN_FILENO, &oldtermios);
	newtermios = oldtermios;

	/*
	 * ECHO		enables echoing of the input characters back to the
	 *		terminal
	 * ICANON	enables canonical mode with editing facilities
	 */
	newtermios.c_lflag &= ~(ECHO | ICANON);

	/*
	 * VMIN		specifies the minimum number of bytes that must be
	 *		available in the input queue in order for read() to
	 *		return
	 * VTIME	specifies how long to wait for input before
	 *		returning, in units of 0.1 seconds
	 */
	newtermios.c_cc[VMIN] = 1;
	newtermios.c_cc[VTIME] = 0;
	tcsetattr(STDIN_FILENO, TCSAFLUSH, &newtermios);

	/* output stuff */

	if ((term = ttyname(STDOUT_FILENO)) != NULL)
	{
		cerr << __FILE__": you are connected in " << term << "\n";
		if (memcmp(term, "/dev/tty", 8) == 0)
		{
			char *tail, *test = term + 8;

			/*
			 * terminals like tty1 are system console, others like
			 * ttyS1 or ttyp1 are remote connections, so if
			 * conversion fails we are running remotely
			 */
			console_no = strtol(test, &tail, 10);
			if (test == tail) console_no = -1;
		}
	}
	if (console_no >= 0)
	{
		char dev[PATH_MAX];

		cerr << __FILE__": running in console mode\n";
		sprintf(dev, "/dev/vcsa%d", console_no);
		if ((out_fd = open(dev, O_WRONLY)) < 0)
		{
			cerr << __FILE__": unable to open " << dev << "\n";
			exit(EXIT_FAILURE);
		}
		stdoutbuf = NULL;
	}
	else
	{
		cerr << __FILE__": running in remote mode\n";
		out_fd = STDOUT_FILENO;
		stdoutbuf = new Video [screen_width * screen_height];
		venusopt |= VO_REMOTEMODE | VO_USE7BIT;

		/* don't draw shadows */

		shadow_x = shadow_y = 0;
	}

	/* setup file descriptors */

	FD_ZERO(&set);
	FD_SET(STDIN_FILENO, &set);
	if (ms_fd != -1) FD_SET(ms_fd, &set);

	/* catch useful signals */

	signal(SIGALRM, sigHandler);
	signal(SIGCONT, sigHandler);
	signal(SIGTSTP, sigHandler);

	/* generates a SIGALRM signal every DELAY_SIGALRM ms */

	timer.it_interval.tv_usec = timer.it_value.tv_usec =
		DELAY_SIGALRM * 1000;
	timer.it_interval.tv_sec = timer.it_value.tv_sec = 0;
	setitimer(ITIMER_REAL, &timer, NULL);
}
