/*
 * venus.h
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef __VENUS_H__
#define __VENUS_H__

#define VERSION		"0.5"

#ifdef __DEBUG__
	void logstr(char *format, void *data);
	#define DEBUG(a, b)	logstr(a, b);
#else
	#define DEBUG(a, b)
#endif

#define COMMANDSET_SIZE	1024

#define	DEBUG_FILE	"log"

#define MAX_WIDTH	160
#define MAX_HEIGHT	60

typedef union
{
	struct
	{
		unsigned char code;
		unsigned char attr;
	}
	split;
	unsigned short pair;
}
Video;

typedef Video DrawBuf[MAX_WIDTH * MAX_HEIGHT];

class Point
{
public:
	int x;
	int y;
};

class Rect	/* don't declare virtual methods here */
{
public:
	Point a;
	Point b;
	Rect()
	{
	}
	Rect(int ax, int ay, int bx, int by)
	{
		makeAssign(ax, ay, bx, by);
	}
	void check();
	int containsPoint(Point pnt);
	int isEmpty();
	int isEqual(Rect rct);
	void makeAssign(int ax, int ay, int bx, int by);
	void makeCopy(Rect rct);
	void makeGrow(int deltax, int deltay);
	void makeIntersect(Rect rct);
	void makeMove(int deltax, int deltay);
	void makeUnion(Rect rct);
};

#define EV_NONE			0x00000000

#define ET_BROADCAST		0x10000000
#define EV_BROADCAST		0x10000000

#define ET_FOCUSED		0x20000000
#define EV_KEYDOWN		0x20000000
#define EV_COMMAND		0x20000001

#define ET_POSITIONAL		0x40000000
#define EV_MOUSEAUTO		0x40000001
#define EV_MOUSEDOUBLE		0x40000002
#define EV_MOUSEDOUBLE_R	0x40000003
#define EV_MOUSEDOWN		0x40000004
#define EV_MOUSEDOWN_R		0x40000005
#define EV_MOUSEMOVE		0x40000006
#define EV_MOUSEUP		0x40000007
#define EV_MOUSEUP_R		0x40000008

class Event
{
public:
	Point where;
	int what;
	int buttons;
	int command;
	int keycode;
	int hour;
	int min;
	int sec;
	void *sender;
};

#define GF_GROWLOX	0x00000001
#define GF_GROWLOY	0x00000002
#define GF_GROWHIX	0x00000004
#define GF_GROWHIY	0x00000008

#define MF_CONFIRM	0x00000000
#define MF_ERROR	0x00000001
#define MF_INFORMATION	0x00000002
#define MF_WARNING	0x00000003

#define MF_YESBUTTON	0x00000100
#define MF_NOBUTTON	0x00000200
#define MF_OKBUTTON	0x00000400
#define MF_CANCELBUTTON	0x00000800

#define MF_OKCANCEL	(MF_OKBUTTON | MF_CANCELBUTTON)
#define MF_YESNOCANCEL	(MF_YESBUTTON | MF_NOBUTTON | MF_CANCELBUTTON)

#define MSGBOX_BUTTONS	4

#define OF_CENTERX	0x00000001
#define OF_CENTERY	0x00000002
#define OF_FIRSTCLICK	0x00000004
#define OF_POSTPROCESS	0x00000008
#define OF_PREPROCESS	0x00000010
#define OF_SCREENLINK	0x00000020
#define OF_SELECTABLE	0x00000040
#define OF_TOPSELECT	0x00000080

#define SF_ACTIVE	0x00000001
#define SF_CURSORVIS	0x00000002
#define SF_DELETE	0x00000004
#define SF_DRAGGING	0x00000008
#define SF_EXPOSED	0x00000010
#define SF_FOCUSED	0x00000020
#define SF_MODAL	0x00000040
#define SF_SHADOW	0x00000080
#define SF_VISIBLE	0x00000100
#define SF_ZOOMED	0x00000200

class Desktop;
class Editor;
class Group;
class MenuBar;
class Program;
class StatusLine;
class Window;

class View
{
	friend Group;
protected:
	static Desktop *desktop;
	static Editor *clipboard;
	static MenuBar *menubar;
	static Program *program;
	static StatusLine *statusline;
	static Window *clipwindow;
	Group *owner;	/* do not change this order */
	Point cursor;
	Point origin;
	Point size;
	View *next;
	View *prev;
	int growmode;
	int options;			
	int state;
	unsigned helpctx;
public:
	View(Rect bounds);
	virtual ~View();
	virtual void calcBounds(Rect &bounds, Point delta);
	virtual void changeBounds(Rect bounds);
	virtual void clearEvent(Event &event);
	virtual int cStrLen(char *str);
	virtual int dataSize();
	virtual void draw() = 0;
	virtual void drawCursor();
	virtual void drawView();
	virtual int execModal();
	virtual void getBounds(Rect &bounds);
	virtual void getClipRect(Rect &clip);
	virtual int getColor(int color);
	virtual void getData(void *buf);
	virtual void getEvent(Event &event);
	virtual void getExtent(Rect &extent);
	virtual int getFocus();
	virtual int getHeight();
	virtual int getHelpCtx();
	virtual int *getPalette();
	virtual Point getSize();
	virtual int getState();
	virtual int getWidth();
	virtual void handleEvent(Event &event);
	virtual void hideCursor();
	virtual int isExposed();
	virtual int isValid(int command);
	virtual int localPointInside(Point pnt);
	virtual void makeBeep();
	virtual void makeGlobal(Point &src, Point &dst);
	virtual void makeLocal(Point &src, Point &dst);
	virtual int messageBox(char *msg, void *info, int flags);
	virtual int mouseInView(Point mouse);
	virtual void moveBuf(Video *dst, unsigned char *src, char attr,
		int count);
	virtual void moveChar(Video *buf, unsigned char code, char attr, int count);
	virtual void moveCStr(Video *buf, char *str, int attrs);
	virtual void moveFrame(Video *buf, int wide, char attr);
	virtual void moveStr(Video *buf, char *str, char attr);
	virtual void moveTo(int x, int y);
	virtual int ownerPointInside(Point pnt);
	virtual void putEvent(Event &event);
	virtual void sendMessage(View *receiver, int what, int command);
	virtual void setBounds(Rect bounds);
	virtual void setCursor(int x, int y);
	virtual void setData(void *buf);
	virtual void setState(int flag, int enable);
	virtual void showCursor();
	virtual void sizeLimits(Point &min, Point &max);
	virtual void updateCommands();
	virtual void writeBuf(int x, int y, int w, int h, Video *buf);
	virtual void writeLine(int x, int y, int w, int h, Video *buf);
};

enum
{
	PH_FOCUSED,
	PH_POSTPROCESS,
	PH_PREPROCESS
};

class Group: public View
{
	friend View;
protected:
	Rect clip;	/* do not change this order */
	Video *buffer;
	View *current;
	View *first;
	int endcommand;
	int lockflag;
	int phase;
public:
	Group(Rect bounds, int buffered);
	~Group();
	void changeBounds(Rect bounds);
	int dataSize();
	virtual void doHandleEvent(Event &event, View *target);
	void draw();
	void drawCursor();
	virtual void endModal(int command);
	int execModal();
	virtual int execView(View *obj);
	virtual int focusNext(int forward);
	virtual int forceFocusNext(int forward);
	void getData(void *buf);
	int getHelpCtx();
	virtual int getPhase();
	virtual void handleBroadcastEvent(Event &event);
	void handleEvent(Event &event);
	virtual void insert(View *obj);
	virtual void insertInternal(View *obj);
	int isValid(int command);
	virtual void lockGroup();
	virtual void makeFirst(View *obj);
	virtual void movePast(View *obj, View *target);
	virtual void redrawAll();
	virtual void remove(View *obj);
	virtual void removeInternal(View *obj);
	virtual void setCurrent(View *obj);
	void setData(void *buf);
	void setState(int flag, int enable);
	virtual void unlockGroup();
};

struct Gpm_Event;
struct node_s;

class System
{
public:
	System();
	virtual ~System();
	virtual void getEvent(Event &event);
	virtual void kbAddKey(char *str, short value);
	virtual void kbBuildTree(short *offset, char *table, int count);
	virtual node_s *kbCreateNode(char code);
	virtual void kbHandle(unsigned char *buf, int len);
	virtual void kbKillTree(node_s *base);
	virtual int kbLoadTerm();
	virtual void msHandle(Gpm_Event *m_ev);
	virtual void putEvent(Event &event);
	static void sigHandler(int signo);
};

class Program: public System, public Group
{
public:
	Program(Desktop *(*pdesktop)(Rect), MenuBar *(*pmenubar)(Rect),
		StatusLine *(*pstatusline)(Rect));
	virtual int execDialog(Group *obj, void *buf);
	void getEvent(Event &event);
	int *getPalette();
	static Desktop *initDesktop(Rect bounds);
	static MenuBar *initMenuBar(Rect bounds);
	static StatusLine *initStatusLine(Rect bounds);
	void putEvent(Event &event);
	virtual int run();
};

enum
{
	CM_NONE,
	CM_CANCEL,		/* Button */
	CM_NO,
	CM_OK,
	CM_YES,
	CM_BACKWARD,		/* Calendar */
	CM_FORWARD,
	CM_CLUSTERCHANGED,	/* Cluster */
	CM_NEXT,		/* Desktop */
	CM_PREV,
	CM_SHOWCLIPBOARD,
	CM_BACKSPACE,		/* Editor */
	CM_CHARLEFT,
	CM_CHARRIGHT,
	CM_COPY,
	CM_CUT,
	CM_DELCHAR,
	CM_DELEND,
	CM_DELLINE,
	CM_DELWORD,
	CM_DELSTART,
	CM_FIND,
	CM_LINEDOWN,
	CM_LINEEND,
	CM_LINESTART,
	CM_LINEUP,
	CM_NEWLINE,
	CM_PAGEDOWN,
	CM_PAGEUP,
	CM_PASTE,
	CM_REPLACE,
	CM_SEARCHAGAIN,
	CM_SELEND,
	CM_SELSTART,
	CM_SETOPTIONS,
	CM_TEXTEND,
	CM_TEXTSTART,
	CM_WORDLEFT,
	CM_WORDRIGHT,
	CM_SAVE,		/* FileEditor */
	CM_SAVEAS,
	CM_LISTVIEWERCHANGED,	/* ListViewer */
	CM_LISTVIEWERDOUBLE,
	CM_MENU,		/* MenuBar */
	CM_QUIT,		/* Program */
	CM_SCROLLBARCHANGED,	/* ScrollBar */
	CM_BEEP,		/* System */
	CM_REPAINT,
	CM_WAKEUP,
	CM_COMMANDSETCHANGED,	/* View */
	CM_GOTFOCUS,
	CM_LOSTFOCUS,
	CM_CLOSE,		/* Window */
	CM_DEFAULT,
	CM_MOVE,
	CM_MOVEMOUSE,
	CM_RESIZE,
	CM_RESIZEMOUSE,
	CM_UPDATETITLE,
	CM_ZOOM,
	CM_USER			/* first user available command */
};

enum
{
	HC_NONE,
	HC_USER			/* first user available help context */
};

/* colors in the Program palette */

#define C_BACKGROUND		0	/*  0 Background */
#define C_MENUBAR		1	/*  1 MenuBar normal */
					/*  2 MenuBar normal disabled */
					/*  3 MenuBar selection */
					/*  4 MenuBar selection disabled */
#define C_STATUSLINE		5	/*  5 StatusLine normal */
					/*  6 StatusLine normal disabled */
					/*  7 StatusLine selection */
					/*  8 StatusLine selection disabled */
#define C_SUBMENU		9	/*  9 SubMenu normal */
					/* 10 SubMenu normal disabled */
					/* 11 SubMenu selection */
					/* 12 SubMenu selection disabled */

/* indices in the Window palette */

#define C_BINEDITOR		0	/*  0 BinEditor left */
					/*  1 BinEditor middle */
					/*  2 BinEditor middle current */
					/*  3 BinEditor right */
					/*  4 BinEditor right current */
#define C_BININDICATOR		5	/*  5 BinIndicator */
#define C_BUTTON		6	/*  6 Button background */
					/*  7 Button default */
					/*  8 Button disabled */
					/*  9 Button focused */
					/* 10 Button normal */
#define C_CALENDARVIEW		11	/* 11 CalendarView */
#define C_CLOCKVIEW		12	/* 12 ClockView */
#define C_CLUSTER		13	/* 13 Cluster disabled */
					/* 14 Cluster normal */
					/* 15 Cluster focused */
#define C_EDITOR		16	/* 16 Editor disabled */
					/* 17 Editor codes normal */
					/* 18 Editor codes selected */
					/* 19 Editor text normal */
					/* 20 Editor text selected */
#define C_FRAME			21	/* 21 Frame active */
					/* 22 Frame dragging */
					/* 23 Frame icon */
					/* 24 Frame interior */
					/* 25 Frame passive */
#define C_INDICATOR		26	/* 26 Indicator disabled */
					/* 27 Indicator normal */
#define C_INPUTLINE		28	/* 28 Inputline arrows */
					/* 29 Inputline disabled */
					/* 30 Inputline normal */
#define C_LABEL			31	/* 31 Label disabled */
					/* 32 Label focused */
					/* 33 Label normal */
#define C_LISTVIEWER		34	/* 34 ListViewer current focused */
					/* 35 ListViewer current normal */
					/* 36 ListViewer disabled */
					/* 37 ListViewer normal */
#define C_PROGRESSBAR		38	/* 38 ProgressBar disabled left */
					/* 39 ProgressBar disabled right */
					/* 40 ProgressBar left */
					/* 41 ProgressBar right */
#define C_SCROLLBAR		42	/* 42 ScrollBar disabled */
					/* 43 ScrollBar normal */
#define C_STATICTEXT		44	/* 44 StaticText disabled */
					/* 45 StaticText normal */

#define MK_ESC(a)	(a | 0x10000000)
#define MK_META(a)	(a | 0x20000000)

/* ASCII keycodes */

#define KC_NONE		-1
#define KC_ESC		27

/* escape keycodes */

#define KC_BACKSPACE	MK_ESC(55)
#define KC_DELETE	MK_ESC(59)
#define KC_ARROWDOWN	MK_ESC(61)
#define KC_F1		MK_ESC(66)
#define KC_F10		MK_ESC(67)
#define KC_F2		MK_ESC(68)
#define KC_F3		MK_ESC(69)
#define KC_F4		MK_ESC(70)
#define KC_F5		MK_ESC(71)
#define KC_F6		MK_ESC(72)
#define KC_F7		MK_ESC(73)
#define KC_F8		MK_ESC(74)
#define KC_F9		MK_ESC(75)
#define KC_HOME		MK_ESC(76)
#define KC_INSERT	MK_ESC(77)
#define KC_ARROWLEFT	MK_ESC(79)
#define KC_PAGEDOWN	MK_ESC(81)
#define KC_PAGEUP	MK_ESC(82)
#define KC_ARROWRIGHT	MK_ESC(83)
#define KC_ARROWUP	MK_ESC(87)
#define	KC_ENTER	MK_ESC(129)
#define KC_TAB		MK_ESC(134)
#define KC_END		MK_ESC(164)
#define KC_F11		MK_ESC(216)
#define KC_F12		MK_ESC(217)

/* Alt-letter keycodes */

#define KC_METATAB	MK_META('\t')
#define KC_METAX	MK_META('X')

inline int MAX(int a, int b)
{
	return a > b ? a : b;
}

inline int MIN(int a, int b)
{
	return a < b ? a : b;
}

inline int RANGE(int test, int min, int max)
{
	return test < min ? min : test > max ? max : test;
}

int commandEnabled(int command);
void enableAllCommands();
void expandPath(char *path, char *dir, char *file);
int fileLength(char *filename);
int hotKey(char *name);
void setCommandState(int command, int enable);
void strncpy2(char *to, char *from, int size);
int toupper2(int code);

extern "C"
{
	extern Video *screen;
	extern int screen_width;
	extern int screen_height;
};
extern int venusopt;

/* these options are used only in remote mode */

#define VO_NOCOLOR	0x00000001
#define VO_REMOTEMODE	0x00000002
#define VO_USE7BIT	0x00000004

#endif
