/*
 * view.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include "button.h"
#include "stattxt.h"
#include "window.h"

Desktop *View::desktop = NULL;
Editor *View::clipboard = NULL;
MenuBar *View::menubar = NULL;
Program *View::program = NULL;
StatusLine *View::statusline = NULL;
Window *View::clipwindow = NULL;

/* assembly linkage */

extern "C"
{
	int isexposed(View *obj);
	void resetcursor(View *obj);
	void writebuf(View *obj, int x, int y, int w, int h, Video *buf);
	void writeline(View *obj, int x, int y, int w, int h, Video *buf);
};

/*
 * Creates a view object with the given bounds rectangle.
 */

View::View(Rect bounds)
{
	DEBUG("new View at %p\n", this);
	cursor.x = cursor.y = 0;
	growmode = 0;
	helpctx = HC_NONE;
	next = prev = NULL;
	options = 0;
	owner = NULL;
	state = SF_ACTIVE | SF_EXPOSED | SF_VISIBLE;
	setBounds(bounds);
}

/*
 * If the view has an owner, deletes it from the group.
 */

View::~View()
{
	DEBUG("delete View at %p\n", this);
	if (owner != NULL) owner->remove(this);
}

/*
 * Calculates the new bounds of the view.
 */

void View::calcBounds(Rect &bounds, Point delta)
{
	Point min, max;

	getBounds(bounds);
	if (growmode & GF_GROWLOX) bounds.a.x += delta.x;
	if (growmode & GF_GROWLOY) bounds.a.y += delta.y;
	if (growmode & GF_GROWHIX) bounds.b.x += delta.x;
	if (growmode & GF_GROWHIY) bounds.b.y += delta.y;
	sizeLimits(min, max);
	bounds.b.x = bounds.a.x + RANGE(bounds.b.x - bounds.a.x, min.x,
		max.x);
	bounds.b.y = bounds.a.y + RANGE(bounds.b.y - bounds.a.y, min.y,
		max.y);
}

/*
 * Changes the view's bounds (origin and size fields) to the rectangle given
 * by the bounds parameter, then redraws the view.
 */

void View::changeBounds(Rect bounds)
{
	setBounds(bounds);
	drawView();
}

/* 
 * This is the standard method used to signal that the view has successfully
 * handled the event.
 */

void View::clearEvent(Event &event)
{
	event.what = EV_NONE;
	event.sender = NULL;
}

/*
 * Returns the length of the string without counting the tilde characters.
 */

int View::cStrLen(char *str)
{
	int len = 0;

	while (*str != '\0')
	{
		if (*str++ != '~') len++;
	}
	return len;
}

/*
 * Returns the size of the view's data.
 */

int View::dataSize()
{
	return 0;
}

/*
 * Shows the cursor in the view.
 */

void View::drawCursor()
{
	if (state & SF_FOCUSED) resetcursor(this);
}

/*
 * Draws the view on the screen.
 */

void View::drawView()
{
	if (isExposed())
	{
		draw();
		drawCursor();
	}
}

/*
 * Executes the View in modal state.
 */

int View::execModal()
{
	return CM_CANCEL;
}

/*
 * Returns, in the bounds variable, the bounding rectangle of the view in
 * its owner's coordinate system.
 */

void View::getBounds(Rect &bounds)
{
	bounds.a = origin;
	bounds.b.x = origin.x + size.x;
	bounds.b.y = origin.y + size.y;
}

/*
 * Returns, in the clip variable, the minimum rectangle that needs redrawing.
 */

void View::getClipRect(Rect &clip)
{
	getBounds(clip);
	if (owner != NULL) clip.makeIntersect(owner->clip);
	clip.makeMove(-origin.x, -origin.y);
}

/*
 * Gets the specified color from the palette tables.
 */

int View::getColor(int color)
{
	View *obj = this;

	while (obj != NULL)
	{
		int *pal = obj->getPalette();

		if (pal != NULL) color = pal[color];
		obj = obj->owner;
	}
	return color;
}

/*
 * Gets the view's data.
 */

void View::getData(void *buf)
{
	buf = NULL;			/* prevent warning messages */
}

/*
 * Returns the next available event in the event queue.  Returns EV_NONE
 * if no event is available.
 */

void View::getEvent(Event &event)
{
	if (owner != NULL) owner->getEvent(event);
}

/*
 * Returns, in the extent variable, the extent rectangle of the view.
 */

void View::getExtent(Rect &extent)
{
	extent.a.x = extent.a.y = 0;
	extent.b = size;
}

/*
 * Makes the view focused.
 */

int View::getFocus()
{
	if (owner != NULL && options & OF_SELECTABLE && state & SF_ACTIVE &&
		state & SF_VISIBLE)
	{
		if (options & OF_TOPSELECT) owner->makeFirst(this);
		owner->setCurrent(this);
		return 1;
	}
	return 0;
}

/*
 * Returns the height of the view.
 */

int View::getHeight()
{
	return size.y;
}

/*
 * Returns the view's help context.
 */

int View::getHelpCtx()
{
	return helpctx;
}

/*
 * Returns the view's palette address.
 */

int *View::getPalette()
{
	return NULL;
}

/*
 * Returns the size of the view.
 */

Point View::getSize()
{
	return size;
}

/*
 * Returns the state of the view.
 */

int View::getState()
{
	return state;
}

/*
 * Returns the width of the view.
 */

int View::getWidth()
{
	return size.x;
}

/*
 * This is the central method through which all event handling is
 * implemented.
 */

void View::handleEvent(Event &event)
{
	switch (event.what)
	{
	case EV_MOUSEDOWN:
		if (!(state & SF_FOCUSED))
		{
			if (getFocus() && !(options & OF_FIRSTCLICK))
			{
				clearEvent(event);
			}
		}
	}
}

/*
 * Hides the cursor.
 */

void View::hideCursor()
{
	setState(SF_CURSORVIS, 0);
}

/*
 * Returns non-zero if any part of the view is visible on the screen.
 */

int View::isExposed()
{
	return isexposed(this);
}

/*
 * Returns nonzero if the command is valid.
 */

int View::isValid(int command)
{
	command = 0;			/* prevent warning messages */
	return 1;
}

/*
 * Returns non-zero if the point is within the object.
 */

int View::localPointInside(Point pnt)
{
	return pnt.x >= 0 && pnt.x < size.x && pnt.y >= 0 && pnt.y < size.y;
}

/*
 * Generates a sound.
 */

void View::makeBeep()
{
	Event event;

	event.what = EV_COMMAND;
	event.command = CM_BEEP;
	putEvent(event);
}

/*
 * Converts the point coordinates from local (view) to global (screen) and
 * returns the result.
 */

void View::makeGlobal(Point &src, Point &dst)
{
	View *obj = this;

	dst = src;
	do
	{
		dst.x += obj->origin.x;
		dst.y += obj->origin.y;
		obj = obj->owner;
	}
	while (obj != NULL);
}

/*
 * Converts the point coordinates from global (screen) to local (view) and
 * returns the result.
 */

void View::makeLocal(Point &src, Point &dst)
{
	View *obj = this;

	dst = src;
	do
	{
		dst.x -= obj->origin.x;
		dst.y -= obj->origin.y;
		obj = obj->owner;
	}
	while (obj != NULL);
}

/*
 * Builds a message box and executes it.
 */

int View::messageBox(char *msg, void *info, int flags)
{
	static char caption[][9] =
	{
		"~Y~es", "~N~o", "O~K~", "~C~ancel"
	};
	static char name[][12] =
	{
		"Confirm", "Error", "Information", "Warning"
	};
	static int command[] =
	{
		CM_YES, CM_NO, CM_OK, CM_CANCEL
	};
	Button *button[MSGBOX_BUTTONS];
	Dialog *dlg;
	char str[MAX_WIDTH];
	int count, i, lenbut, lenstr, width, tmpflags, x;

	/* calculate button length */

	count = lenbut = 0;
	tmpflags = flags;
	for (i = 0; i < MSGBOX_BUTTONS; i++)
	{
		if (tmpflags & 0x100)
		{
			button[count] = new Button(Rect(0, 0, 9, 2),
				caption[i], command[i]);
			lenbut += button[count++]->getWidth() + 2;
		}
		tmpflags >>= 1;
	}

	/* calculate string length */

	sprintf(str, msg, info);
	lenstr = strlen(str);

	/* insert objects */

	width = MAX(MAX(lenbut + 4, lenstr + 4), strlen(msg) + 10);
	dlg = new Dialog(Rect(0, 0, width, 9), name[flags & 0xff]);
	x = (width - lenstr) / 2;
	dlg->insert(new StaticText(Rect(x, 3, x + lenstr, 4), str));
	x = (width - lenbut) / 2 + 1;
	for (i = 0; i < count; i++)
	{
		button[i]->moveTo(x, 6);
		dlg->insert(button[i]);
		x += button[i]->getWidth() + 2;
	}
	return program->execDialog(dlg, NULL);
}

/*
 * Returns nonzero if the global point is inside the view.
 */

int View::mouseInView(Point mouse)
{
	makeLocal(mouse, mouse);
	return localPointInside(mouse);
}

/*
 * Draws a text buffer in a video buffer.
 */

void View::moveBuf(Video *dst, unsigned char *src, char attr, int count)
{
	if (attr == 0) while (count-- > 0) dst++->split.code = *src++;
	else while (count-- > 0)
	{
		dst->split.code = *src++;
		dst++->split.attr = attr;
	}
}

/*
 * Draws a character some times in a video buffer.
 */

void View::moveChar(Video *buf, unsigned char code, char attr, int count)
{
	if (attr == 0) while (count-- > 0) buf++->split.code = code;
	else while (count-- > 0)
	{
		buf->split.code = code;
		buf++->split.attr = attr;
	}
}

/*
 * Draws a two-colored string in a video buffer.
 */

void View::moveCStr(Video *buf, char *str, int attrs)
{
	while (*str != '\0')
	{
		if (*str == '~') attrs = attrs >> 8 | (attrs & 0xff) << 8;
		else
		{
			buf->split.code = *str;
			buf++->split.attr = attrs & 0xff;
		}
		str++;
	}
}

/*
 * Draws a frame in the view buffer.
 */

void View::moveFrame(Video *buf, int wide, char attr)
{
	Video *end = buf + size.x * size.y - 1;
	int i;
	static unsigned char pieces[] =
	{
		0xda, 0xd9, 0xc4, 0xbf, 0xc0, 0xb3,
		0xc9, 0xbc, 0xcd, 0xbb, 0xc8, 0xba,
	};
	char *data = (char *) &pieces[wide * 6];

	/* write upper left and lower right corners */

	buf->split.code = data[0];
	end->split.code = data[1];
	buf->split.attr = end->split.attr = attr;
	buf++;
	end--;

	/* write upper and lower horizontal lines */

	for (i = size.x; i > 2; i--)
	{
		buf->split.code = end->split.code = data[2];
		buf->split.attr = end->split.attr = attr;
		buf++;
		end--;
	}

	/* write upper right and lower left corners */

	buf->split.code = data[3];
	end->split.code = data[4];
	buf->split.attr = end->split.attr = attr;

	/* write right and left vertical lines */

	for (i = size.y; i > 2; i--)
	{
		buf += size.x;
		end -= size.x;
		buf->split.code = end->split.code = data[5];
		buf->split.attr = end->split.attr = attr;
	}
}

/*
 * Draws a string in a video buffer.
 */

void View::moveStr(Video *buf, char *str, char attr)
{
	if (attr == 0) while (*str != '\0') buf++->split.code = *str++;
	else while (*str != '\0')
	{
		buf->split.code = *str++;
		buf++->split.attr = attr;
	}
}

/*
 * Moves the view in another position.
 */

void View::moveTo(int x, int y)
{
	origin.x = x;
	origin.y = y;
}

/*
 * Returns non-zero if the point is within the object in the owner's
 * coordinate system.
 */

int View::ownerPointInside(Point pnt)
{
	return pnt.x >= origin.x && pnt.x < origin.x + size.x &&
		pnt.y >= origin.y && pnt.y < origin.y + size.y;
}

/*
 * Puts the event given into the event queue, causing it to be the next event
 * returned.
 */

void View::putEvent(Event &event)
{
	if (owner != NULL) owner->putEvent(event);
}

/*
 * Sends a message to the specified object.
 */

void View::sendMessage(View *receiver, int what, int command)
{
	if (receiver != NULL)
	{
		Event event;

		event.what = what;
		event.command = command;
		event.sender = this;
		receiver->handleEvent(event);
	}
}

/*
 * Sets the bounding rectangle of the view to the value given by the bounds
 * parameter.  The origin field is set to bounds.a and the size field is set
 * to the difference between bounds.b and bounds.a.
 */

void View::setBounds(Rect bounds)
{
	origin = bounds.a;
	size.x = bounds.b.x - bounds.a.x;
	size.y = bounds.b.y - bounds.a.y;
}

/*
 * Moves the cursor.
 */

void View::setCursor(int x, int y)
{
	cursor.x = x;
	cursor.y = y;
	drawCursor();
}

/*
 * Sets the view's data.
 */

void View::setData(void *buf)
{
	buf = NULL;			/* prevent warning messages */
}

/*
 * Sets or clears a state flag in the state field.  The flag parameter
 * specifies the state flag to modify.
 */

void View::setState(int flag, int enable)
{
	if (enable) state |= flag; else state &= ~flag;
	switch (flag)
	{
	case SF_CURSORVIS:
		drawCursor();
		break;
	case SF_FOCUSED:
		drawView();
		sendMessage(owner, EV_BROADCAST, enable ? CM_GOTFOCUS :
			CM_LOSTFOCUS);
		break;
	case SF_ACTIVE:
	case SF_VISIBLE:
		if (owner != NULL)
		{
			if (!enable && state & SF_FOCUSED)
			{
				owner->forceFocusNext(0);
			}
			owner->redrawAll();
		}
	}
}

/*
 * Shows the cursor.
 */

void View::showCursor()
{
	setState(SF_CURSORVIS, 1);
}

/*
 * Returns the size limits of the view.
 */

void View::sizeLimits(Point &min, Point &max)
{
	min.x = min.y = 1;
	max.x = MAX_WIDTH;
	max.y = MAX_HEIGHT;
}

/*
 * Sends to all the objects a broadcast message to notify that the command
 * set is changed.
 */

void View::updateCommands()
{
	sendMessage(program, EV_BROADCAST, CM_COMMANDSETCHANGED);
}

/*
 * Writes the given buffer to the screen.  Starts at the coordinates x, y and
 * fills the region of width w and height h.  The buf parameter is an array of
 * words, each word containing a character in the low byte and an attribute in
 * the high byte.
 */

void View::writeBuf(int x, int y, int w, int h, Video *buf)
{
	writebuf(this, x, y, w, h, buf);
}

/*
 * Writes the line contained in the buffer buf to the screen, beginning at the
 * point x, y and within the rectangle defined by the width w and the height
 * h.  If h is greater than 1, the line will be repeated h times.  The buf
 * parameter is an array of words, each word containing a character in the low
 * byte and an attribute in the high byte.
 */

void View::writeLine(int x, int y, int w, int h, Video *buf)
{
	writeline(this, x, y, w, h, buf);
}
