/*
 * window.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h> 
#include <string.h>
#include "window.h"

/*
 * Frame constructor.
 */

Frame::Frame(Rect bounds, Window *awnd): View(bounds)
{
	DEBUG("new Frame at %p\n", this);
	growmode = GF_GROWHIX | GF_GROWHIY;
	wnd = awnd;
}

/*
 * Draws the frame on the screen.
 */

void Frame::draw()
{
	DrawBuf buf;

	moveChar(buf, ' ', getColor(C_FRAME + 3), size.x * size.y);
	if (wnd->state & SF_DRAGGING)
	{
		moveFrame(buf, 0, getColor(C_FRAME + 1));
	}
	else
	{
		Video *where = wnd->flags & WF_CLOSE ? buf + 6 : buf + 2;
		char *title = wnd->getTitle();
		int col;

		if (wnd->state & SF_FOCUSED)
		{
			int icon = getColor(C_FRAME + 2);

			col = getColor(C_FRAME);
			moveFrame(buf, 1, col);
			if (wnd->flags & WF_CLOSE) 
			{
				moveStr(buf + 2, "[", col);
				moveStr(buf + 3, "\xfe", icon);
				moveStr(buf + 4, "]", col);
			}
			if (wnd->flags & WF_RESIZE)
			{
				moveStr(buf + size.x * size.y - 1, "\xd9",
					icon);
			}
			if (wnd->flags & WF_ZOOM)
			{
				moveStr(buf + size.x - 5, "[", col);
				moveStr(buf + size.x - 4, wnd->state &
					SF_ZOOMED ? "\x12" : "\x18", icon);
				moveStr(buf + size.x - 3, "]", col);
			}
		}
		else
		{
			col = getColor(C_FRAME + 4);
			moveFrame(buf, 0, col);
		}
		if (title[0] != '\0')	/* draw title ? */
		{
			char str[MAX_WIDTH];

			sprintf(str, " %s ", title);
			moveStr(where, str, col);
		}
	}
	writeBuf(0, 0, size.x, size.y, buf);
}

/*
 * Handles the Frame's events.
 */

void Frame::handleEvent(Event &event)
{
	View::handleEvent(event);
	switch (event.what)
	{
	case EV_MOUSEDOWN:
		makeLocal(event.where, event.where);
		event.what = EV_COMMAND;
		if (event.where.y == 0)
		{
			if (wnd->flags & WF_CLOSE && event.where.x == 3)
			{
				event.command = CM_CLOSE;
				putEvent(event);
			}
			else if (wnd->flags & WF_ZOOM &&
				event.where.x == size.x - 4)
			{
				event.command = CM_ZOOM;
				putEvent(event);
			}
			else if (wnd->flags & WF_MOVE)
			{
				event.command = CM_MOVEMOUSE;
				putEvent(event);
			}
		}
		else if (wnd->flags & WF_RESIZE &&
			event.where.x == size.x - 1 &&
			event.where.y == size.y - 1)
		{
			event.command = CM_RESIZEMOUSE;
			putEvent(event);
		}
		clearEvent(event);
	}
}

/*
 * Window constructor.
 */

Window::Window(Rect bounds, char *atitle, View *(*pframe)(Rect, Window *)):
	Group(bounds, 1)
{
	DEBUG("new Window at %p\n", this);
	growmode = GF_GROWHIX | GF_GROWHIY;
	options |= OF_TOPSELECT;
	state |= SF_SHADOW;
	flags = WF_CLOSE | WF_MOVE | WF_RESIZE | WF_ZOOM;
	palette = WP_BLUEWINDOW;
	getBounds(zoomrect);
	strncpy2(title, atitle, sizeof(title));
	if (pframe != NULL)
	{
		getExtent(bounds);
		insert(frame = pframe(bounds, this));
	}
	else frame = NULL;
}

/*
 * Window destructor.
 */

Window::~Window()
{
	DEBUG("delete Window at %p\n", this);
	enableCommands(0);
}

/*
 * Closes the Window.
 */

void Window::doClose()
{
	if (isValid(CM_CLOSE)) delete this;
}

/*
 * Zoomes the Window.
 */

void Window::doZoom()
{
	if (state & SF_ZOOMED) changeBounds(zoomrect);
	else
	{
		Rect bounds;

		getBounds(zoomrect);
		owner->getExtent(bounds);
		changeBounds(bounds);
	}
	state ^= SF_ZOOMED;
	if (frame != NULL) frame->drawView();	/* update zoom icon */
	owner->redrawAll();
}

/*
 * Update the Window's commands.
 */

void Window::enableCommands(int enable)
{
	setCommandState(CM_NEXT, enable);
	setCommandState(CM_PREV, enable);
	setCommandState(CM_CLOSE, enable && flags & WF_CLOSE);
	setCommandState(CM_MOVE, enable && flags & WF_MOVE);
	setCommandState(CM_RESIZE, enable && flags & WF_RESIZE);
	setCommandState(CM_ZOOM, enable && flags & WF_ZOOM);
	updateCommands();
}

/*
 * Returns the Window's palette address.
 */

int *Window::getPalette()
{
	static int pal[][64] =
	{
		{
			16, 17, 18, 19, 20, 21, 22, 23,
			24, 25, 26, 27, 28, 29, 30, 31,
			32, 33, 34, 35, 36, 37, 38, 39,
			40, 41, 42, 43, 44, 45, 46, 47,
			48, 49, 50, 51, 52, 53, 54, 55,
			56, 57, 58, 59, 60, 61, 62, 63,
			64, 65, 66, 67, 68, 69, 70, 71,
			72, 73, 74, 75, 76, 77, 78, 79
		},
		{
			80, 81, 82, 83, 84, 85, 86, 87,
			88, 89, 90, 91, 92, 93, 94, 95,
			96, 97, 98, 99, 100, 101, 102, 103,
			104, 105, 106, 107, 108, 109, 110, 111,
			112, 113, 114, 115, 116, 117, 118, 119,
			120, 121, 122, 123, 124, 125, 126, 127,
			128, 129, 130, 131, 132, 133, 134, 135,
			136, 137, 138, 139, 140, 141, 142, 143
		}
	};
	return pal[palette];
}

/*
 * Returns the title of the Window.
 */

char *Window::getTitle()
{
	return title;
}

/*
 * Handles the Window's events.
 */

void Window::handleEvent(Event &event)
{
	Group::handleEvent(event);
	switch (event.what)
	{
	case EV_BROADCAST:
		switch (event.command)
		{
		case CM_UPDATETITLE:
			if (frame != NULL) frame->drawView();
		}
		break;
	case EV_COMMAND:
		switch (event.command)
		{
		case CM_CLOSE:
			if (flags & WF_CLOSE)
			{
				if (state & SF_MODAL)
				{
					event.what = EV_COMMAND;
					event.command = CM_CANCEL;
					putEvent(event);
				}
				else doClose();
			}
			clearEvent(event);
			break;
		case CM_MOVE:
		case CM_MOVEMOUSE:
			if (flags & WF_MOVE) moveOrResize(event);
			break;
		case CM_RESIZE:
		case CM_RESIZEMOUSE:
			if (flags & WF_RESIZE) moveOrResize(event);
			break;
		case CM_ZOOM:
			if (flags & WF_ZOOM) doZoom();
			clearEvent(event);
		}
		break;
	case EV_KEYDOWN:
		switch (event.keycode)
		{
		case KC_TAB:
			focusNext(1);
			clearEvent(event);
			break;
		case KC_METATAB:
			focusNext(0);
			clearEvent(event);
		}
	}
}

/*
 * Creates a standard Frame object.
 */

View *Window::initFrame(Rect bounds, Window *wnd)
{
	return new Frame(bounds, wnd);
}

/*
 * Moves or resizes the window.
 */

void Window::moveOrResize(Event &offset)
{
	Event event;
	Point delta, max = { owner->getWidth() - 1, owner->getHeight() - 1 };
	Rect bounds;
	int usemouse, quit = 0;

	getBounds(bounds);
	setState(SF_DRAGGING, 1);
	usemouse = offset.command == CM_MOVEMOUSE || offset.command ==
		CM_RESIZEMOUSE;
	do
	{
		getEvent(event);
		if (usemouse && event.what == EV_MOUSEMOVE)
		{
			Point cur;

			owner->makeLocal(event.where, cur);
			cur.x = RANGE(cur.x, 0, max.x);
			cur.y = RANGE(cur.y, 0, max.y);
			if (offset.command == CM_MOVEMOUSE)
			{
				origin.x = cur.x - offset.where.x;
				origin.y = cur.y - offset.where.y;
			}
			else
			{
				delta.x = cur.x - bounds.b.x + 1;
				delta.y = cur.y - bounds.b.y + 1;
				calcBounds(bounds, delta);
				changeBounds(bounds);
			}
			owner->redrawAll();
		}
		else if (!usemouse && event.what == EV_KEYDOWN)
			switch (offset.command)
		{
		case CM_MOVE:
			switch (event.keycode)
			{
			case KC_ARROWLEFT:
				origin.x--;
				break;
			case KC_ARROWUP:
				origin.y--;
				break;
			case KC_ARROWRIGHT:
				origin.x++;
				break;
			case KC_ARROWDOWN:
				origin.y++;
				break;
			default:
				quit = 1;
			}
			if (!quit)
			{
				origin.x = RANGE(origin.x,
					bounds.a.x - bounds.b.x + 1, max.x);
				origin.y = RANGE(origin.y, 0, max.y);
				owner->redrawAll();
			}
			break;
		case CM_RESIZE:
			delta.x = delta.y = 0;
			switch (event.keycode)
			{
			case KC_ARROWLEFT:
				if (bounds.b.x > 1) delta.x = -1;
				break;
			case KC_ARROWUP:
				if (bounds.b.y > 1) delta.y = -1;
				break;
			case KC_ARROWRIGHT:
				if (bounds.b.x <= max.x) delta.x = 1;
				break;
			case KC_ARROWDOWN:
				if (bounds.b.y <= max.y) delta.y = 1;
				break;
			default:
				quit = 1;
			}
			if (delta.x != 0 || delta.y != 0)
			{
				calcBounds(bounds, delta);
				changeBounds(bounds);
				owner->redrawAll();
			}
		}
	}
	while (!(usemouse && event.what == EV_MOUSEUP || quit));
	state &= ~SF_ZOOMED;
	getBounds(zoomrect);
	setState(SF_DRAGGING, 0);
	clearEvent(offset);
}

/*
 * Changes the Window's state.
 */

void Window::setState(int flag, int enable)
{
	if ((flag == SF_FOCUSED || flag == SF_VISIBLE) && !enable)
	{
		/* disable Window commands because Window is invisible */
		/* or it losts the focus */

		enableCommands(0);
	}
	Group::setState(flag, enable);
	switch (flag)
	{
	case SF_DRAGGING:
		if (frame != NULL) frame->drawView();
		break;
	case SF_FOCUSED:
	case SF_VISIBLE:
		if (state & SF_FOCUSED && state & SF_VISIBLE)
		{
			/* enable Window commands because Window is */
			/* visible and it has the focus */

			enableCommands(1);
		}
	}
}

/*
 * Returns the size limits of the Window.
 */

void Window::sizeLimits(Point &min, Point &max)
{
	Group::sizeLimits(min, max);
	min.x = strlen(getTitle()) + 6;
	if (flags & WF_CLOSE) min.x += 4;
	if (flags & WF_ZOOM) min.x += 4;
	min.y = 2;
}

/*
 * Dialog constructor.
 */

Dialog::Dialog(Rect bounds, char *atitle): Window(bounds, atitle,
	Window::initFrame)
{
	DEBUG("new Dialog at %p\n", this);
	options |= OF_CENTERX | OF_CENTERY;
	flags = WF_CLOSE | WF_MOVE;
	palette = WP_GRAYWINDOW;
}

/*
 * Handles the Dialog's events.
 */

void Dialog::handleEvent(Event &event)
{
	Window::handleEvent(event);
	switch (event.what)
	{
	case EV_COMMAND:
		switch (event.command)
		{
		case CM_CANCEL:
		case CM_NO:
		case CM_OK:
		case CM_YES:
			if (state & SF_MODAL)
			{
				endModal(event.command);
				clearEvent(event);
			}
		}
		break;
	case EV_KEYDOWN:
		switch (event.keycode)
		{
		case KC_ENTER:
			event.what = EV_BROADCAST;
			event.command = CM_DEFAULT;
			putEvent(event);
			clearEvent(event);
			break;
		case KC_ESC:
			event.what = EV_COMMAND;
			event.command = CM_CANCEL;
			putEvent(event);
			clearEvent(event);
		}
	}
}
