// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of Common C++.
// 
// The exception is that, if you link the Common C++ library with other files
// to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the Common C++ library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name Common C++.  If you copy code from other releases into a copy of
// Common C++, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for Common C++, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

// needed for GNU/LINUX glibc otherwise pread/pwrite wont work

#ifdef	__linux__
#ifndef	_XOPEN_SOURCE
#define	_XOPEN_SOURCE 500
#endif
/*
 * on old glibc's, this has to be
 * defined explicitly
 */
#ifndef _XOPEN_SOURCE_EXTENDED
#define _XOPEN_SOURCE_EXTENDED
#endif
#endif

#include "config.h"
#include "macros.h"
#include "thread.h"
#include "file.h"
#include <stdio.h>
#include <sys/stat.h>
#include <errno.h>

#ifdef	HAVE_SYS_FILE_H
#include <sys/file.h>
#endif

#ifdef	COMMON_AIX_FIXES
#undef	LOCK_EX
#undef	LOCK_SH
#endif

#ifndef	F_LOCK

enum
{
	F_ULOCK = 1,
	F_LOCK,
	F_TLOCK,
	F_TEST
};

static	int lockf(int fd, int cmd, long len)
{
	struct	flock lck;

	lck.l_start = 0l;
	lck.l_whence = SEEK_CUR;
	lck.l_len = len;

	switch(cmd)
	{
	case F_ULOCK:
		lck.l_type = F_UNLCK;
		return fcntl(fd, F_SETLK, &lck);
	case F_LOCK:
		lck.l_type = F_WRLCK;
		return fcntl(fd, F_SETLKW, &lck);
	case F_TLOCK:
		lck.l_type = F_WRLCK;
		return fcntl(fd, F_SETLK, &lck);
	case F_TEST:
		lck.l_type = F_WRLCK;
		fcntl(fd, F_GETLK, &lck);
		if(lck.l_type == F_UNLCK)
			return 0;
		return -1;
	}
}
		
#endif

#if 0
/*
 * not used anymore ? (hen)
 */
static const char *clearfile(const char *pathname)
{
	remove(pathname);
	return pathname;
}

static const char *clearfifo(const char *pathname, int mode)
{
	remove(pathname);
	mkfifo(pathname, mode);
	return pathname;
}
#endif

RandomFile::RandomFile() {
	fd = -1;
	flags.count = 0;
	flags.thrown = flags.initial = flags.immediate = flags.temp = false;
}

RandomFile::RandomFile(const RandomFile &rf) : Mutex()
{
	if(rf.fd > -1)
		fd = dup(rf.fd);
	else
		fd = -1;

	flags = rf.flags;
	flags.count = 0;

	if(rf.pathname)
	{
		pathname = new char[strlen(rf.pathname) + 1];
		strcpy(pathname, rf.pathname);
	}
	else
		pathname = NULL;
}

void RandomFile::Final(void)
{
	if(fd > -1)
	{
		close(fd);
		if(flags.temp)
			remove(pathname);
	}

	if(pathname)
	{
		delete[] pathname;
		pathname = NULL;
	}

	fd = -1;
	flags.count = 0;
	flags.initial = false;
}

fileerror_t RandomFile::Error(fileerror_t id, char *str)
{
	errstr = str;
	errid = id;
	if(!flags.thrown)
	{
		flags.thrown = true;
		if(getException() == THROW_OBJECT)
			throw(this);
	}
	return id;
}

bool RandomFile::Initial(void)
{
	bool init;
	int mode;

	if(fd < 0)
		return false;

	EnterMutex();
	init = flags.initial;
	flags.initial = false;

	if(!init)
	{
		LeaveMutex();
		return false;
	}

	mode = (int)Initialize();	
	if(!mode)
	{
		close(fd);
		fd = -1;
		if(pathname)
			remove(pathname);
		Error(FILE_INIT_FAILED);
	}
	else
		fchmod(fd, mode);

	LeaveMutex();
	return init;
}

fileerror_t RandomFile::setCompletion(filecomplete_t mode)
{
	long flag = fcntl(fd, F_GETFL);

	if(fd < 0)
		return FILE_NOT_OPENED;

	flags.immediate = false;
#ifdef O_SYNC
	flag &= ~(O_SYNC | O_NDELAY);
#else
	flag &= ~O_NDELAY;
#endif
	switch(mode)
	{
	case FILE_COMPLETION_IMMEDIATE:
#ifdef O_SYNC
		flag |= O_SYNC;
#endif
		flags.immediate = true;
		break;
		
	case FILE_COMPLETION_DELAYED:
		flag |= O_NDELAY;

	//FILE_COMPLETION_DEFERRED: ? (hen)
	}
	fcntl(fd, F_SETFL, flag);
	return FILE_SUCCESS;
}

off_t RandomFile::getCapacity(void)
{
	off_t eof;
	off_t pos;

	if(fd < 0)
		return 0;

	EnterMutex();
	pos = lseek(fd, 0l, SEEK_CUR);
	eof = lseek(fd, 0l, SEEK_END);
	lseek(fd, pos, SEEK_SET);
	LeaveMutex();
	return eof;
}

bool RandomFile::operator!(void)
{
	if(fd < 0)
		return true;

	return false;
}

ThreadFile::ThreadFile(const char *path) : RandomFile()
{
	first = NULL;
	Open(path);
}

ThreadFile::~ThreadFile()
{
	Final();
	fcb_t *next;
	while(first)
	{
		next = first->next;
		delete first;
		first = next;
	}
}

fileerror_t ThreadFile::Open(const char *path)
{
	int len;

	if(fd > -1)
		Final();

	if(path != pathname)
	{
		len = strlen(path);
		pathname = new char[len + 1];		
		strcpy(pathname, path);
	}

	flags.initial = false;
	fd = open(pathname, O_RDWR);
	if(fd < 0)
	{
		flags.initial = true;
		fd = creat(pathname, (int)FILE_ATTR_PRIVATE);
	}
	if(fd < 0)
		return Error(FILE_OPEN_FAILED);

#ifdef	LOCK_EX
	if(flock(fd, LOCK_EX | LOCK_NB))
	{
		close(fd);
		fd = -1;
		return Error(FILE_OPEN_INUSE);
	}
#endif
	return FILE_SUCCESS;
}

fcb_t *ThreadFile::getFCB(void)
{
	fcb_t *fcb = (fcb_t *)state.getKey();

	if(!fcb)
	{
		fcb = new fcb_t;
		fcb->next = first;
		first = fcb;
		fcb->address = NULL;
		fcb->len = 0;
		fcb->pos = 0;
		state.setKey(fcb);
	}
	return fcb;
}

fileerror_t ThreadFile::Fetch(caddr_t address, size_t len, off_t pos)
{
	fcb_t *fcb = getFCB();
	int io;

	if(fd < 0)
		return FILE_NOT_OPENED;

	if(address)
		fcb->address = address;

	if(len)
		fcb->len = len;

	if(pos != -1)
		fcb->pos = pos;

#ifdef	HAVE_PREAD_PWRITE
	io = ::pread(fd, fcb->address, fcb->len, fcb->pos);
#else
	EnterMutex();
	lseek(fd, fcb->pos, SEEK_SET);
	io = ::read(fd, fcb->address, fcb->len);
	LeaveMutex();
#endif

	if((size_t) io == len)
		return FILE_SUCCESS;

	if(io > -1)
		return FILE_READ_INCOMPLETE;

	switch(errno)
	{
	case EINTR:
		return FILE_READ_INTERRUPTED;
	default:
		return FILE_READ_FAILURE;
	}
}

fileerror_t ThreadFile::Update(caddr_t address, size_t len, off_t pos)
{
	fcb_t *fcb = getFCB();
	int io;

	if(fd < 0)
		return FILE_NOT_OPENED;

	if(address)
		fcb->address = address;

	if(len)
		fcb->len = len;

	if(pos != -1)
		fcb->pos = pos;

#ifdef	HAVE_PREAD_PWRITE
	io = ::pwrite(fd, fcb->address, fcb->len, fcb->pos);
#else
	EnterMutex();
	lseek(fd, fcb->pos, SEEK_SET);
	io = ::write(fd, fcb->address, fcb->len);
	LeaveMutex();
#endif

	if((size_t) io == len)
		return FILE_SUCCESS;

	if(io > -1)
		return FILE_WRITE_INCOMPLETE;

	switch(errno)
	{
	case EINTR:
		return FILE_WRITE_INTERRUPTED;
	default:
		return FILE_WRITE_FAILURE;
	}
}

fileerror_t ThreadFile::Append(caddr_t address, size_t len)
{
	fcb_t *fcb = getFCB();
	int io;

	if(fd < 0)
		return FILE_NOT_OPENED;

	if(address)
		fcb->address = address;

	if(len)
		fcb->len = len;

	EnterMutex();
	fcb->pos = lseek(fd, 0l, SEEK_END);
	io = ::write(fd, fcb->address, fcb->len);
	LeaveMutex();

	if((size_t) io == len)
		return FILE_SUCCESS;

	if(io > -1)
		return FILE_WRITE_INCOMPLETE;

	switch(errno)
	{
	case EINTR:
		return FILE_WRITE_INTERRUPTED;
	default:
		return FILE_WRITE_FAILURE;
	}
}

off_t ThreadFile::getPosition(void)
{
	fcb_t *fcb = getFCB();

	return fcb->pos;
}

bool ThreadFile::operator++(void)
{
	off_t eof;
	fcb_t *fcb = getFCB();

	fcb->pos += fcb->len;
	EnterMutex();
	eof = lseek(fd, 0l, SEEK_END);
	LeaveMutex();

	if(fcb->pos >= eof)
	{
		fcb->pos = eof;
		return true;
	}
	return false;
}

bool ThreadFile::operator--(void)
{
	fcb_t *fcb = getFCB();

	fcb->pos -= fcb->len;
	if(fcb->pos <= 0)
	{
		fcb->pos = 0;
		return true;
	}
	return false;
}

SharedFile::SharedFile(const char *path) :
RandomFile()
{
	fcb.address = NULL;
	fcb.len = 0;
	fcb.pos = 0;
	Open(path);
}

SharedFile::SharedFile(const SharedFile &sh) :
RandomFile(sh)
{
}
	
SharedFile::~SharedFile()
{
	Final();
}

fileerror_t SharedFile::Open(const char *path)
{
	int len;

	if(fd > -1)
		Final();

	if(path != pathname)
	{
		len = strlen(path);
		pathname = new char[len + 1];		
		strcpy(pathname, path);
	}

	flags.initial = false;
	fd = open(pathname, O_RDWR);
	if(fd < 0)
	{
		flags.initial = true;
		fd = creat(pathname, (int)FILE_ATTR_PRIVATE);
	}
	if(fd < 0)
		return Error(FILE_OPEN_FAILED);

#ifdef	LOCK_SH
	if(flock(fd, LOCK_SH | LOCK_NB))
	{
		close(fd);
		fd = -1;
		return Error(FILE_OPEN_INUSE);
	}
#endif
	return FILE_SUCCESS;
}

fileerror_t SharedFile::Fetch(caddr_t address, size_t len, off_t pos)
{
	int io;

	if(fd < 0)
		return FILE_NOT_OPENED;

	EnterMutex();
	if(address)
		fcb.address = address;

	if(len)
		fcb.len = len;

	if(pos != -1)
		fcb.pos = pos;

	lseek(fd, fcb.pos, SEEK_SET);
	lockf(fd, F_LOCK, fcb.len);
	io = ::read(fd, fcb.address, fcb.len);
	LeaveMutex();

	if((size_t) io == len)
		return FILE_SUCCESS;

	if(io > -1)
		return FILE_READ_INCOMPLETE;

	switch(errno)
	{
	case EINTR:
		return FILE_READ_INTERRUPTED;
	default:
		return FILE_READ_FAILURE;
	}
}

fileerror_t SharedFile::Clear(size_t len, off_t pos)
{
	if(fd < 0)
		return FILE_NOT_OPENED;

	EnterMutex();
	if(len)
		fcb.len = len;

	if(pos != -1)
		fcb.pos = pos;

	lseek(fd, fcb.pos, SEEK_SET);
	lockf(fd, F_ULOCK, fcb.len);
	LeaveMutex();
	return FILE_SUCCESS;
}

fileerror_t SharedFile::Update(caddr_t address, size_t len, off_t pos)
{
	int io;

	if(fd < 0)
		return FILE_NOT_OPENED;

	EnterMutex();
	if(address)
		fcb.address = address;

	if(len)
		fcb.len = len;

	if(pos != -1)
		fcb.pos = pos;

	lseek(fd, fcb.pos, SEEK_SET);
	io = ::write(fd, fcb.address, fcb.len);
	lockf(fd, F_ULOCK, fcb.len);
	LeaveMutex();

	if((size_t) io == len)
		return FILE_SUCCESS;

	if(io > -1)
		return FILE_WRITE_INCOMPLETE;

	switch(errno)
	{
	case EINTR:
		return FILE_WRITE_INTERRUPTED;
	default:
		return FILE_WRITE_FAILURE;
	}
}

fileerror_t SharedFile::Append(caddr_t address, size_t len)
{
	int io;

	if(fd < 0)
		return FILE_NOT_OPENED;

	EnterMutex();
	if(address)
		fcb.address = address;

	if(len)
		fcb.len = len;

	fcb.pos = lseek(fd, 0l, SEEK_END);
	lockf(fd, F_LOCK, -1);
	fcb.pos = lseek(fd, 0l, SEEK_END);
	io = ::write(fd, fcb.address, fcb.len);
	lseek(fd, fcb.pos, SEEK_SET);
	lockf(fd, F_ULOCK, -1);
	LeaveMutex();

	if((size_t) io == len)
		return FILE_SUCCESS;

	if(io > -1)
		return FILE_WRITE_INCOMPLETE;

	switch(errno)
	{
	case EINTR:
		return FILE_WRITE_INTERRUPTED;
	default:
		return FILE_WRITE_FAILURE;
	}
}

off_t SharedFile::getPosition(void)
{
	return fcb.pos;
}

bool SharedFile::operator++(void)
{
	off_t eof;

	EnterMutex();
	fcb.pos += fcb.len;
	eof = lseek(fd, 0l, SEEK_END);

	if(fcb.pos >= eof)
	{
		fcb.pos = eof;
		LeaveMutex();
		return true;
	}
	LeaveMutex();
	return false;
}

bool SharedFile::operator--(void)
{
	EnterMutex();
	fcb.pos -= fcb.len;
	if(fcb.pos <= 0)
	{
		fcb.pos = 0;
		LeaveMutex();
		return true;
	}
	LeaveMutex();
	return false;
}

MappedFile::MappedFile(const char *fname, fileaccess_t mode) :
RandomFile()
{
	fd = open(fname, (int)mode);
	if(fd < 0 && mode != FILE_ACCESS_READONLY)
		fd = creat(fname, (int)FILE_ATTR_PRIVATE);

	if(fd < 0)
	{
		Error(FILE_OPEN_FAILED);
		return;
	}

	switch(mode)
	{
	case O_RDONLY:
		prot = PROT_READ;
		break;
	case O_WRONLY:
		prot = PROT_WRITE;
		break;
	default:
		prot = PROT_READ | PROT_WRITE;
	}
}

MappedFile::MappedFile(const char *fname, pos_t pos, size_t len, fileaccess_t mode) :
RandomFile()
{
	fd = open(fname, (int)mode);
	if(fd < 0)
	{
		Error(FILE_OPEN_FAILED);
		return;
	}

	switch(mode)
	{
	case O_RDONLY:
		prot = PROT_READ;
		break;
	case O_WRONLY:
		prot = PROT_WRITE;
		break;
	default:
		prot = PROT_READ | PROT_WRITE;
	}

	EnterMutex();
	lseek(fd, pos + len, SEEK_SET);
	fcb.address = (caddr_t)mmap(NULL, len, prot, MAP_SHARED, fd, pos);
	fcb.len = len;
	fcb.pos = pos;
	LeaveMutex();
	if((caddr_t)(fcb.address) == (caddr_t)(MAP_FAILED))
	{
		close(fd);
		fd = -1;
		Error(FILE_MAP_FAILED);
	}
} 
	
MappedFile::~MappedFile()
{
	Final();
}

void MappedFile::Release(caddr_t address, size_t len)
{
	EnterMutex();
	if(address)
		fcb.address = address;

	if(len)
		fcb.len = len;

	munmap(fcb.address, fcb.len);
	LeaveMutex();
}

caddr_t MappedFile::Fetch(off_t pos, size_t len)
{
	EnterMutex();
	fcb.len = len;
	fcb.pos = pos;	
	lseek(fd, fcb.pos + len, SEEK_SET);
	fcb.address = (caddr_t)mmap(NULL, len, prot, MAP_SHARED, fd, pos);
	LeaveMutex();
	return fcb.address;
}

void MappedFile::Update(size_t offset, size_t len)
{
	int mode = MS_ASYNC;
	caddr_t address;

	if(flags.immediate)
		mode = MS_SYNC;

	EnterMutex();
	address = fcb.address;
	address += offset;
	if(!len)
		len = fcb.len;
	LeaveMutex();
	msync(address, len, mode);
}

void MappedFile::Update(caddr_t address, size_t len)
{
	int mode = MS_ASYNC;
	if(flags.immediate)
		mode = MS_SYNC;

	msync(address, len, mode);
}

bool isDir(const char *path)
{
	struct stat ino;

	if(stat(path, &ino))
		return false;

	if(S_ISDIR(ino.st_mode))
		return true;

	return false;
}

bool isFile(const char *path)
{
	struct stat ino;

	if(stat(path, &ino))
		return false;

	if(S_ISREG(ino.st_mode))
		return true;

	return false;
}

bool isDevice(const char *path)
{
	struct stat ino;

	if(stat(path, &ino))
		return false;

	if(S_ISCHR(ino.st_mode))
		return true;

	return false;
}

bool canAccess(const char *path)
{
	if(!access(path, R_OK))
		return true;

	return false;
}

bool canModify(const char *path)
{
	if(!access(path, W_OK | R_OK))
		return true;

	return false;
}

/** EMACS **
 * Local variables:
 * mode: c++
 * c-basic-offset: 8
 * End:
 */
