// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of Common C++.
// 
// The exception is that, if you link the Common C++ library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the Common C++ library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name Common C++.  If you copy code from other releases into a copy of
// Common C++, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for Common C++, it is your
// choice whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#ifndef	__CCXX_FILE_H__
#define	__CCXX_FILE_H__

#ifndef	__CCXX_THREAD_H__
#include <thread.h>
#endif

#include <direct.h>

typedef	unsigned long pos_t;
typedef LONG off_t;

typedef struct _fcb
{
	struct _fcb *next;
	void *address;
	DWORD len;
	LONG pos;
} fcb_t;

typedef enum{
	FILE_ATTR_INVALID=0,
	FILE_ATTR_PRIVATE,
	FILE_ATTR_GROUP,
	FILE_ATTR_PUBLIC
} fileattr_t;

typedef enum{
	FILE_ACCESS_READONLY = GENERIC_READ,
	FILE_ACCESS_WRITEONLY = GENERIC_WRITE,
	FILE_ACCESS_READWRITE = GENERIC_READ | GENERIC_WRITE
} fileaccess_t;

typedef enum{
	FILE_SUCCESS = 0,
	FILE_NOT_OPENED,
	FILE_MAP_FAILED,
	FILE_INIT_FAILED,
	FILE_OPEN_FAILED,
	FILE_OPEN_INUSE,
	FILE_READ_INTERRUPTED,
	FILE_READ_INCOMPLETE,
	FILE_READ_FAILURE,
	FILE_WRITE_INTERRUPTED,
	FILE_WRITE_INCOMPLETE,
	FILE_WRITE_FAILURE,
	FILE_EXTENDED_ERROR
} fileerror_t;

class __EXPORT Dir
{
private:
	char path[256];
	HANDLE hDir;
	WIN32_FIND_DATA data;
	char *name;

public:
	Dir(const char *name);
	~Dir();
	char *getName(void);
	bool operator!()
		{return hDir != INVALID_HANDLE_VALUE;};
};

class __EXPORT RandomFile : public Mutex
{
private:
	fileerror_t errid;
	char *errstr;

protected:
	HANDLE hFile;
	char *filepath;
	struct
	{
		unsigned count : 16;
		bool initial : 1;
		bool thrown : 1;
		bool temp : 1;
	} flags;
	RandomFile();
	RandomFile(const RandomFile &rf);
	virtual fileattr_t Initialize(void)
		{return FILE_ATTR_PUBLIC;};
	void Final(void);

public:
	virtual ~RandomFile()
		{Final();};

	fileerror_t Error(fileerror_t id, char *str = NULL);
	bool Initial(void);
	off_t getCapacity(void);

	virtual fileerror_t Restart(void)
		{return FILE_OPEN_FAILED;};

	inline fileerror_t getErrorNumber(void)
		{return errid;};

	inline char *getErrorString(void)
		{return errstr;};

	inline fileerror_t Error(char *str)
		{return Error(FILE_EXTENDED_ERROR, str);};
	
	inline void setError(bool enable)
		{flags.thrown = !enable;};

	inline void setTemporary(bool enable)
		{flags.temp = enable;};

	bool operator!(void);
};	

class __EXPORT ThreadFile : public RandomFile
{
private:
	ThreadKey state;
	fcb_t *first;
	fcb_t *getFCB(void);

	fileerror_t Open(const char *name);

public:
	ThreadFile(const char *name);
	~ThreadFile();
	
	fileerror_t Fetch(void *address = NULL, DWORD len = 0, LONG pos = -1);
	fileerror_t Update(void *address = NULL, DWORD len = 0, LONG pos = -1);
	fileerror_t Append(void *address = NULL, DWORD len = 0);
	LONG getPosition(void);

	fileerror_t Restart(void)
		{return Open(filepath);};

	bool operator++(void);
	bool operator--(void);
};

class __EXPORT SharedFile : public RandomFile
{
private:
	fcb_t fcb;

	fileerror_t Open(const char *name);

public:
	SharedFile(const char *name);
	SharedFile(const SharedFile &sh);
	~SharedFile();
	
	fileerror_t Fetch(void *address = NULL, DWORD len = 0, LONG pos = -1);
	fileerror_t Update(void *address = NULL, DWORD len = 0, LONG pos = -1);
	fileerror_t Append(void *address = NULL, DWORD len = 0);
	fileerror_t Clear(DWORD len = 0, LONG pos = -1);
	LONG getPosition(void);

	fileerror_t Restart(void)
		{return Open(filepath);};

	bool operator++(void);
	bool operator--(void);
};

class __EXPORT DSO
{
private:
	static Mutex mutex;
	static DSO *first;
	static DSO *last;
	DSO *next, *prev;
	char *err;
	HINSTANCE hImage;

public:
	DSO(char *filename);
	~DSO();

	inline char *getError(void)
		{return err;};

	void *operator[](const char *sym);

	friend void dynunload(void);
};
	
class __EXPORT Pipe 
{
private:
	HANDLE reader;
	HANDLE writer;
	int objsize, objcount;

public:
	Pipe(int size, int count);
	Pipe(const Pipe &p);
	~Pipe();

	int Receive(void *addr);
	int Send(void *addr);

	Pipe &operator=(const Pipe &p);
};

bool isDir(const char *path);
bool isFile(const char *path);
bool canAccess(const char *path);
bool canModify(const char *path);

#endif

/** EMACS **
 * Local variables:
 * mode: c++
 * c-basic-offset: 8
 * End:
 */
