/****************************************************************************/
/*   xmObject.h - Xm++ main header file - (03/30/1992) 03/31/1993           */
/****************************************************************************/
/*                                                                          */
/*   Copyright (c) 1992, 1993, 1994 Bernhard Strassl                        */
/*       Vienna User Interface Group                                        */
/*       Institute for Applied Computer Science and Information Systems     */
/*       University of Vienna, Austria                                      */
/*                                                                          */
/*   See file COPYRIGHT in this directory for details. If this file is      */
/*   missing please mail to xmplus@ani.univie.ac.at                         */
/****************************************************************************/

#ifndef XmObject_included
#define XmObject_included 1

#include <xmconfig.h>

#include <string.h>	
#include <time.h>	
#include <iostream.h>		// for debugging...

#ifdef XMPP_DVLP

#include <Xlib.h>
#include <Xos.h> 
#include <keysym.h>
#include <Xutil.h>
#include <Intrinsic.h> 
#include <Xatom.h>
#include <StringDefs.h>
#include <Core.h>
#include <Shell.h>

#ifdef XAW
typedef char* XmString;
#else
#include <Xm.h>
#define MOTIF 1
#endif

typedef String XString;

#else

#if defined(ASCII) || defined(TCL) || defined(RCI)
#ifdef ASCII
#include "CwDefs.h"
#else
#ifdef TCL
#include "TclDefs.h"
#else
#ifndef WIN31
#include "RciDefs.h"
#else
#include "Rci_Defs.h"
#endif
#endif
#endif

#else

#ifdef XAW
#include "XawDefs.h"
#else
#include "MotifDefs.h"
#define MOTIF 1
#endif

#endif

typedef char* XString;

#endif

#include "xmStyles.h"

#ifndef NULL
#define NULL 0
#endif

#define MAGIC_OBJECT_CONST 123456

#ifdef XmObject
#undef XmObject
#endif

#undef TRUE
#undef FALSE

#if !defined( BOOL_DEFINED ) && !defined( _BOOL_DEFINED )
#if defined( __GNUC__ ) && __GNUC_MINOR__ >= 6
#define TRUE true
#define FALSE false
#else
#undef true
#undef false
typedef enum bool { TRUE = 1, true = 1, FALSE = 0, false = 0 };
#endif
#define _bool_h
#define BOOL_DEFINED
#define _BOOL_DEFINED
#endif

class XmObject;

extern void translateWindowStyles(xmStyle, long&, long&);

class XmApp
{
	Display* display;
	Screen* screen;
	XrmDatabase rscDb;
	int appArgc;
	char** appArgv;
	XtAppContext appContext;
	XmObject* appShell;
	XmObject** toplevelShells;
	int numShells;
	unsigned char unitType;
	bool useWindows;
	bool active;

	void error(char*);
friend void itself();			// keep gcc quiet...
	~XmApp();
public:
	XmApp(int, char**);

	void setScreenUnitType(unsigned char);
	unsigned char getScreenUnitType()	{ return(unitType); }

	void initialize();		// to be implemented by user...

	void run();
	void quit(bool = FALSE);

	char* getName()				{ return(appArgv[0]); }
	Display* getDisplay()			{ return(display); }
	Screen* getScreen()			{ return(screen); }
	XrmDatabase getResourceDatabase()	{ return(rscDb); }
	XtAppContext getAppContext()		{ return(appContext); }
	char* getCmdArg(int);
	XmObject* getMainObject()		{ return(appShell); }
	bool setMainObject(XmObject*);
	XmObject* findObject(char*);
	Widget getMainWindow();
	Widget newMainWindow(XmObject*, char* = NULL, int = 0, int = 0, xmStyle = 0);
	void removeMainWindow(XmObject*);
	bool forceWindows(bool w)		{ bool prev = useWindows; useWindows = w; return(prev); }
	bool windowsForced()			{ return(useWindows); }
	bool isXok()				{ return(display ? TRUE : FALSE); }
	bool isActive()				{ return(active); }

	bool setCursor(char*);
	bool setCursor(int);

	char* xmPlusPlusVersion()		{ return("0.61"); }
};

extern char* XmImageDir;
extern XmApp* App;

#define display App->getDisplay()
#define screen App->getScreen()
#define mainWindow App->getMainWindow()

#ifndef NO_CPP_CONCAT
#ifndef XAW
#define cb(a) XmN##a##Callback
#else
#define cb(a) XtN##a##Callback
#endif
#endif

extern Pixmap _createXmPixmap(char*, int, int);
#define getRuntimeImage(n) _createXmPixmap(n, 0, 0)
#ifndef NO_CPP_CONCAT
#define getCompiledImage(n) _createXmPixmap(n##_bits, n##_width, n##_height)
#else
#define getCompiledImage(n, w, h) _createXmPixmap(n, w, h)
#endif

class XmRootClass { };

typedef void (XmRootClass::*cbProc)(void*);

#ifndef NO_PROC_CAST
#define CBK(c, m) ((cbProc )& c :: m)
#else
struct ciClientData;
template<class T> class _CB
{
	cbProc p;
public:
	_CB(void ( T ::*f)(void*))		{ memcpy((void*)&p, (void*)&f, sizeof(cbProc)); }
	_CB(void ( T ::*f)(XmRootClass*))	{ memcpy((void*)&p, (void*)&f, sizeof(cbProc)); }
	_CB(void ( T ::*f)(char*))		{ memcpy((void*)&p, (void*)&f, sizeof(cbProc)); }
	_CB(void ( T ::*f)(ciClientData*))	{ memcpy((void*)&p, (void*)&f, sizeof(cbProc)); }
	operator cbProc ()	{ return(p); }
};
#define CBK(c, m) (_CB< c >(& c :: m))
#endif

#ifdef NO_PROC_UEXPR
#ifndef NO_PROC_CAST
#define validProc(p) (p != (cbProc )NULL)
#else
#define validProc(p) (p != NULL)
#endif
#else
#define validProc(p) p
#endif

class XmManager;

struct cb_data
{
	cbProc callback;
	XmManager* object;
	void* client_data;
};

#define CB_NO_DATA ((void* )0L)
#define CB_XM_DATA ((void* )1L)
#define CB_XM_HANDLE ((void* )2L)
#define CB_OBJ_NAME ((void* )3L)
#define CB_OBJ_PTR ((void* )4L)

typedef enum objType { SystemDialog, UserDialog, WindowObj, DialogWindow, Menu, Control };

class XmObject;
class XmWindow;
class XmDialogWindow;
class XmControl;

class XmManager : public XmRootClass
{
protected:
	XmManager* manager;
	XmObject** objects;
	int maxObjects, objectCount;
friend class XmObject;
friend class XmDialogWindow;
friend class XmUserDialog;
	bool addObject(XmObject*, objType);
	bool removeObject(XmObject*);
	XmObject* findObject(char*, objType);
public:
	XmManager()								{ manager = NULL; objects = NULL; maxObjects = objectCount = 0; }
	virtual ~XmManager();

	XmWindow* window(char* n)				{ return((XmWindow* )findObject(n, WindowObj)); }
	XmUserDialog* dialog(char* n)			{ return((XmUserDialog* )findObject(n, UserDialog)); }
	XmDialogWindow* dialogWindow(char* n)	{ return((XmDialogWindow* )findObject(n, DialogWindow)); }

	void changed(char*, XmObject* = NULL);
};

class XmObject : public XmManager
{
protected:
friend class XmApp;
friend class XmManager;
	long magic;
	XmObject* basePtr;
	XmObject* parent;
    	Widget wid;
    	Window win;
	char* wname;
    	Arg wargs[20];
    	int wargcount;
	GC gc;
    	int inputMask;
	bool destroyWidget;
	cb_data** cdptrs;
	int numcdptrs;

	XmObject()						{ init(NULL, NULL); }
	XmObject(char* n, XmObject* p)	{ init(n, p); }
	~XmObject();

	void init(char*, XmObject*);
	virtual char* hookedCb()		{ return(NULL); }
	virtual bool setCbFor(Widget, char*, XmObject*, cbProc, bool, void*);
	char parse(char*);
friend class XmControlPane;
friend class XmDialogPane;
friend class XmGroupBox;
friend class XmControl;
friend class XmWindow;
friend class XmStatusBar;
#ifdef __GNUG__
friend class XmComboBox;
#endif
	void addArg(XString s, XtArgVal v)	{ wargs[wargcount].name = s; wargs[wargcount].value = v; wargcount++; } 
	void setArg(XString, XtArgVal);
	void setArg(XString s, void* v)		{ setArg(s, (XtArgVal )v); }
	void changeArg(XString, XtArgVal);
	void changeArg(XString s, void* v)	{ changeArg(s, (XtArgVal )v); }
	void forceArg(XString, XtArgVal);
	void forceArg(XString s, void* v)	{ forceArg(s, (XtArgVal )v); }
	void getArg(XString s)				{ wargs[wargcount].name = s; wargs[wargcount].value = (XtArgVal )&(wargs[wargcount].value); wargcount++; }
	int argIndex(XString s);
	bool hasArg(XString s)				{ return(argIndex(s) != -1 ? TRUE : FALSE); }

	virtual XmDialogPane* asDialogPane()	{ return(NULL); }
	virtual XmUserDialog* asDialog()	{ return(NULL); }
	void resetParent()			{ parent = NULL; }

	void syscall(char*);
	void error(char*);
public:
	char* getResourcePath(bool = FALSE);
	XmObject* getBase()					{ return(basePtr); }
	XmObject* getParent()				{ return(parent); }
	const char* getName()				{ return(wname); }
	bool isNamed(char* n)				{ return(strcmp(wname, n) == 0 ? TRUE : FALSE); }
	Widget handle()						{ return(wid); }
	virtual Widget topHandle()			{ return(wid); }
	bool valid()						{ return((bool )(magic == MAGIC_OBJECT_CONST)); }

	bool checkResource(char*);
	virtual objType objectType()=0;
	virtual char* defaultAction()		{ return(NULL); }
	virtual bool realize();
	virtual bool destroy();
	bool destroyOnDelete()				{ return(destroyWidget = TRUE); }

	virtual bool hide();
	virtual bool show();
	bool isRealized();
	bool isVisible();

	virtual bool move(int, int);
	virtual bool resize(int, int);
	virtual bool reframe(int, int, int, int);
	virtual bool getFrame(int&, int&, int&, int&);
protected:
	bool setCallback(cbProc cbCode, bool setOrReset = TRUE, void* data = CB_OBJ_PTR)												{ return(setCbFor(wid, defaultAction(), this, cbCode, setOrReset, data)); }
	bool setCallback(char* cbName, cbProc cbCode, bool setOrReset = TRUE, void* data = CB_OBJ_PTR)									{ return(setCbFor(wid, cbName, this, cbCode, setOrReset, data)); }
public:
	bool setCallback(XmObject* receiver, cbProc cbCode, bool setOrReset = TRUE, void* data = CB_OBJ_PTR)							{ return(setCbFor(wid, defaultAction(), receiver, cbCode, setOrReset, data)); }
	bool setCallback(char* cbName, XmObject* receiver, cbProc cbCode, bool setOrReset = TRUE, void* data = CB_OBJ_PTR)				{ return(setCbFor(wid, cbName, receiver, cbCode, setOrReset, data)); }
	bool setCallback(Widget w, char* cbName, XmObject* receiver, cbProc cbCode, bool setOrReset = TRUE, void* data = CB_OBJ_PTR)	{ return(setCbFor(w, cbName, receiver, cbCode, setOrReset, data)); }

	bool setBackgroundColor(char*);
	bool setForegroundColor(char*);

	bool setCursor(char*);
	bool setCursor(int);

	virtual void update(char*)			{ }
};

inline bool XmApp::setCursor(char* n)	{ return(appShell->setCursor(n)); }
inline bool XmApp::setCursor(int n)	{ return(appShell->setCursor(n)); }

struct Entry
{
	cbProc callback;
	XmManager* receiver;
	void* client_data;
	char* name;
	char* text;
	bool checkable;
	bool radio;

	Entry()
{ name = text = NULL; receiver = NULL; callback = NULL; client_data = NULL; checkable = radio = FALSE; }
	Entry(char* t, cbProc c, void* cd = NULL)
{ name = text = t; receiver = NULL; callback = c; client_data = cd; checkable = radio = FALSE; }
	Entry(char* t, XmObject* o, cbProc c, void* cd = NULL)
{ name = text = t; receiver = o; callback = c; client_data = cd; checkable = radio = FALSE; }
	Entry(char* n, char* t)
{ name = n; text = t; receiver = NULL; callback = NULL; client_data = NULL; checkable = radio = FALSE; }
	Entry(char* n, char* t, cbProc c, void* cd = NULL)
{ name = n; text = t; receiver = NULL; callback = c; client_data = cd; checkable = radio = FALSE; }
	Entry(char* n, char* t, XmObject* o, cbProc c, void* cd = NULL)
{ name = n; text = t; receiver = o; callback = c; client_data = cd; checkable = radio = FALSE; }
};

struct CheckEntry : Entry
{
	CheckEntry(char* t, cbProc c, void* cd = NULL, bool rb = FALSE) : Entry(t, c, cd)
{ checkable = TRUE; radio = rb; }
	CheckEntry(char* t, XmObject* o, cbProc c, void* cd = NULL, bool rb = FALSE) : Entry(t, o, c, cd)
{ checkable = TRUE; radio = rb; }
	CheckEntry(char* n, char* t, cbProc c, void* cd = NULL, bool rb = FALSE) : Entry(n, t, c, cd)
{ checkable = TRUE; radio = rb; }
	CheckEntry(char* n, char* t, XmObject* o, cbProc c, void* cd = NULL, bool rb = FALSE) : Entry(n, t, o, c, cd)
{ checkable = TRUE; radio = rb; }
};


#define CB(x) ((cbProc )&x)
#define NULLENTRY Entry()

#ifndef MAX_SUB_MENUES
#define MAX_SUB_MENUES 10
#endif

class XmMenu : public XmObject
{
friend class XmSubMenu;
friend int locateItem(Widget);
	int nextItemPos;
	XmSubMenu* submenues[MAX_SUB_MENUES];
	Widget subHandles[MAX_SUB_MENUES];
	int numSubmenues;
protected:
	XmMenu(char*, XmObject*);
	~XmMenu();

	virtual Widget getCurrentLabel()	{ return((Widget )NULL); }
	XmObject* parentOfMenu();
	bool checkMenuResource(char*);
public:
	objType objectType()				{ return(Menu); }

	bool resize(int, int)				{ return(FALSE); }
	bool reframe(int, int, int, int)	{ return(FALSE); }

	void insertItemPos(int pos)			{ nextItemPos = pos; } 
	bool addSeparator(char* = (char* )NULL);
	bool removeSeparator(char* = (char* )NULL);
#ifndef NO_VA_OBJS
	bool addItem(Entry e)				{ return(addItems(e, Entry())); }
	bool addItems(Entry, ...);
#else
	VA_OBJ_SUBST(addItem, addItems)
#endif
	bool removeItem(char* n)			{ return(removeItems(n, NULL)); }
	bool removeItems(char*, ...);
	bool enableItem(char* n)			{ return(enableItems(n, NULL)); }
	bool enableItems(char*, ...);
	bool disableItem(char* n)			{ return(disableItems(n, NULL)); }
	bool disableItems(char*, ...);

	XmSubMenu* addSubmenu(char*);
	XmSubMenu* addSubmenu(Entry);
	XmSubMenu* submenuAt(char*);
	bool removeSubmenu(char*);
public:
	bool changeItemText(char*, char*);
	bool getItemStatus(char*);
	bool setItemStatus(char*, bool);
};

class XmPopupMenu : public XmMenu
{
friend class XmDialogWindow;
	void makeDefault(bool);
	Widget getCurrentLabel()			{ return(wid); }
public:
	XmPopupMenu(char*, XmObject*, bool = FALSE);

	bool setLabel(char*);
	bool popup();
};

#ifndef MAX_DROPDOWN_LABELS
#define MAX_DROPDOWN_LABELS 15
#endif

class XmDropdownMenu : public XmMenu
{
friend int locateLabel(Widget);
	Widget dropdowns[MAX_DROPDOWN_LABELS];
	Widget currentLabel;
	int numDropdowns;
	int nextLabelPos;
friend class XmDialogWindow;
	XmDropdownMenu(char*, XmObject*);

	Widget getCurrentLabel()			{ return(currentLabel); }
public:
	void insertLabelPos(int pos)		{ nextLabelPos = pos; }
	bool addLabel(char* n)				{ return(addLabels(n, NULL)); }
	bool addLabels(char*, ...);
#ifndef NO_VA_OBJS
	bool addLabel(Entry e)				{ return(addItems(e, Entry())); }
	bool addLabels(Entry, ...);
#else
	VA_OBJ_SUBST(addLabel, addLabels)
#endif
	bool removeLabel(char* n)			{ return(removeLabels(n, NULL)); }
	bool removeLabels(char*, ...);
	bool setCurrentLabel(char*);

	bool changeLabelText(char*, char*);
	bool changeItemText(char* l, char* i, char* txt)	{ return((bool )(setCurrentLabel(l) && XmMenu::changeItemText(i, txt))); }
};

class XmSubMenu : public XmMenu
{
friend class XmMenu;
	XmSubMenu(char*, XmMenu*);
	~XmSubMenu();

	Widget getCurrentLabel()			{ return(wid); }
};

typedef enum dlgType { FORM, MESSAGE, PROMPT, SELECTION, FILES };
typedef enum rscMode { INTERNAL, EXTERNAL };

#ifdef __OLDGNU__ // __GNUG__
class XmDialog : virtual public XmObject
#else
class XmDialog : public XmObject
#endif
{
	bool completed, returned;
protected:
	xmStyle dlgStyle;

	XmDialog()							{ dlgStyle = XmSdlgModeless; }
	XmDialog(char*, XmObject*, xmStyle = 0);
	~XmDialog();

	bool setLabelAndMode(char*);
	bool run();
private:
friend class XmMsgBox;
	void ok(void*);
	void cancel(void*);
	void close(void*);
friend class XmObject;
	xmStyle getStyle()	{ return(dlgStyle); }
};

typedef enum msgButton { OK, CANCEL, HELP };

class XmSystemDialog : public XmDialog
{
protected:
	XmSystemDialog(char*, XmObject*, xmStyle = 0);

	objType objectType()				{ return(SystemDialog); }
};

class XmMsgBox : public XmSystemDialog
{
protected:
	XmMsgBox(char*, XmObject*, xmStyle);

	void setDefaultButtonProcs(XmObject*, cbProc);
public:
	XmMsgBox(char*, char*, xmStyle = XmSmboxInfo, XmObject* = (XmObject* )NULL, cbProc = NULL);

	bool showMsg();
	bool setButtonText(msgButton, char*);
};

class XmPrompter : public XmMsgBox
{
protected:
	XmPrompter(char*, XmObject*, xmStyle);
public:
	XmPrompter(char*, char*, char*, xmStyle = XmSmboxOption, XmObject* = (XmObject* )NULL, cbProc = NULL);

	char* prompt();
	bool setText(char*);
	char* getText();
};

class XmListPrompter : public XmPrompter
{
public:
	XmListPrompter(char**, char*, char*, char*, xmStyle = XmSmboxOption, XmObject* = (XmObject* )NULL, cbProc = NULL);

	int promptIndex();
	XmListPrompter* selectText(char*);
	XmListPrompter* selectIndex(int);

	int getIndex();
};

class XmFileSelector : public XmPrompter
{
public:
	XmFileSelector(char*, char* = NULL, char* = NULL, xmStyle = XmSmboxOption, XmObject* = (XmObject* )NULL, cbProc = NULL);

	char* promptFile();
	bool setPath(char*);
	char* getPath();
};


#ifndef MAX_DLG_CONTROLS
#define MAX_DLG_CONTROLS 100
#endif

class XmControl;
class XmGroupBox;
class XmStaticText;
class XmStaticImage;
class XmPushButton;
class XmCheckBox;
class XmRadioButton;
class XmValuator;
class XmEdit;
class XmScrollBar;
class XmListBox;
class XmComboBox;

// class XmCheckBoxGroup;
// class XmRadioButtonGroup;

#undef PushButton
#undef CheckBox

typedef enum ctrlType { 	GroupBox, StaticText, StaticImage, PushButton,
							CheckBox, RadioButton, Valuator, Edit, ScrollBar, ListBox,
							ComboBox, Drawing, Pane, PaneArea, ToolBar   };
class XmDrawing;
class XmDialogWindow;

class XmControlPane
{
protected:
	XmObject* debugPtr;
public:
	virtual const char* getName()=0;
	virtual Widget handle()=0;
	virtual XmObject* asObject()=0;
};

class XmDialogPane : public XmControlPane
{
friend class XmControl;
friend class XmUserDialog;
friend class XmWindow;
friend class XmStatusBar;
	XmRadioButton* curRadio;
	XmControl* controls[MAX_DLG_CONTROLS];
	int controlCount;
	XmControl* lastFocus;

	int controlIndex(XmControl*);
protected:
	virtual XmControl* findControl(ctrlType, char*, bool = TRUE);
	virtual XmGroupBox* nextGroup(int&);
	virtual void getFocus(Widget);

	XmDialogPane()								{ debugPtr = NULL; curRadio = NULL; controlCount = 0; lastFocus = NULL; }
	virtual ~XmDialogPane();
public:
	virtual bool add(XmControl*, XmObject* = NULL, cbProc = NULL, void* = CB_OBJ_PTR);
	bool add(XmComboBox* c, XmObject* o = NULL, cbProc p = NULL, void* cd = CB_OBJ_PTR)	{ return(add((XmControl* )((void* )c), o, p, cd)); }
	bool removeControl(int);
	bool remove(XmControl* c)	{ return(removeControl(controlIndex(c))); }
	virtual bool remove(char*);
	XmControl* find(char*);

#ifndef __GNUG__
	XmGroupBox* groupBox(char* n)				{ return((XmGroupBox* )findControl(GroupBox, n)); }
	XmStaticText* staticText(char* n)			{ return((XmStaticText* )findControl(StaticText, n)); }
	XmStaticImage* staticImage(char* n)			{ return((XmStaticImage* )findControl(StaticImage, n)); }
	XmPushButton* pushButton(char* n)			{ return((XmPushButton* )findControl(PushButton, n)); }
	XmRadioButton* radioButton(char* n)			{ return((XmRadioButton* )findControl(RadioButton, n)); }
	XmCheckBox* checkBox(char* n)				{ return((XmCheckBox* )findControl(CheckBox, n)); }
	XmValuator* valuator(char* n)				{ return((XmValuator* )findControl(Valuator, n)); }
	XmEdit* edit(char* n)						{ return((XmEdit* )findControl(Edit, n)); }
	XmScrollBar* scrollBar(char* n)				{ return((XmScrollBar* )findControl(ScrollBar, n)); }
	XmListBox* listBox(char* n)					{ return((XmListBox* )findControl(ListBox, n)); }
	XmComboBox* comboBox(char* n)				{ return((XmComboBox* )(void*)findControl(ComboBox, n)); }
	XmDrawing* drawing(char* n)					{ return((XmDrawing* )findControl(Drawing, n)); }
#else
	// avoid calling virtual (gcc bug?), may cause incorrect results
	// in some subclasses - workaround: more steps...
	XmGroupBox* groupBox(char* n)				{ return((XmGroupBox* )XmDialogPane::findControl(GroupBox, n)); }
	XmStaticText* staticText(char* n)			{ return((XmStaticText* )XmDialogPane::findControl(StaticText, n)); }
	XmStaticImage* staticImage(char* n)			{ return((XmStaticImage* )XmDialogPane::findControl(StaticImage, n)); }
	XmPushButton* pushButton(char* n)			{ return((XmPushButton* )XmDialogPane::findControl(PushButton, n)); }
	XmRadioButton* radioButton(char* n)			{ return((XmRadioButton* )XmDialogPane::findControl(RadioButton, n)); }
	XmCheckBox* checkBox(char* n)				{ return((XmCheckBox* )XmDialogPane::findControl(CheckBox, n)); }
	XmValuator* valuator(char* n)				{ return((XmValuator* )XmDialogPane::findControl(Valuator, n)); }
	XmEdit* edit(char* n)						{ return((XmEdit* )XmDialogPane::findControl(Edit, n)); }
	XmScrollBar* scrollBar(char* n)				{ return((XmScrollBar* )XmDialogPane::findControl(ScrollBar, n)); }
	XmListBox* listBox(char* n)					{ return((XmListBox* )XmDialogPane::findControl(ListBox, n)); }
	XmComboBox* comboBox(char* n)				{ return((XmComboBox* )XmDialogPane::findControl(ComboBox, n)); }
	XmDrawing* drawing(char* n)					{ return((XmDrawing* )XmDialogPane::findControl(Drawing, n)); }

	// these virtuals come from the XmUserDialog class, gcc doesn't handle them there... 
	virtual bool initWindowSize(int&, int&, int&, int&)		{ return(FALSE); }
	virtual void initialize()					{ createContents(); initContents(); }
	virtual void createContents()				{ }
	virtual void initContents()					{ }
#endif
	bool disable(char**);
	bool disable(char*, ...);
	bool enable(char**);
	bool enable(char*, ...);
	bool hideAll(char**);
	bool hideAll(char*, ...);
	bool showAll(char**);
	bool showAll(char*, ...);
};

class XmUserDialog : public XmDialog, public XmDialogPane
{
friend class XmDialogWindow;
friend class XmWindow;
	Position ix, iy;
	Dimension iw, ih;

	void initDialog(char*, XmObject*, xmStyle);
	void getFocusCb(void*);
	void destroyCb(void*);
public:
	XmUserDialog(char*, XmObject* = NULL, xmStyle = 0);
	XmUserDialog(char*, XmObject*, XmManager*, xmStyle = 0);
	~XmUserDialog();

	virtual bool initWindowSize(int&, int&, int&, int&)		{ return(FALSE); }

	virtual void initialize()					{ createContents(); initContents(); }
	virtual void createContents()				{ }
	virtual void initContents()					{ }

#ifndef __GNUG__
	Widget handle()								{ return(XmObject::handle()); }
#else
	Widget handle()								{ return(wid); }
#endif
	const char* getName()						{ return(wname); }
	XmObject* asObject()						{ return(this); }

	objType objectType()						{ return(UserDialog); }
	XmDialogPane* asDialogPane()					{ return(this); }
	XmUserDialog* asDialog()					{ return(this); }

	void setInitWindowSize(int x, int y, int w, int h)		{ ix = x; iy = y; iw = w; ih = h; }

	bool run(char*, char* = NULL);
	bool open()							{ initialize(); return(realize()); }
	bool realize();
	bool hide();
	bool show();
	bool setLabel(char*);

	bool setFocus(char*);
	bool setDefaultButton(char*);

	virtual bool queryClose()					{ return(TRUE); }
};

#define isAn(t) isA(t)

class XmControl : public XmObject
{
protected:
friend class XmDialogWindow;
friend class XmWindow;
friend class XmControlPane;
friend class XmDialogPane;
friend class XmPane;
friend class XmUserDialog;
friend class XmGroupBox;
friend class XmPaneArea;
friend class XmToolBar;
friend class XmComboBox;
	Position cx, cy;
	Dimension cw, ch;
	xmStyle style;
	Entry userData;

	virtual bool createControl(XmControlPane*)=0;
	virtual char* defaultCallback()=0;
	virtual ctrlType controlType()=0;

	bool setCbFor(Widget, char*, XmObject*, cbProc, bool, void*);

	virtual void makeRelativeTo(XmGroupBox*);
	virtual void realized()							{ }
	virtual void cleanup()							{ }
	virtual bool hasWidget(Widget w)				{ return((wid == w) ? TRUE : FALSE); }
	void initTabstop(bool = FALSE);
	char* currentLabel();
	void setLabelAndMnemonic();

	XmControl(char*, int, int, int, int, xmStyle);
public:
	~XmControl();
	XmUserDialog* getDialog();

	objType objectType()							{ return(Control); }
	char* defaultAction()							{ return(defaultCallback()); }
	bool isA(ctrlType t)							{ return((bool )(controlType() == t)); }
	void setUserData(XmObject* rec, cbProc code)	{ userData.receiver = rec; userData.callback = code; }

	xmStyle getStyle()								{ return(style); }
	bool changeStyle(xmStyle s)						{ style = s; return(wid ? FALSE : TRUE); }
	bool move(int, int);
	bool resize(int, int);
	bool reframe(int, int, int, int);
	bool getFrame(int&, int&, int&, int&);
	bool updateDimensions();

	virtual XmControl* setText(char*);
	virtual XmControl* setFont(char*);
	virtual char* getText();
	virtual XmControl* setImage(Pixmap);
	virtual XmControl* setFocus();
	virtual XmControl* disable();
	virtual XmControl* enable();
};

class XmGroupBox : public XmControl, public XmDialogPane
{
	XmRadioButton* curRadio;
friend class XmComboBox;
	bool createControl(XmControlPane*);
	char* defaultCallback()							{ return(NULL); }
	ctrlType controlType()							{ return(GroupBox); };

	void getFocusCb(void*);
public:
	XmGroupBox(char* name, int x, int y, int w, int h, xmStyle s = 0) : XmControl(name, x, y, w, h, s)	{ curRadio = NULL; }

	const char* getName()							{ return(wname); }
	Widget handle()								{ return(XmObject::handle()); }
	XmObject* asObject()							{ return(getBase()); }
	XmDialogPane* asDialogPane()						{ return(((XmDialogPane* )((void* )getBase()))); }
	XmUserDialog* asDialog()						{ return(((XmUserDialog* )((void* )parent->getBase()))->asDialog()); }

	bool addAbs(XmControl*, XmObject* = NULL, cbProc = NULL, void* = CB_OBJ_PTR);
	bool addAbs(XmComboBox* c, XmObject* o = NULL, cbProc p = NULL, void* cd = CB_OBJ_PTR)	{ return(addAbs((XmControl* )((void* )c), o, p, cd)); }
	bool remove(char*);

	bool parentFor(XmControl*);
};

class XmPane : public XmControl, public XmControlPane
{
friend class XmPaneArea;
	Widget hLabel;
	XmControl* contents;

	const char* getName()							{ return(wname); }
	Widget handle()									{ return(XmObject::handle()); }
	XmObject* asObject()							{ return(getBase()); }

	bool createControl(XmControlPane*);
	char* defaultCallback()							{ return(NULL); }
	ctrlType controlType()							{ return(Pane); };

	XmPane(char* name, int x, int y, int w, int h) : XmControl(name, x, y, w, h, 0)	{ contents = NULL; }

	bool setContents(XmControl*, XmObject* = NULL, cbProc = NULL);
	bool changeLabel(char*);
public:
	XmControl* getControl(ctrlType t, char* n)		{ return((contents && contents->isA(t) && !strcmp(contents->getName(), n)) ? contents : NULL); }
};

class XmPaneArea : public XmGroupBox
{
	int subPaneCount;

	bool createControl(XmControlPane*);
	char* defaultCallback()							{ return(NULL); }
	ctrlType controlType()							{ return(PaneArea); };

	bool add(XmControl*, XmObject* = NULL, cbProc = NULL, void* = CB_OBJ_PTR);
	bool remove(char*);

	void changed(void*);
public:
	XmPaneArea(char* name, int x, int y, int w, int h) : XmGroupBox(name, x, y, w, h)	{ subPaneCount = 0; }

	int numSubpanes()								{ return(subPaneCount); }

	bool addSubpane(XmControl*, char* = NULL, XmObject* = NULL, cbProc = NULL);
	bool removeSubpane(char* name)					{ return(remove(name)); }

	bool changeSubpaneLabel(char*, char*);
};

class XmToolBar : public XmGroupBox
{
	int rows;
friend class XmStatusBar;
	bool createControl(XmControlPane*);
	char* defaultCallback()							{ return(NULL); }
	ctrlType controlType()							{ return(ToolBar); };
public:
	XmToolBar(char* name, xmStyle s = XmShorizontal) : XmGroupBox(name, 0, 0, 0, 0, s)	{ rows = 1; }

	bool setRows(int n)								{ rows = n; return(wid ? FALSE : TRUE); }

	bool add(XmControl*, XmObject* = NULL, cbProc = NULL, void* = CB_OBJ_PTR);
	bool disableAll(char** l)						{ return(XmDialogPane::disable(l)); }
	bool enableAll(char** l)						{ return(XmDialogPane::enable(l)); }
};

class XmStatusBar : public XmGroupBox
{
	XmEdit* statusLine;

	bool createControl(XmControlPane*);
public:
	XmStatusBar(char* name) : XmGroupBox(name, 0, 0, 0, 0)		{ statusLine = NULL; }

	bool add(XmControl*, XmObject* = NULL, cbProc = NULL, void* = CB_OBJ_PTR);
	bool remove(char*);

	bool setMessage(char*);
};

class XmStaticText : public XmControl
{
	bool createControl(XmControlPane*);
	char* defaultCallback()							{ return(NULL); }
	ctrlType controlType()							{ return(StaticText); };
public:
	XmStaticText(char* name, int x = 0, int y = 0, int w = 0, int h = 0, xmStyle s = 0) : XmControl(name, x, y, w, h, s)	{ }
};

class XmStaticImage : public XmControl
{
	bool createControl(XmControlPane*);
	char* defaultCallback()							{ return(NULL); }
	ctrlType controlType()							{ return(StaticImage); };
public:
	XmStaticImage(char* name, int x = 0, int y = 0, int w = 0, int h = 0, xmStyle s = 0) : XmControl(name, x, y, w, h, s)	{ }
};

class XmPushButton : public XmControl
{
friend class XmComboBox;
	bool createControl(XmControlPane*);
#ifdef MOTIF
	char* defaultCallback()							{ return(XmNactivateCallback); }
#else
	char* defaultCallback()							{ return(XtNcallback); }
#endif
	ctrlType controlType()							{ return(PushButton); };
public:
	XmPushButton(char* name, int x = 0, int y = 0, int w = 0, int h = 0, xmStyle s = 0) : XmControl(name, x, y, w, h, s)	{ }
};

class XmCheckBox : public XmControl
{
friend class XmRadioButton;
	bool createControl(XmControlPane*);
#ifdef MOTIF
	char* defaultCallback()							{ return(XmNvalueChangedCallback); }
#else
	char* defaultCallback()							{ return(XtNcallback); }
#endif
	ctrlType controlType()							{ return(CheckBox); };
public:
	XmCheckBox(char* name, int x = 0, int y = 0, int w = 0, int h = 0, xmStyle s = 0) : XmControl(name, x, y, w, h, s)	{ }

	virtual bool setState(bool);
	bool getState();
};

class XmRadioButton : public XmCheckBox
{
friend class XmDialogPane;
friend class XmUserDialog;
friend class XmGroupBox;
	XmRadioButton* prev, *next;

	bool createControl(XmControlPane*);
	void chainTo(XmRadioButton* n)					{ next = n; n->prev = this; }
#ifdef MOTIF
	char* hookedCb()								{ return(XmNvalueChangedCallback); }
#else
	char* hookedCb()								{ return(XtNcallback); }
#endif
	ctrlType controlType()							{ return(RadioButton); };

	void selected(void*);
	void reset(bool);
public:
	XmRadioButton(char* name, int x = 0, int y = 0, int w = 0, int h = 0, xmStyle s = 0) : XmCheckBox(name, x, y, w, h, s)	{ prev = next = NULL; }

	bool setState(bool);
	XmRadioButton* getSelected();
};

class XmValuator : public XmControl
{
	int decimals;

	bool createControl(XmControlPane*);
#ifdef MOTIF
	char* defaultCallback()							{ return(XmNvalueChangedCallback); }
#else
	char* defaultCallback()							{ return(XtNcallback); }
#endif
	ctrlType controlType()							{ return(Valuator); };

	int f2i(float);
	float i2f(int);
public:
	XmValuator(char* name, int x = 0, int y = 0, int w = 0, int h = 0, xmStyle s = 0) : XmControl(name, x, y, w, h, s)	{ decimals = 0; setOrientation(w > h ? TRUE : FALSE); }

	bool setOrientation(bool);	// TRUE - horizontal
	bool setDirection(bool);	// TRUE - left2right/top2bottom
	short setDecimalPoints(short);
	void displayValue(bool = TRUE);

	XmControl* setText(char*);
	XmControl* setRange(int, int);
	XmControl* setRange(float fmin, float fmax)	{ return(setRange(f2i(fmin), f2i(fmax))); }
	XmControl* setValue(int);
	XmControl* setValue(float f)			{ return(setValue(f2i(f))); }
	int getValue();
	float getFloatValue()				{ return(i2f(getValue())); }
};

class XmEdit : public XmControl
{
friend class XmComboBox;
	bool multiLine;
	bool hScroll, vScroll;
	char* pasteBuffer, * lastVersion;

	bool createControl(XmControlPane*);
#ifdef MOTIF
	char* hookedCb()								{ return(XmNvalueChangedCallback); }
	char* defaultCallback()							{ return(XmNvalueChangedCallback); }
#else
	char* hookedCb()								{ return(XtNcallback); }
	char* defaultCallback()							{ return(XtNcallback); }
#endif
	ctrlType controlType()							{ return(Edit); };

	void changed(void*);
public:
	XmEdit(char* name, int x = 0, int y = 0, int w = 0, int h = 0, xmStyle s = 0) : XmControl(name, x, y, w, h, s)	{ multiLine = hScroll = vScroll = FALSE; pasteBuffer = lastVersion = NULL; }
	~XmEdit();

	Widget topHandle()								{ return((vScroll || hScroll) ? (wid ? XtParent(wid) : NULL) : wid); }
	bool resize(int, int);
	bool reframe(int, int, int, int);

	bool enableMultipleLines()						{ style |= XmSmultiLine; return(wid ? FALSE : TRUE); }
	bool enableHScroll()							{ style |= XmSautovscroll; return(wid ? FALSE : TRUE); }
	bool enableVScroll()							{ style |= XmSautohscroll; return(wid ? FALSE : TRUE); }

	XmControl* setText(char*);
	char* getText();

	bool setEditable(bool);

	long getInsertPosition(int* = NULL, int* = NULL);
	bool setInsertPosition(long);
	bool setInsertPosition(int, int);

	char* getSelection();
	bool getSelection(long&, long&);
	bool setSelection(long, long);
	bool clearSelection();

	bool cut();
	bool copy();
	bool paste();
};

class XmListBox : public XmControl
{
friend class XmComboBox;
	bool multiSel;
	char** initialContents;

	bool createControl(XmControlPane*);
#ifdef MOTIF
	char* defaultCallback()							{ return(multiSel ? XmNextendedSelectionCallback : XmNbrowseSelectionCallback); }
#else
	char* defaultCallback()							{ return(XtNcallback); }
#endif
	ctrlType controlType()							{ return(ListBox); };
	void cleanup();
public:
	XmListBox(char* name, int x = 0, int y = 0, int w = 0, int h = 0, xmStyle s = 0) : XmControl(name, x, y, w, h, s)	{ multiSel = (s & XmSmultipleSel ? TRUE : FALSE); initialContents = NULL; }

	Widget topHandle()								{ return(wid ? XtParent(wid) : NULL); }
	bool resize(int, int);
	bool reframe(int, int, int, int);

	XmListBox* add(char* n)							{ return(addAll(n, NULL)); }
	XmListBox* insert(int i, char* n)				{ return(insertAll(i, n, NULL)); }
	bool remove(char* n)							{ return(removeAll(n, NULL)); }
	bool remove(int i)								{ return(removeAll(i, NULL)); }
	bool change(char*, char*);
	bool change(int, char*);

	XmListBox* addAll(char**);
	XmListBox* addAll(char*,...);
	XmListBox* insertAll(int, char**);
	XmListBox* insertAll(int, char*,...);
	bool removeAll(char**);
	bool removeAll(char*,...);
	bool removeAll(int*);
	bool removeAll(int,...);
	bool setContents(char** lines)					{ if(handle()) { clear(); addAll(lines); } else initialContents = lines; return(TRUE); }

	bool clear();

	bool selectText(char*, bool = FALSE);
	bool selectIndex(int, bool = FALSE);
	bool deselectText(char*);
	bool deselectIndex(int);
	bool deselectAll();

	bool setDoubleClickCallback(XmObject*, cbProc, bool = TRUE, void* = NULL);
	
	char** getItems();
	int getItemCount();

	int selectedCount();
	char* selectedItem()			{ char** all = selectedItems(); return(all ? all[0] : NULL); }
	char** selectedItems();
	int selectedIndex()			{ int* all = selectedIndices(); return(all ? all[0] : 0); }
	int* selectedIndices();
};

typedef enum comboType { SIMPLE, DROPDOWN, DROPDOWNLIST };

class XmComboBox : public XmGroupBox, public XmPushButton, public XmListBox, public XmEdit
{
	Widget popup;
	comboType cbStyle;
	bool lbMapped, ignoreSelect;

	bool createControl(XmControlPane*);
#ifdef MOTIF
	char* hookedCb()								{ return(XmNvalueChangedCallback); }
	char* defaultCallback()							{ return(XmNvalueChangedCallback); }
#else
	char* hookedCb()								{ return(XtNcallback); }
	char* defaultCallback()							{ return(XtNcallback); }
#endif
	ctrlType controlType()							{ return(ComboBox); }

	void makeRelativeTo(XmGroupBox*);
	void showList(bool);
	void cleanup()				{ ((XmListBox*)this)->cleanup(); }

	void selectArrow(void*);
	void changeText(void*);
	void selectInList(void*);
	void cleanupCb(void*);

	bool setCbFor(Widget w, char* n, XmObject* rec, cbProc p, bool sr, void* cd)	{ return(MEMBER(XmEdit, setCbFor)(w, n, rec, p, sr, cd)); }
public:
	XmComboBox(char* name, int x = 0, int y = 0, int w = 35, int h = 35, xmStyle s = 0);
	~XmComboBox();

	operator XmControl* ()		{ return((XmGroupBox* )this); }

//	bool setCallback(XmObject* receiver, cbProc cbCode, bool setOrReset = TRUE, void* data = CB_OBJ_PTR)	{ return(((XmEdit*)this)->setCallback(receiver, cbCode, setOrReset, data)); }

	bool setCallback(XmObject* receiver, cbProc cbCode, bool setOrReset = TRUE, void* data = CB_OBJ_PTR)							{ return(setCbFor(MEMBER(XmEdit, wid), defaultCallback(), receiver, cbCode, setOrReset, data)); }

	bool add(XmControl* c, XmObject* o = NULL, cbProc p = NULL, void* cd = CB_OBJ_PTR)	{ return(MEMBER(XmGroupBox, add)(c, o, p, cd)); }
	bool add(XmComboBox* c, XmObject* o = NULL, cbProc p = NULL, void* cd = CB_OBJ_PTR)	{ return(MEMBER(XmGroupBox, add)((XmControl* )((void* )c), o, p, cd)); }

        // and to cover bugs in gcc...	
	XmListBox* add(char* n)		{ return(((XmListBox*)this)->add(n)); }
	bool remove(char* n)		{ return(((XmListBox*)this)->remove(n)); }
	
	void kbdSelect(int i)	{ ignoreSelect = TRUE; selectIndex(i, TRUE); }
	void toggleList()	{ showList(lbMapped ? FALSE : TRUE); }

	Widget topHandle()								{ return(XmGroupBox::handle()); }
	bool move(int, int);
	bool resize(int, int);
	bool reframe(int, int, int, int);

	bool realize();
	const char* getName()							{ return(XmGroupBox::getName()); }
	XmUserDialog* getDialog()						{ return(XmGroupBox::getDialog()); }

	XmControl* setText(char* txt)					{ return(XmEdit::setText(txt)); }
	char* getText()									{ return(XmEdit::getText()); }

	XmControl* setFocus();
	XmControl* disable();
	XmControl* enable();
};

class XmDialogWindow : public XmUserDialog
{
friend class XmWindow;
friend class XmToolBox;
	xmStyle style;
	XmPopupMenu* popup;
	XmDropdownMenu* dropdown;

	bool initWindowSize(int&, int&, int&, int&)		{ return(FALSE); }
	virtual void menuSet()							{ }
protected:
	XmControl* findControl(ctrlType, char*, bool = TRUE);
public:
	XmDialogWindow(char*, XmObject* = NULL, xmStyle = 0);
	XmDialogWindow(char*, XmObject*, XmManager*, xmStyle = 0);
	~XmDialogWindow();

	objType objectType()					{ return(DialogWindow); }
	void initialize()						{ }
	bool realize();

	bool add(XmControl*, XmObject* = NULL, cbProc = NULL, void* = CB_OBJ_PTR);
	bool add(XmComboBox* c, XmObject* o = NULL, cbProc p = NULL, void* cd = CB_OBJ_PTR)	{ return(add((XmControl* )((void* )c), o, p, cd)); }

	XmPopupMenu* createPopupMenu(char* n = "&Menu")	{ popup = new XmPopupMenu(n, this, TRUE); /* menuSet() */; return(popup); }
	XmDropdownMenu* createDropdownMenu()		{ dropdown = new XmDropdownMenu("dropdown", this); /* menuSet(); */ return(dropdown); }
	XmPopupMenu* getPopup()				{ return(popup); }
	XmDropdownMenu* getDropdown()			{ return(dropdown); }
	XmPopupMenu* setPopup(XmPopupMenu* p)		{ XmPopupMenu* oldpopup = popup; oldpopup->makeDefault(FALSE); popup = p; p->makeDefault(TRUE); return(oldpopup); }
};

class XmWindow : public XmDialogWindow
{
friend class XmStatusBar;
	XmPaneArea* panes;
	XmToolBar* toolbars[4];
	XmStatusBar* statusbar;

	void initWindowContents(char*);
	void checkLayout();
	void menuSet();

	XmControl* findControl(ctrlType t, char* n, bool w = TRUE)		{ return(panes->findControl(t, n, w)); }
public:
	XmWindow(char* n, XmObject* par = NULL, xmStyle s = 0) : XmDialogWindow(n, par, s) { initWindowContents(n); }
	XmWindow(char* n, XmObject* par, XmManager* m, xmStyle s = 0) : XmDialogWindow(n, par, m, s) { initWindowContents(n); }
#ifdef __GNUG__
	bool initWindowSize(int&, int&, int&, int&)		{ return(FALSE); }
	void initialize()								{ }
#endif
	objType objectType()							{ return(WindowObj); }
	bool realize();

	bool addSubpane(ctrlType, char*, char* = NULL, int = 0, int = 0, XmObject* = NULL, cbProc = NULL);
	bool addSubpane(XmControl*, char* = NULL, XmObject* = NULL, cbProc = NULL);
	bool removeSubpane(char* name)					{ return(panes->removeSubpane(name)); }
	bool changeSubpaneLabel(char* n, char* l)		{ return(panes->changeSubpaneLabel(n, l)); }

	XmToolBar* addToolbar(xmStyle);
	bool removeToolbar(XmToolBar*);
	XmToolBar* toolbar(xmStyle);

	XmStatusBar* createStatusBar();
	XmStatusBar* getStatusBar()						{ return(statusbar); }
};

#ifndef __GNUG__

class XmToolBox : public XmDialogWindow, public XmToolBar
{
public:
	XmToolBox(char*, XmObject*, xmStyle = XmSvertical, int = 1);
	XmToolBox(char*, XmObject*, XmManager*, xmStyle = XmSvertical, int = 1);

	bool move(int x, int y)						{ return(XmDialogWindow::move(x, y)); }
	bool resize(int x, int y)					{ return(XmDialogWindow::resize(x, y)); }
	bool reframe(int x, int y, int w, int h)	{ return(XmDialogWindow::reframe(x, y, w, h)); }
	char* defaultAction()						{ return(XmToolBar::defaultAction()); }
	objType objectType()						{ return(XmDialogWindow::objectType()); }
	bool setCbFor(Widget w, char* n, XmObject* r, cbProc cbk, bool sr, void* cd)	{ return(XmDialogWindow::setCbFor(w, n, r, cbk, sr, cd)); }
	XmUserDialog* asDialog()					{ return(XmDialogWindow::asDialog()); }

	bool realize()								{ return(XmUserDialog::realize()); }
};

#endif

#endif
