#include <qpopmenu.h>
#include <qmsgbox.h>

#include "DlgParser.h"
#include "pixmap/Pixmap.h"
#include "pixmap/PixmapWnd.h"
#include "pixmap/PixmapDraw.h"
#include "pixmap/PixmapGenerate.h"

Pixmap::Pixmap( ProjectWnd &prj, const QString &name )
  : DlgObject( prj, name ),
    itsPixmapWnd( new PixmapWnd( this, &itsProjectWnd ))
{
  itsPixmapFormat = "BMP";
  itsPixmapName = "";

  /* Create three popupmenus for the List:
   * 1. Not in a project
   * 2. In a project, not hidden
   * 3. In a project, hidden
   */
  unboundPopup = new QPopupMenu;
  unboundPopup->insertItem( "Save", this, 
                              SLOT(Save()) );
  unboundPopup->insertItem( "Save As...", this, 
                              SLOT(SaveAs()) );
  unboundPopup->insertItem( "Close", this, 
                              SLOT(Hide()) );
  unboundPopup->insertSeparator();
  unboundPopup->insertItem( "Raise", this, 
                              SLOT(Raise()) );

  unboundHiddenPopup = new QPopupMenu;
  unboundHiddenPopup->insertItem( "Show", this, SLOT(Show()) );

  boundPopup = new QPopupMenu;
  boundPopup->insertItem( "Save", this, SLOT(Save()) );
  boundPopup->insertItem( "Save As...", this, SLOT(SaveAs()) );
  boundPopup->insertSeparator();
  boundPopup->insertItem( "Raise", this, SLOT(Raise()) );
  boundPopup->insertSeparator();
  boundPopup->insertItem( "Remove", this, SLOT(Remove()) );

  // Popupmenu for closed dialog entries in projects
  boundHiddenPopup = new QPopupMenu;
  boundHiddenPopup->insertItem( "Show", this, SLOT(Show()) );
  boundHiddenPopup->insertSeparator();
  boundHiddenPopup->insertItem( "Remove", this, SLOT(Remove()) );

  status = DlgObject_Init;
}

Pixmap::~Pixmap()
{
  delete itsPixmapWnd;
}

void Pixmap::GetGeneratedFiles( QString &src, QString &hdr, QString &moc )
{
  src = src;
  hdr = hdr;
  moc = moc;
}

bool Pixmap::Load()
{
  bool ok;

  if( !itsFilename.isEmpty() )
  {
    itsParser = new DlgParser( itsFilename );

    connect( itsParser, SIGNAL(PixmapKeyValue(QString &,QString &)), 
             SLOT(RestorePixmap(QString &,QString &)) );

    // And start the action
    itsParser->StartParsing();

    // That's it for the Parser object
    delete itsParser;    

    itsParser = NULL;
    status = DlgObject_Loaded;
  
    ok = TRUE;
  } else
    ok = FALSE;

  return ok;
}

void Pixmap::RestorePixmap
(
 QString&	key,
 QString&	value
 )
{
  if( key == "Filename" )
  {
    itsPixmapName = value;
    PixmapDraw *drawer = itsPixmapWnd->GetPixmapDraw();
    if( ! drawer->LoadFile( itsPixmapName ) )
    {
      QString str = "Failed to load pixmap " + itsPixmapName;
      QMessageBox::message( "Error...", str, "OK" );
    }
  } else if( key == "Source" )
  {
    itsSource = value;
  } else if( key == "Header" )
  {
    itsHeader = value;
  } else if( key == "VarName" )
  {
    itsVarName = value;
  } else if( key == "VarLengthName" )
  {
    itsVarLengthName = value;
  } else
  {
    // do some error msg
  }
}

bool Pixmap::Save( )
{
  QFile file( itsFilename );
  if( !file.open( IO_WriteOnly ) )
  {
    // error msg
    return FALSE;
  }

  if( itsPixmapName.isEmpty() )
  {
    QString pixN;
    if( itsFilename.right( 4 ) == ".pix" )
    {
      pixN = itsFilename.left( itsFilename.length() - 3 ) + "bmp";
      itsPixmapFormat = "BMP";
    } else
    {
      pixN = "NewPixmap.bmp";
      itsPixmapFormat = "BMP";
    }
    
    QString msg = "This pixmap hasn't got a filename yet.\n"
      "The default would be: " + pixN + "\n"
      "You can choose a new one by pressing Browse...";

    bool def = QMessageBox::query( "Note...", msg, "Take Default", "Browse" );
                  
    if( def )
      itsPixmapName = pixN;
    else
    {
      itsPixmapName = QFileDialog::getSaveFileName( NULL, "*.bmp" );

      if( itsPixmapName.isEmpty() )
      {
        QMessageBox::message( "Error...", "No filename for pixmap. "
                           "Saving aborted", "OK" );
        return FALSE;
      }
    }
  }
    
  QTextStream stream( &file );
    
  stream << "DlgEdit:v1.2:Pixmap:\n";
  stream << "Pixmap {\n";
  stream << "\tFilename {" << itsPixmapName << "}\n";

  if( ! itsVarName.isEmpty() )
    stream << "\tVarName {" << itsVarName << "}\n";

  if( ! itsVarLengthName.isEmpty() )
    stream << "\tVarLengthName {" << itsVarLengthName << "}\n";

  if( ! itsSource.isEmpty() )
    stream << "\tSource {" << itsSource << "}\n";

  if( ! itsHeader.isEmpty() )
    stream << "\tHeader {" << itsHeader << "}\n";

  stream << "}\n";

  PixmapDraw *drawer = itsPixmapWnd->GetPixmapDraw();
  drawer->WriteFile( itsPixmapName, itsPixmapFormat );

  status = DlgObject_Saved;

  file.close();

  emit Saved( this );

  return TRUE;
}

bool Pixmap::Generate( bool force )
{
  QString fName;

  // First time
  if( itsSource.isEmpty() )
  {
    // To estimate the names for generate, 
    // remove path and extension from name
    if( itsPixmapName.find( '/' ) != -1 )
      fName = itsPixmapName.right( itsPixmapName.length() - 
                                   itsPixmapName.findRev( '/' ) - 1 );
    if( fName.find( '.' ) )
      fName = fName.left( fName.find( '.' ) );

    itsHeader = fName + ".h";
    itsSource = fName + ".cpp";
    itsVarName = fName + "_data";
    itsVarLengthName = fName + "_len";
  }

  // Show dialog if necessary
  if( ! force || (itsSource == ".cpp") )
  {
    PixmapGenerate dlg( itsPixmapWnd );
    dlg.SetData( itsVarName, itsVarLengthName, itsHeader, itsSource );

    // Maybe no generation anymore ?
    if( ! dlg.exec() )
      return FALSE;

    dlg.GetData( itsVarName, itsVarLengthName, itsHeader, itsSource );
  }

  // Now using methods from Trolls qembed.cpp
  QFile source( itsSource );
  QFile header( itsHeader );
  source.open( IO_WriteOnly );
  header.open( IO_WriteOnly );

  QTextStream headerStream( &header );
  QTextStream sourceStream( &source );

  sourceStream << "#include \"" << itsHeader << "\"\n";

  // Embed data for all input files
  headerStream << "extern const unsigned int " << itsVarLengthName << ";\n";
  headerStream << "extern const unsigned char " << itsVarName << "[];\n\n";

  PixmapDraw *drawer = itsPixmapWnd->GetPixmapDraw();
  drawer->GeneratePixmap( itsVarName, itsVarLengthName, &source ); 

  source.close();
  header.close();

  status = DlgObject_Generated;
  return TRUE;
}

void Pixmap::Hide()
{
  itsPixmapWnd->hide();
  status = DlgObject_Hidden;
}

void Pixmap::Show()
{
  itsPixmapWnd->show();
  status = DlgObject_Shown;
}

void Pixmap::Raise()
{
  itsPixmapWnd->raise();
}

void Pixmap::CPAvailable( bool available )
{
  available = available;
}

void Pixmap::PixmapChanged()
{
  status = DlgObject_Changed;
  emit Changed( this );
}


void Pixmap::SetName( const QString &dlgFile, bool change )
{
  DlgObject::SetName( dlgFile, change );
  itsPixmapWnd->SetCaption( dlgFile );
}



