/*
 * twininfo -- window information utility for X11, LessTif, Motif and TOAD
 * Copyright (C) 1997 Mark-Andr Hopf (eMail: hopf@informatik.uni-rostock.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xmd.h>
#include <X11/Xatom.h>

#define _TOAD_PRIVATE

#include <toad/toad.hh>

#define THROW(text) { fprintf(stderr,"%s\n",text); exit(EXIT_FAILURE); }

class TLessTifDragTargets
{
	public:
		TLessTifDragTargets(unsigned char *data);
		~TLessTifDragTargets();
		
		void Print(int);
		
	private:
		unsigned char *data;
		struct THeader
		{
			BYTE 		byte_order;
			BYTE 		protocol_version;
			CARD16	num_lists;
			CARD32	size;
		} *header;
		
		struct TTargetList
		{
			CARD16	num_targets;
			CARD32	target[0] __attribute__ ((packed));
		};

		TTargetList* find_index(unsigned index);
};

TLessTifDragTargets::TLessTifDragTargets(unsigned char *d)
{
	data = d;
	header = (THeader*)data;
	data+=sizeof(THeader);
	printf("size: %08lx\n",header->size);
}

TLessTifDragTargets::~TLessTifDragTargets()
{
	XFree((void*)header);
}

TLessTifDragTargets::TTargetList* 
TLessTifDragTargets::find_index(unsigned index)
{
	unsigned char *ptr = data;
	
	if (index>=header->num_lists)
		return (TTargetList*)ptr;
		
	for(unsigned i=0; i<index; i++)
	{
		ptr += 2 + 4 * ((TTargetList*)ptr)->num_targets;
	}
	return (TTargetList*)ptr;
}

static bool error_flag;

static int
DWError(Display *display, XErrorEvent *event)
{
    error_flag = true;
    return 0;
}

void 
TLessTifDragTargets::Print(int i)
{
	int (*old_handler)(Display *, XErrorEvent *);
	old_handler = XSetErrorHandler(DWError);
	
	TTargetList *t;
	t = find_index(i);
	printf("target list %u has %u targets:\n",i,t->num_targets);
	for(unsigned j=0; j<t->num_targets; j++)
	{
		if (t->target[j]==0)
		{
			printf("  None\n");
		}
		else
		{
			const char *name;
			error_flag=false;
			name = XGetAtomName(TOADBase::display,t->target[j] );
			if (error_flag)
				name="(unknown Atom)";
			printf("  %s\n",name);
			XFree((void*)name);
		}
	}
	XSetErrorHandler(old_handler);
}

static TLessTifDragTargets *drag_targets;

void read_lesstif_drag_targets()
{
	Atom xaDragWindow = XInternAtom(TOADBase::display, "_MOTIF_DRAG_WINDOW", True);
	if (xaDragWindow==None)
		THROW("Couldn't find '_MOTIF_DRAG_WINDOW'");
	
	Atom xaDragTargets = XInternAtom(TOADBase::display, "_MOTIF_DRAG_TARGETS", True);
	if (xaDragTargets==None)
		THROW("Couldn't find '_MOTIF_DRAG_TARGETS'");
	
	Window wndDrag=0;
	Atom ret_type;
	int ret_format;
	unsigned long ret_items, ret_rest;
	unsigned char *data;
	
	if (XGetWindowProperty(	TOADBase::display, DefaultRootWindow(TOADBase::display),
													xaDragWindow,
													0L, 100000L,
													False, AnyPropertyType,
													&ret_type, &ret_format, &ret_items, &ret_rest,
													&data ) == Success
			&& ret_type == XA_WINDOW
			&& ret_format == 32
			&& ret_items	== 1	)
	{
		wndDrag = *((Window *)data);
	}
	if (data)
		XFree((void*)data);
		
	if (!wndDrag)
		THROW("Couldn't find LessTif Drag Window");
											
	if (XGetWindowProperty( TOADBase::display, wndDrag, xaDragTargets,
													0L, 100000L,
													False, AnyPropertyType,
													&ret_type, &ret_format, &ret_items, &ret_rest,
													&data ) == Success )
	{
		drag_targets = new TLessTifDragTargets(data);
		return;
	}
	THROW("Couldn't read drag targets from drag window.");
}

class TMyWindow: public TWindow
{
	public:
		TMyWindow(TWindow *parent, const string &title):
			TWindow(parent,title){};
	protected:
		void create();
		void mouseLDown(int,int,unsigned);
		void mouseLUp(int,int,unsigned);
		void PrintEventMask(long events);
};

int ToadMain()
{
	read_lesstif_drag_targets();
	return TMyWindow(NULL,"Identify").Run();
}


void TMyWindow::create()
{
	SetSize(100,50);
}

void TMyWindow::mouseLDown(int,int,unsigned)
{
	printf("\n\nMove pointer above window and release the button.\n");
}

void TMyWindow::mouseLUp(int x,int y,unsigned modifier)
{
	bool bPrintEventMasks = modifier & MK_SHIFT;

	int zx,zy;
	Window wnd;
	XWindowAttributes attr;
	
	if (!XTranslateCoordinates(
		display,
		// Quelle, dieses Fenster
		window,
		// Ziel
		DefaultRootWindow(display),
		// von Position
		x,y,
		// nach
		&zx, &zy,
		// eventuelles Kindfenster
		&wnd))
	{
		printf("Thy window is not of this world, err, display.\n");
		return;
	}
	printf("Maus im Root Window: %i,%i\n",zx,zy);
	if (wnd)
	{
		printf("1. Window: 0x%08lx\n",wnd);
		XGetWindowAttributes(display, wnd, &attr);
		printf("  x:%i y:%i w:%i h:%i\n",attr.x,attr.y,attr.width,attr.height);
		if (bPrintEventMasks)
		{
			printf("  all event masks:\n");
			PrintEventMask(attr.all_event_masks);
			printf("  my event mask:\n");
			PrintEventMask(attr.your_event_mask);
			printf("  do not propagate mask:\n");
			PrintEventMask(attr.do_not_propagate_mask);
		}

		int n;
		const char *name;
		Atom *al = XListProperties(display, wnd, &n);
		if (al)
		{
			printf("  Properties:\n");
			for(int i=0; i<n; i++)
			{
				name = XGetAtomName(display,al[i]);
				printf("    %s\n",name);
				XFree((void*)name);
			}
			XFree((void*)al);
		}

		SetSize(attr.width,attr.height);
		ClearWindow();
	}
	else
		printf("It's the root window! Boahhh....\n");
	
	TPen pen(this);
	unsigned n=1;
	int last_x=0, last_y=0;
	Window oldwnd = DefaultRootWindow(display);
	Atom xaDragReceiverInfo = XInternAtom(display, "_MOTIF_DRAG_RECEIVER_INFO",False);
	while(wnd)
	{
		Window newwnd;
		if (XTranslateCoordinates(
			display,
			oldwnd,
			wnd,
			zx,zy,
			&zx,&zy,
			&newwnd) )
		{
			if (newwnd)
			{
				printf("%u. Window: 0x%08lx\n",n+1,newwnd);
				XGetWindowAttributes(display, newwnd, &attr);
				pen.DrawRectangle(attr.x+last_x,attr.y+last_y,attr.width,attr.height);
				printf("  x:%i y:%i w:%i h:%i\n",attr.x,attr.y,attr.width,attr.height);
				if (bPrintEventMasks)
				{
					printf("  all event masks:\n");
					PrintEventMask(attr.all_event_masks);
					printf("  my event mask:\n");
					PrintEventMask(attr.your_event_mask);
					printf("  do not propagate mask:\n");
					PrintEventMask(attr.do_not_propagate_mask);
				}
				int n;
				const char *name;
				Atom *al = XListProperties(display, newwnd, &n);
				if (al)
				{
					printf("  Properties:\n");
					for(int i=0; i<n; i++)
					{
						name = XGetAtomName(display,al[i]);
						printf("    %s\n",name);
						XFree((void*)name);
						if (al[i]==xaDragReceiverInfo)
						{
							struct XmDndReceiverProp {
						    BYTE        byte_order;
						    BYTE        protocol_version __attribute__ ((packed));
						    BYTE        protocol_style __attribute__ ((packed));
						    BYTE        pad1 __attribute__ ((packed));
						    CARD32      proxy_window __attribute__ ((packed));
						    CARD16      num_drop_sites __attribute__ ((packed));
						    CARD16      pad2 __attribute__ ((packed));
						    CARD32      total_size __attribute__ ((packed));
							} *dri;
							Atom ret_type;
							int ret_format;
							unsigned long ret_length, ret_bytesleft;
					    if (XGetWindowProperty(
								display, 
								newwnd, 
								xaDragReceiverInfo, // property to read
								0L, 100000L, 				// offset, length
								False,							// don't delete
         				xaDragReceiverInfo, // requested type
         				&ret_type, 
         				&ret_format, 
         				&ret_length, 
         				&ret_bytesleft,
			         	(unsigned char **)&dri) == Success)
							{
								printf("        read %lu bytes\n", ret_length);
								switch(dri->byte_order)
								{
									case 'B':
										name="MSB first";
										break;
									case 'l':
										name="LSB first";
										break;
									default:
										name="(unknown)";
								}
								printf("        byte order          :%s\n",name);
								printf("        protocol version    :%u\n",dri->protocol_version);
								switch(dri->protocol_style)
								{
									case 0:
										name="drag_none";
										break;
									case 1:
									case 3:
										name="preregister";
										break; 
									case 5:
									case 2:
									case 4:
										name="dynamic";
										break;
									default:
										name="(unknown)";
								}
								printf("        protocol style      :%s\n",name);
								printf("        ?                   :%u\n",dri->pad1);
								printf("        proxy window        :%lx\n",dri->proxy_window);
								printf("        number of drop sites:%u\n",dri->num_drop_sites);
								printf("        ?                   :%u\n",dri->pad2);
								printf("        total property size :%lu\n",dri->total_size);
								
								unsigned char *ptr = (unsigned char*) dri;
								ptr+=16; // sizeof(XmDndReceiverProp);
								
								struct DSHeader
								{
									CARD16 flags __attribute__ ((packed));
									CARD16 import_targets_id __attribute__ ((packed));
									CARD32 n_boxes __attribute__ ((packed));
								};
								
								struct DANone	// animation style = none
								{
									CARD16 borderWidth __attribute__ ((packed));
									CARD16 pad1 __attribute__ ((packed));
									CARD32 pas2 __attribute__ ((packed));
								} *pDANone;
								
								struct DBox
								{
									INT16 x1 __attribute__ ((packed));
									INT16 x2 __attribute__ ((packed));
									INT16 y1 __attribute__ ((packed));
									INT16 y2 __attribute__ ((packed));
								} *pDBox;
								
								DSHeader *dh;
								
								pen.SetColor(255,0,0);
								pen.SetLineWidth(3);
								int box_count = 0;
								
								for(int j=0; j<dri->num_drop_sites; j++)
								{
									dh=(DSHeader*)ptr;
									printf("        %i. drop site\n",j+1);
									printf("          animation style : %4x\n", dh->flags&0xF000);
									
									if (dh->flags&0x0F00==0)
										name = "noop";
									else if (dh->flags&0x0100==0)
										;
									printf("          operations      : %4x\n", dh->flags&0x0F00);
									
									
									printf("          drop type       : %4x\n", dh->flags&0x00F0);
									printf("          drop activity   : %4x\n", dh->flags&0x000C);
									printf("          traversal type  : %4x\n", dh->flags&0x0003);
									printf("          import targets  : %x\n", dh->import_targets_id);
									drag_targets->Print(dh->import_targets_id);
									printf("          number of boxes : %lu\n", dh->n_boxes);
									ptr+=sizeof(DSHeader);

									switch(dh->import_targets_id)
									{
										case XA_PIXMAP:	// magenta
											pen.SetColor(255,0,255);
											break;
										case XA_DRAWABLE:	// violett
											pen.SetColor(127,0,127);
											break;
										case XA_INTEGER:	// lightblue
											pen.SetColor(0,0,255);
											break;
										case XA_ATOM:	// lightred
											pen.SetColor(255,0,0);
											break;
										case XA_FONT:	// green
											pen.SetColor(0,127,0);
											break;
										default:
											pen.SetColor(0,0,127);
									}
	
									switch(dh->flags&0xF000)
									{
										case 0x0000: // none
											pDANone = (DANone*)ptr;
											printf("          border width: %u\n",pDANone->borderWidth);
											ptr+=8; // sizeof(DANone);
											break;
										case 0x4000: // highlight
											ptr+=16;		// 8,16,24,(32),40
											break;
										default:
											printf("          can't handle this animation style\n");
											goto stop;
									}

									for(unsigned k=0; k<dh->n_boxes; k++)
									{
										pDBox = (DBox*) ptr;
										printf("          box %i: (%u,%u,%u,%u)\n",box_count+1,
											pDBox->x1,pDBox->y1,pDBox->x2,pDBox->y2);
										pen.DrawRectangle(
											pDBox->x1+last_x,
											pDBox->y1+last_y,
											pDBox->x2-pDBox->x1+1,
											pDBox->y2-pDBox->y1+1);
										ptr+=8; // sizeof(DBox);
										box_count++;
/*										if (box_count>=12)
											goto stop; */
									}									
								}
								
								stop:
								pen.SetLineWidth(0);
								pen.SetColor(0,0,0);
								
								XFree((void*)dri);
							}
						}
					}
					XFree((void*)al);
				}
				else
				{
					printf("  No Properties\n");
				}
				
				last_x = attr.x+last_x;
				last_y = attr.y+last_y;
			}
			else
			{
				printf("That's all folks...\n");
			}
		}
		n++;
		oldwnd = wnd;
		wnd = newwnd;
	}
}

void TMyWindow::PrintEventMask(long events)
{
	#define CHECK(mask) if (events&mask) printf("    " #mask "\n")
	CHECK(ButtonPressMask);
	CHECK(ButtonReleaseMask);
  CHECK(StructureNotifyMask);
	CHECK(SubstructureNotifyMask);
	CHECK(SubstructureRedirectMask);
	CHECK(ColormapChangeMask);
	CHECK(EnterWindowMask);
	CHECK(LeaveWindowMask);
	CHECK(ExposureMask);
	CHECK(FocusChangeMask);
	// CHECK(KeyMapStateMask);
	CHECK(KeyPressMask);
	CHECK(KeyReleaseMask);
	CHECK(ButtonMotionMask);
	CHECK(Button1MotionMask);
	CHECK(Button2MotionMask);
	CHECK(Button3MotionMask);
	CHECK(Button4MotionMask);
	CHECK(Button5MotionMask);
	CHECK(PointerMotionHintMask);
	CHECK(PointerMotionMask);
	CHECK(PropertyChangeMask);
	CHECK(ResizeRedirectMask);
	CHECK(VisibilityChangeMask);
}
