/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-98 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#define _TOAD_PRIVATE

#include <toad/toad.hh>
#include <vector>

#warning "translation in _dx,_dy is ignored"

static void curve(TPolygon&,double,double,double,double,double,double,double,double);
static void fcurve(TDPolygon&,double,double,double,double,double,double,double,double);

static TPolygon lst;

void TPen::DrawBezier(int x1,int y1, 
											int x2,int y2, 
											int x3,int y3, 
											int x4,int y4)
{
	lst.erase(lst.begin(), lst.end());
	lst.push_back(TPoint(x1, y1));
	curve(lst, x1,y1, x2,y2, x3,y3, x4,y4);
	DrawLines(lst.begin(), lst.size());
}

void TPen::DrawBezier(double x1,double y1, 
											double x2,double y2, 
											double x3,double y3, 
											double x4,double y4)
{
	lst.erase(lst.begin(), lst.end());
	lst.push_back(TPoint(x1, y1));
	curve(lst, x1,y1, x2,y2, x3,y3, x4,y4);
	DrawLines(lst.begin(), lst.size());
}

void TPen::DrawPolyBezier(const TPoint *p, int n)
{
	lst.erase(lst.begin(), lst.end());
	lst.push_back(p[0]);
	n-=3;
	int i=0;
	while(i<=n) {
		curve(lst,
					p[i].x, p[i].y,
					p[i+1].x, p[i+1].y,
					p[i+2].x, p[i+2].y,
					p[i+3].x, p[i+3].y);
		i+=3;
	}
	DrawLines(lst.begin(), lst.size());
}

void TPen::DrawPolyBezier(const TDPoint *p, int n)
{
	lst.erase(lst.begin(), lst.end());
	lst.push_back(TPoint(p[0].x, p[0].y));
	n-=3;
	int i=0;
	while(i<=n) {
		curve(lst,
					p[i].x, p[i].y,
					p[i+1].x, p[i+1].y,
					p[i+2].x, p[i+2].y,
					p[i+3].x, p[i+3].y);
		i+=3;
	}
	DrawLines(lst.begin(), lst.size());
}

void TPen::FillPolyBezier(const TPoint *p, int n)
{
	lst.erase(lst.begin(), lst.end());
	lst.push_back(p[0]);
	n-=3;
	int i=0;
	while(i<=n) {
		curve(lst,
					p[i].x, p[i].y,
					p[i+1].x, p[i+1].y,
					p[i+2].x, p[i+2].y,
					p[i+3].x, p[i+3].y);
		i+=3;
	}
	FillPolygon(lst.begin(), lst.size());
}

void TPen::FillPolyBezier(const TDPoint *p, int n)
{
	lst.erase(lst.begin(), lst.end());
	lst.push_back(TPoint(p[0].x, p[0].y));
	n-=3;
	int i=0;
	while(i<=n) {
		curve(lst,
					p[i].x, p[i].y,
					p[i+1].x, p[i+1].y,
					p[i+2].x, p[i+2].y,
					p[i+3].x, p[i+3].y);
		i+=3;
	}

	FillPolygon(lst.begin(), lst.size());
}

//. This method converts a TDPolygon into a TDPolygon bezier curve and
//. offers an opportunity for more sophisticated operations on bezier
//. curves. E.g. calculating the curves bounding box, rotation, etc.
void TPen::Poly2Bezier(const TPoint* src, int n, TPolygon &dst)
{
	dst.erase(dst.begin(), dst.end());
	dst.push_back(TPoint(src[0].x, src[0].y));
	n-=3;
	int i=0;
	while(i<=n) {
		curve(dst,
					src[i].x,   src[i].y,
					src[i+1].x, src[i+1].y,
					src[i+2].x, src[i+2].y,
					src[i+3].x, src[i+3].y);
		i+=3;
	}
}

void TPen::Poly2Bezier(const TDPoint* src, int n, TDPolygon &dst)
{
	dst.erase(dst.begin(), dst.end());
	dst.push_back(TDPoint(src[0].x, src[0].y));
	n-=3;
	int i=0;
	while(i<=n) {
		fcurve(dst,
					src[i].x,   src[i].y,
					src[i+1].x, src[i+1].y,
					src[i+2].x, src[i+2].y,
					src[i+3].x, src[i+3].y);
		i+=3;
	}
}

// This is almost the same algorithm as used by the Fresco Toolkit but I've 
// removed a bug and improved it's speed [MAH]

inline double mid(double a, double b)
{
	return (a + b) / 2.0;
}

static void curve(
	TPolygon &poly,
	double x0, double y0, 
	double x1, double y1,
	double x2, double y2,
	double x3, double y3)
{
	double tx, ty, ax, ay, bx, by;
	ax = x1-x0; ay = y1-y0;
	tx = x2-x1; ty = y2-y1;
	bx = x3-x2; by = y3-y2;
	double f = fabs(ax*ty - ay*tx) + fabs(tx*by - ty*bx);
	if (f*f < 100.0 ) {
		poly.push_back(TPoint((int)x3, (int)y3));
	} else {
		double xx  = mid(x1, x2);
		double yy  = mid(y1, y2);
		double x11 = mid(x0, x1);
		double y11 = mid(y0, y1);
		double x22 = mid(x2, x3);
		double y22 = mid(y2, y3);
		double x12 = mid(x11, xx);
		double y12 = mid(y11, yy);
		double x21 = mid(xx, x22);
		double y21 = mid(yy, y22);
		double cx  = mid(x12, x21);
		double cy  = mid(y12, y21);
		curve(poly, x0, y0, x11, y11, x12, y12, cx, cy);
    curve(poly, cx, cy, x21, y21, x22, y22, x3, y3);
	}
}

static void fcurve(
	TDPolygon &poly,
	double x0, double y0, 
	double x1, double y1,
	double x2, double y2,
	double x3, double y3)
{
	double tx, ty, ax, ay, bx, by;
	ax = x1-x0; ay = y1-y0;
	tx = x2-x1; ty = y2-y1;
	bx = x3-x2; by = y3-y2;
	double f = fabs(ax*ty - ay*tx) + fabs(tx*by - ty*bx);
	if (f*f < 100.0 ) {
		poly.push_back(TDPoint((int)x3, (int)y3));
	} else {
		double xx  = mid(x1, x2);
		double yy  = mid(y1, y2);
		double x11 = mid(x0, x1);
		double y11 = mid(y0, y1);
		double x22 = mid(x2, x3);
		double y22 = mid(y2, y3);
		double x12 = mid(x11, xx);
		double y12 = mid(y11, yy);
		double x21 = mid(xx, x22);
		double y21 = mid(yy, y22);
		double cx  = mid(x12, x21);
		double cy  = mid(y12, y21);
		fcurve(poly, x0, y0, x11, y11, x12, y12, cx, cy);
    fcurve(poly, cx, cy, x21, y21, x22, y22, x3, y3);
	}
}
