/* linux_system.c
 * linux system.c file for xperfmon++
 * S.Coffin  GV Computing   3/96		[v1.2 5/98]
 */

/*  Copyright (c) 1996, 1997 by Stephen Coffin.  All rights reserved.
 *
 *  This program is distributed in the hope that it will be useful.
 *  Use and copying of this software and preparation of derivative works
 *  based upon this software are permitted, so long as the following
 *  conditions are met:
 *      o credit to the authors is acknowledged following current
 *         academic behavior
 *      o no fees or compensation are charged for use, copies, or
 *         access to this software
 *      o this copyright notice is included intact.
 *  This software is made available AS IS, and no warranty is made about 
 *  the software or its performance. 
 * 
 *  Bug descriptions, use reports, comments or suggestions are welcome.
 *  Send them to    scoffin@netcom.com
 */

/* TO DO:
 *	1)  do not collect statistics that are not requested
 *	2)  so far (kernel 2.1.54), only /proc/cpuinfo varies depending
 *      on SMP CPU's....  but this might change in the future :-)
 *			=SC
 */

#include <X11/IntrinsicP.h>

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "system.h"

#define LINESIZE 255
#define VERSIONINFO "/proc/version"
#define CPUINFO "/proc/stat"
#define LOADINFO "/proc/loadavg"
#define MEMINFO "/proc/meminfo"
#define PACKETINFO "/proc/net/dev"

int current_values[NUM_GRAPHS];
int version_major = 0, version_minor = 0, version_dot = 0;

static int filldata();
static int report();

struct sdata {
	long user_cpu;			/* USR_CPU */
	long sys_cpu;			/* SYS_CPU */
	long nice_cpu;
	long idle_cpu;			/* IDL_CPU */
	long intr;				/* INTRPTS */
	long ctxt;				/* CPU_SWITCH */
	float load_current;		/* LOAD_AVG */
	float load_last;
	float load_prev;
	int load_proc;			/* PROCESSES */
	int load_active_proc;
	long mem_avail;
	long mem_used;
	long mem_free;			/* FREE_MEM */
	long mem_shared;
	long mem_buf;
	long swap_avail;
	long swap_used;
	long swap_free;			/* FREE_SWAP */
	long rec_packet;		/* INP_PKT */
	long xmt_packet;		/* OUT_PKT */
	long coll_packet;		/* COLL_PKT */
	long disk_xfers;		/* DSK_XFR */
	long swap_in;			/* SWAP_IN */
	long swap_out;			/* SWAP_OUT */
	long page_in;			/* PAGE_IN */
	long page_out;			/* PAGE_OUT */
};

int dbg = FALSE;
static char path[80];
static FILE *fp_version = NULL;
static FILE *fp_cpu = NULL;
static FILE *fp_meminfo = NULL;
static FILE *fp_packet;
static struct sdata sdata, odata;
char line[LINESIZE];


/* This routine does all necessary setting up of structures
 * that will handle system calls.  It returns num_stats.
 */
sys_setup() {

	/* got to deal with some different file formats in /proc
	 * for different kernel versions
	 */
	sprintf( path, VERSIONINFO );
	if( (fp_version = fopen( path, "r" )) == NULL ) {
		fprintf( stderr, "ERROR: Linux version information not available\n" );
	}
	else {
		fgets( line, LINESIZE, fp_version );
		if( strncmp( line, "Linux version", 13 ) == 0 ) {
			sscanf( &line[13], "%d.%d.%d", &version_major,
				 &version_minor, &version_dot );
		}
		else fprintf( stderr, "ERROR: Linux version information not available\n" );
		fclose( fp_version );
	}
	if( dbg ) {
		 fprintf( stderr, "xperfmon++ version = %s\n", XPERFVERSION );
		 fprintf( stderr, "Linux version = %d.%d.%d\n", 
			version_major, version_minor, version_dot );
	}
	if( version_major < 1 ||
		 version_major > 2 ||
		 (version_major == 2 && version_minor > 2) ) {
		 fprintf( stderr, "Warning:  untested with this Linux version\n" );
 	}

	sprintf( path, CPUINFO );
	if( (fp_cpu = fopen( path, "r" )) == NULL ) {
		fprintf( stderr, "ERROR: cannot open %s\n", path );
		exit( (-1) );
	}
	sprintf( path, MEMINFO );
	if( (fp_meminfo = fopen( path, "r" )) == NULL ) {
		fprintf( stderr, "ERROR: cannot open %s\n", path );
		exit( (-1) );
	}
	sprintf( path, PACKETINFO );
	if( (fp_packet = fopen( path, "r" )) < 0 ) {
		fprintf( stderr, "ERROR: cannot open %s\n", path );
		exit( (-1) );
	}

	return( NUM_GRAPHS );
}

update_stats() {

	FILE *fp_loadavg;
	long rec_packet, xmt_packet, coll_packet;

	/* this fopen()/fclose() is inside update_stats() because repeated reads
	 * on /proc/loadavg produces the same output.... so I re-open it each
	 * time it is needed =SC
	 */
	sprintf( path, LOADINFO );
	if( (fp_loadavg = fopen( path, "r" )) == NULL ) {
		fprintf( stderr, "ERROR: cannot open %s\n", path );
		exit( (-1) );
	}

	/* get the cpu data for this interval */
	while( fgets( line, LINESIZE, fp_cpu ) != NULL ) {
		if( strncmp( line, "cpu ", 4 ) == 0 ) {
			sscanf( &line[5], "%ld %ld %ld %ld", &sdata.user_cpu,
				 &sdata.nice_cpu, &sdata.sys_cpu, &sdata.idle_cpu );
		}
		else if( strncmp( line, "disk ", 5 ) == 0 ) {
			long d1, d2, d3, d4;
			sscanf( &line[5], "%ld %ld %ld %ld", &d1, &d2, &d3, &d4 );
			sdata.disk_xfers = d1 + d2 + d3 + d4;
		}
		else if( strncmp( line, "page ", 5 ) == 0 ) {
			sscanf( &line[5], "%ld %ld", &sdata.page_in, &sdata.page_out );
		}
		else if( strncmp( line, "swap ", 5 ) == 0 ) {
			sscanf( &line[5], "%ld %ld", &sdata.swap_in, &sdata.swap_out );
		}
		else if( strncmp( line, "intr ", 5 ) == 0 ) {
			sscanf( &line[5], "%ld", &sdata.intr );
		}
		else if( strncmp( line, "ctxt ", 5 ) == 0 ) {
			sscanf( &line[5], "%ld", &sdata.ctxt );
		}
	}

	/* get the load data for this interval */
	if( fgets( line, LINESIZE, fp_loadavg ) != NULL ) {
		sscanf( line, "%f %f %f %d/%d",
			&sdata.load_current, &sdata.load_last, &sdata.load_prev,
			&sdata.load_active_proc, &sdata.load_proc );
	}
	fclose( fp_loadavg );

	/* get the memory data for this interval */
	if( version_major < 2 ||
	    (version_major == 2 && version_minor == 0 ) ||
	    (version_major == 2 && version_minor < 2 && version_dot < 41) ) {

		while( fgets( line, LINESIZE, fp_meminfo ) != NULL ) {
			if( strncmp( line, "Mem:", 4 ) == 0 ) {
				sscanf( &line[6], "%ld %ld %ld %ld %ld", &sdata.mem_avail,
				 	&sdata.mem_used, &sdata.mem_free, &sdata.mem_shared,
				  	&sdata.mem_buf );
				sdata.mem_avail /= 1024;
				sdata.mem_used /= 1024;
				sdata.mem_free /= 1024;
				sdata.mem_shared /= 1024;
				sdata.mem_buf /= 1024;
			}
			if( strncmp( line, "Swap:", 5 ) == 0 ) {
				sscanf( &line[6], "%ld %ld %ld", &sdata.swap_avail,
				 	&sdata.swap_used, &sdata.swap_free );
				sdata.swap_avail /= 1024;
				sdata.swap_used /= 1024;
				sdata.swap_free /= 1024;
			}
		}
	}
	else {
		while( fgets( line, LINESIZE, fp_meminfo ) != NULL ) {
			if( strncmp( line, "MemTotal:", 9 ) == 0 )
				sscanf( &line[10], "%ld", &sdata.mem_avail );
			else if( strncmp( line, "MemFree:", 8 ) == 0 )
				sscanf( &line[9], "%ld", &sdata.mem_free );
			else if( strncmp( line, "MemShared:", 10 ) == 0 )
				sscanf( &line[11], "%ld", &sdata.mem_shared );
			else if( strncmp( line, "Buffers:", 8 ) == 0 )
				sscanf( &line[9], "%ld", &sdata.mem_buf );
			else if( strncmp( line, "SwapTotal:", 10 ) == 0 )
				sscanf( &line[11], "%ld", &sdata.swap_avail );
			else if( strncmp( line, "SwapFree:", 9 ) == 0 )
				sscanf( &line[10], "%ld", &sdata.swap_free );
		}
		sdata.mem_used = sdata.mem_avail - sdata.mem_free;
		sdata.swap_used = sdata.swap_avail - sdata.swap_free;
	}

	/* get the TCP packet data for this interval */
	sdata.rec_packet = sdata.xmt_packet = sdata.coll_packet = 0;
	while( fgets( line, LINESIZE, fp_packet ) != NULL ) {
		if( line[6] == ':' && strlen( line ) > 55 ) {

			/* Here we need to deal with linux version differences */
			if( version_major >= 2 && version_minor >= 1 )
				sscanf( &line[8],
/*XXX correct is recbytes,recpackets,recerrs,recdrop,recfifo,recframe,
				 reccompressed, recmulticast,xmitbytes,xmitpackets,xmiterrs,
				 xmitdrop, xmitfifo,colls,carrier,compressed
XXX*/
				 "%*ld %ld %*ld %*ld %*ld %*ld %*ld %*ld %*ld %ld %*ld %*ld %*ld %ld",
				 		&rec_packet, &xmt_packet, &coll_packet );
			else 
				sscanf( &line[8],
				 "%ld %*ld %*ld %*ld %*ld %ld %*ld %*ld %*ld %ld",
				 		&rec_packet, &xmt_packet, &coll_packet );

			sdata.rec_packet += rec_packet;
			sdata.xmt_packet += xmt_packet;
			sdata.coll_packet += coll_packet;
		}
	}
	fseek( fp_cpu, 0, SEEK_SET );
	fseek( fp_meminfo, 0, SEEK_SET );
	fseek( fp_packet, 0, SEEK_SET );

	if( dbg ) report( &odata, &sdata );

	filldata( &odata, &sdata );
	odata = sdata;
}

static int report( od, sd )
	struct sdata *od, *sd;
{
	static int count = 0;
	long ucpu, ncpu, icpu, scpu, totcpu;
	long in, ctxt;

	if( count++ <= 0 ) return( (-1) );

	ucpu = sd->user_cpu - od->user_cpu;
	ncpu = sd->nice_cpu - od->nice_cpu;
	icpu = sd->idle_cpu - od->idle_cpu;
	scpu = sd->sys_cpu - od->sys_cpu;
	totcpu = ucpu + ncpu + icpu + scpu;

	fprintf( stderr, "\n=========== STATUS ===========\n" );
	fprintf( stderr, "user=%4.1f, nice=%4.1f, sys=%4.1f, idle=%4.1f\n",
						(float)ucpu/(float)totcpu * 100.0,
						(float)ncpu/(float)totcpu * 100.0,
						(float)scpu/(float)totcpu * 100.0,
						(float)icpu/(float)totcpu * 100.0 );

	in = sd->intr - od->intr;
	ctxt = sd->ctxt - od->ctxt;

	fprintf( stderr, "interrupts=%ld, context_switches=%ld\n", in, ctxt );
	fprintf( stderr, "%d processes, %d active\n",
			sd->load_proc, sd->load_active_proc );
	fprintf( stderr, "load average: %5.2f, %5.2f, %5.2f\n",
			sd->load_current, sd->load_last, sd->load_prev );
	fprintf( stderr, "Mem: %ld avail, %ld used, %ld free, %ld shared, %ld buf\n",
			sd->mem_avail, sd->mem_used, sd->mem_free, sd->mem_shared,
			 sd->mem_buf );
	fprintf( stderr, "Swap: %ld avail, %ld used, %ld free\n",
			sd->swap_avail, sd->swap_used, sd->swap_free );
	fprintf( stderr, "Net: packets received=%ld, transmitted=%ld, collision=%ld\n",
				sd->rec_packet - od->rec_packet,
				sd->xmt_packet - od->xmt_packet,
				sd->coll_packet - od->coll_packet );

	fprintf( stderr, "Disk xfers: %ld\n", sd->disk_xfers - od->disk_xfers );
	fprintf( stderr, "Page in: %ld, out: %ld\n",
		 sd->page_in - od->page_in, sd->page_out - od->page_out );
	fprintf( stderr, "Swap in: %ld, out: %ld\n",
		 sd->swap_in - od->swap_in, sd->swap_out - od->swap_out );

	return( 0 );
}

static int filldata( od, sd )
	struct sdata *od, *sd;
{
    static int count = 0;
    long ucpu, ncpu, icpu, scpu, totcpu;

    if( count++ <= 0 ) return( (-1) );

	ucpu = sd->user_cpu - od->user_cpu;
	ncpu = sd->nice_cpu - od->nice_cpu;
	icpu = sd->idle_cpu - od->idle_cpu;
	scpu = sd->sys_cpu - od->sys_cpu;
	totcpu = ucpu + ncpu + icpu + scpu;


	current_values[USR_CPU] = (int)((float)ucpu/(float)totcpu * 100.0);
	current_values[SYS_CPU] = (int)((float)scpu/(float)totcpu * 100.0);
	current_values[IDL_CPU] = (int)((float)icpu/(float)totcpu * 100.0);
	current_values[INTRPTS] = (int)(sd->intr - od->intr);
	current_values[CPU_SWITCH] = (int)(sd->ctxt - od->ctxt);
	current_values[LOAD_AVG] = (int)(sd->load_current *100);
	current_values[FREE_MEM] = (int)sd->mem_free;
	current_values[FREE_SWAP] = (int)sd->swap_free;

	current_values[INP_PKT] = (int)(sd->rec_packet - od->rec_packet);
	current_values[OUT_PKT] = (int)(sd->xmt_packet - od->xmt_packet);
	current_values[COL_PKT] = (int)(sd->coll_packet - od->coll_packet);
	current_values[DSK_XFR] = (int)(sd->disk_xfers - od->disk_xfers);
	current_values[PAGE_IN] = (int)(sd->page_in - od->page_in);
	current_values[PAGE_OUT] = (int)(sd->page_out - od->page_out);
	current_values[SWAP_IN] = (int)(sd->swap_in - od->swap_in);
	current_values[SWAP_OUT] = (int)(sd->swap_out - od->swap_out);
	current_values[PROCESSES] = (int)sd->load_proc;
}
