/* vim: set sw=8 ts=8 si : */
/*
 * Serial line daemon program for shutdown of your server.
 * A blinking LED indicates the server is running.
 * The program uses the serial interface lines
 * DTR (output: data terminal ready), RTS (output:request to send)  
 * and CD (input: carrier detect)
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License.
 * See http://www.gnu.org/copyleft/ for details.
 *
 * Written by Guido Socher <guido@linuxfocus.org> 
 *
 * Some ideas were taken from an earlier progam called 
 * cpanel written by Mirko Dlle <dg2fer@qsl.net>
 */
#define VERINFO "version 0.1"
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>


static int fd = 0;
enum led {rts_yellow,dtr_green,all};

/* Signal handler: all led off then exit */
void offandexit(int code)
{
	int state = 0;
	ioctl(fd, TIOCMSET, &state);
	exit(0);
}

/* Switch the LEDs on or off */
int setled( enum led whichled, int fd, int onoff, int *ledstate){
	int bitpat;
	/* the blinking LED first */
	if (whichled == rts_yellow){
		bitpat=TIOCM_RTS;
	}else if (whichled == dtr_green){
		bitpat=TIOCM_DTR;
	}else if (whichled == all){
		bitpat=(TIOCM_RTS | TIOCM_DTR);
	}
	if (onoff){
		/* on */
		*ledstate |= bitpat;
	}else{
		*ledstate &= ~bitpat;
	}
	ioctl(fd, TIOCMSET, ledstate);
	return *ledstate;
}

/* switch off green led and run shutdown 
 * the init scripts will finally kill us. We do not
 * terminate here 
 */
int runshutdown(int fd,int *ledstate)
{
	setled( dtr_green, fd, 0, ledstate);
	system("/sbin/shutdown -t2 -h now");
	return (0);
}

/* get the current state of the push button 
 * and return it. */
int getpushbutton(int fd)
{
	int state;
	ioctl(fd, TIOCMGET, &state);	/* read interface */
	if (state & TIOCM_CAR) {
		return (1);
	}
	return (0);
}

void help()
{
	printf("sled -- serial line pushbutton and LED control daemon\n\
USAGE: sled serial-device\n\
\n\
EXAMPLE:\n\
         sled /dev/ttyS0\n\
\n\
Use command setserial -g /dev/ttyS* to see available ttyS devices.\n\
This program forks it self to background. It must have permissions\n\
to execute /sbin/shutdown\n\
");
#ifdef VERINFO
	puts(VERINFO);
#endif	 
exit(0); 
}


int main(int argc, char **argv)
{
	int lstate = 0;
	int i = 0;
	pid_t pid;

	/* exactly one argument must be given */
	if (argc != 2)
		help();
	/* open device */
	fd = open(argv[1], O_RDWR | O_NDELAY);
	if (fd < 0) {
		fprintf(stderr, "ERROR: can not open \"%s\"\n", argv[1]);
		exit(2);
	}

	/* catch signals INT and TERM and switch of LEDs before 
	 * terminating */
	signal(SIGINT, offandexit);
	signal(SIGTERM, offandexit);

	/* first set a defined state, all off */
	ioctl(fd, TIOCMSET, &lstate);
	/* switch on the green LED to indicate that the 
	 * shutdown pushbutton is active */
        setled( dtr_green, fd, 1, &lstate);
	/* now we fork to background */
	if ((pid = fork())<0){
		perror("sled server fork");
		exit(1);
	}else if (pid > 0){
		/* parent dies */
		exit(0);
	}
	/* we are the child and run as a daemon */
	while(1){
		/* blink led on */
		setled( rts_yellow, fd, 1, &lstate);
		if (getpushbutton(fd)){
			runshutdown(fd,&lstate);
		}
		sleep(2);
		/* blink led off */
		setled( rts_yellow, fd,0, &lstate);
		if (getpushbutton(fd)){
			runshutdown(fd,&lstate);
		}
		sleep(1);
	}
	/* we never get here */
	return (2);
}
