/* $Header: /cvs/gnome/gIDE/src/gI_document.c,v 1.23 2000/04/22 17:23:01 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gide.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/utsname.h>
#include <netdb.h>
#include "gI_compile.h"
#include "gI_file.h"
#include "gI_edit.h"
#include "gI_common.h"
#include "gI_help.h"
#include "gI_menus.h"
#include "gI_commands.h"
#include "gI_hilite.h"

#ifdef HAVE_GTKTEXT_PATCH
#include <gtksctext.h>
#include <gtkeditor/gtkeditor.h>
#include "gI_globpatterns.h"
#endif

#ifdef HAVE_GNU_REGEXP
#include <regex.h>
#endif

/* Class signals */
enum {
	DOC_MODIFIED_SIGNAL,
	DOC_UNMODIFIED_SIGNAL,
	DOC_READONLY_SIGNAL,
	DOC_UNREADONLY_SIGNAL,
	DOC_SOURCE_SIGNAL,
	LAST_SIGNAL
};
static gint doc_signals[LAST_SIGNAL] = { 0 };

/* Prototypes */
static void gI_document_class_init (GideDocumentClass *class);
static void gI_document_init (GideDocument *document);
static void gI_document_keypressed( GtkWidget *widget,
				    GdkEventKey *event,
				    gpointer data );
static void gI_document_buttonpressed( GtkWidget *widget,
				       GdkEventButton *event,
				       gpointer data );
static void gI_document_buttonpressed_after( GtkWidget *widget,
					     GdkEventButton *event,
					     gpointer data );
static void gI_document_insert_text( GtkEditable *editable,
				     const gchar *new_text,
				     gint new_text_length,
				     gint *position,
				     gpointer data );
static void gI_document_auto_indent_text( GtkEditable *editable,
			const gchar *new_text,
			gint new_text_length,
			gint *position,
			gpointer data );
static void gI_document_delete_text( GtkEditable *editable,
				     gint start_pos, gint end_pos,
				     gpointer data );
static void gI_document_changed (GtkWidget *widget, gpointer data);
static void gI_document_destroy( GtkWidget *widget, gpointer data);
static void update_document_CB( GideDocument *doc, GtkStyle* style );

guint
gI_document_get_type()
{
	static guint document_type = 0;

	if (!document_type) {
           GtkTypeInfo document_info = {
               "GideDocument",
               sizeof (GideDocument),
               sizeof (GideDocumentClass),
               (GtkClassInitFunc) gI_document_class_init,
               (GtkObjectInitFunc) gI_document_init,
               (GtkArgSetFunc) NULL,
               (GtkArgGetFunc) NULL
             };

#if HAVE_GTKTEXT_PATCH
	   document_type = gtk_type_unique (gtk_editor_get_type (),
					      &document_info);
#else
	   document_type = gtk_type_unique (gtk_text_get_type (),
					      &document_info);
#endif
	}

         return document_type;
}

static void
gI_document_class_init( GideDocumentClass *class )
{
	GtkObjectClass *object_class;
	
	object_class = (GtkObjectClass*) class;

	doc_signals[DOC_MODIFIED_SIGNAL]
		= gtk_signal_new ("doc_modified",
				  GTK_RUN_FIRST,
				  object_class->type,
				  GTK_SIGNAL_OFFSET (GideDocumentClass,
						     modified),
				  gtk_marshal_NONE__NONE,
				  GTK_TYPE_NONE, 0);

	doc_signals[DOC_UNMODIFIED_SIGNAL]
		= gtk_signal_new ("doc_unmodified",
				  GTK_RUN_FIRST,
				  object_class->type,
				  GTK_SIGNAL_OFFSET (GideDocumentClass,
						     unmodified),
				  gtk_marshal_NONE__NONE,
				  GTK_TYPE_NONE, 0);

	doc_signals[DOC_READONLY_SIGNAL]
		= gtk_signal_new ("doc_readonly",
				  GTK_RUN_FIRST,
				  object_class->type,
				  GTK_SIGNAL_OFFSET (GideDocumentClass,
						     readonly),
				  gtk_marshal_NONE__NONE,
				  GTK_TYPE_NONE, 0);

	doc_signals[DOC_UNREADONLY_SIGNAL]
		= gtk_signal_new ("doc_unreadonly",
				  GTK_RUN_FIRST,
				  object_class->type,
				  GTK_SIGNAL_OFFSET (GideDocumentClass,
						     unreadonly),
				  gtk_marshal_NONE__NONE,
				  GTK_TYPE_NONE, 0);

	doc_signals[DOC_SOURCE_SIGNAL]
		= gtk_signal_new ("doc_source",
				  GTK_RUN_FIRST,
				  object_class->type,
				  GTK_SIGNAL_OFFSET (GideDocumentClass,
						     source),
				  gtk_marshal_NONE__POINTER,
				  GTK_TYPE_NONE, 1,
				  GTK_TYPE_POINTER);


         gtk_object_class_add_signals (object_class,
				       doc_signals,
				       LAST_SIGNAL);
	 
}

static void
gI_document_init( GideDocument *document )
{
	/* Finish up object construction */
	gtk_object_default_construct( GTK_OBJECT(document) );

	/* Initialize everything */
	document->filename = NULL;
	document->compile_window = NULL;
	document->x = 0;
	document->y = 0;
	document->read_only = 0;
	document->working = 0;
	document->last_mod = 0;
	document->changed = FALSE;
	
#ifdef HAVE_GTKTEXT_PATCH
	gtk_sctext_set_adjustments(GTK_SCTEXT(document), NULL, NULL);
#else
	gtk_text_set_adjustments(GTK_TEXT(document), NULL, NULL);
#endif

	/* Set various attributes */
	gI_text_set_editable( document, TRUE );
	gI_text_set_word_wrap( document, cfg->wordwrap );
	gI_text_set_point( document, 0 );
	
	/* Set the tab stuff */
	gI_text_default_tab_width( document ) = cfg->tab_width;
	gI_text_tab_stops( document ) = NULL;
	gI_text_tab_stops( document ) =
		g_list_prepend( gI_text_tab_stops( document ),
				(void*)cfg->tab_width );
	gI_text_tab_stops( document ) =
		g_list_prepend( gI_text_tab_stops( document ),
				(void*)cfg->tab_width );

	/* Connect signals */
	gI_document_connect_event_signals( document );			     
	gI_document_connect_text_signals( document );	

	gtk_signal_connect( GTK_OBJECT( document ), "changed",
			    GTK_SIGNAL_FUNC( gI_document_changed ),
			    NULL );
			    
	gtk_signal_connect_object(GTK_OBJECT(document),"event",
	                          GTK_SIGNAL_FUNC(gI_document_show_popup_menu),
	                          GTK_OBJECT(main_window->popup));

	gtk_signal_connect( GTK_OBJECT( document ), "destroy",
	                    GTK_SIGNAL_FUNC( gI_document_destroy ),
			    NULL );

#ifdef HAVE_GTKTEXT_PATCH
	/* Set other config related values */
	if( !cfg->highlight )
	{
		gtk_editor_hilite_when_idle( GTK_EDITOR( document ), FALSE );
	}
	if( !cfg->edtab )
	{
		GTK_EDITOR( document )->one_line_ind = NULL;
	}
#endif

	/* Assign document with current style configuration. */
	if( cfg->style )
		gtk_widget_set_style( GTK_WIDGET(document), cfg->style );
}

GtkWidget *
gI_document_new( void )
{
	GideDocument *document;
	
	document = gtk_type_new( gI_document_get_type() );

	return GTK_WIDGET( document );
}

GtkWidget *
gI_document_new_with_file( const gchar *filename, FILE *file )
{
	GideDocument *document;

	document = GIDE_DOCUMENT(gI_document_new());
	gI_document_load_file( document, filename, file );
	
	return GTK_WIDGET( document );
}

void
gI_document_load_file( GideDocument *document, gchar const *filename,
		       FILE *file )
{
	gchar buf[STRLEN];
	
	/* Change the filename */
	if (document->filename)
		g_free(document->filename);

	document->filename = g_strdup(filename);

	/* Load the file text into the buffer */
	gI_document_disconnect_text_signals( document );
	
	gI_document_clear_text(document);

    	gI_text_freeze( document );

    	while( fgets( buf, sizeof(buf), file ) ) {
        	gI_text_insert( document, NULL, NULL, NULL,
				buf, strlen( buf ) );
    	}

    	gI_text_thaw( document );

	/* Check if the doc is writable */
    	if( !get_write_stat( document->filename ) ) {
        	document->read_only = 1;
		gI_text_set_editable( document, FALSE );
    	} 

#ifdef HAVE_GTKTEXT_PATCH
	/* Hilight the buffer if possible */
	if( cfg->highlight ) {
		gI_HilitePattern *pat =
			gI_glob_lookup( g_basename (filename) );
			
		if (pat) {
	    		/* If it fails...we just return to default pattern... 
	     		 * so there's no reson to test for return value. */
	    		gI_hilite_install_pattern( GTK_EDITOR( document ),
				      pat );
	  	}
		gtk_editor_hilite_buffer( GTK_EDITOR( document ) );
      	} else {
		gtk_editor_hilite_when_idle( GTK_EDITOR( document ),
					     FALSE );
	}
#endif

	fclose(file);

	/* Store mod info */
	document->last_mod = get_last_mod( filename );
	gI_document_set_changed_state( document, FALSE );

	/* Clear the undo/redo lists */ 
	command_list_release (document->undo_commands);
	command_list_release (document->redo_commands);
	document->undo_commands = document->redo_commands = NULL;
	command_undo_redo_menu_labels( document );
	
	gI_document_connect_text_signals( document );
	
	gtk_signal_emit( GTK_OBJECT(document),
			 doc_signals[DOC_SOURCE_SIGNAL],
			 document->filename );
}

void
gI_document_save_file( GideDocument *document, gchar const *filename,
		       FILE *file )
{
	glong length, i;
	gchar *c;
	
	length = gI_text_get_length( document );
    	for(i=1;i<=length;i++) {
        	c = gtk_editable_get_chars( GTK_EDITABLE( document ), i-1, i );
        	fputc( c[0], file );
		g_free( c );
	}
		
	if ( !document->filename ||
	     (document->filename && strcmp(document->filename, filename)) ) {

		if (document->filename)
			g_free(document->filename);
		
		document->filename = g_strdup(filename);
		
		gtk_signal_emit( GTK_OBJECT(document),
				 doc_signals[DOC_SOURCE_SIGNAL],
				 document->filename );
	}
	fclose(file);
	
	document->last_mod = get_last_mod( document->filename );
	gI_document_set_changed_state( document, FALSE );
}

void
gI_document_connect_event_signals( GideDocument *document )
{
	gtk_signal_connect( GTK_OBJECT( document ), "key_press_event",
	                    GTK_SIGNAL_FUNC( gI_document_keypressed ),
			    NULL );
	gtk_signal_connect( GTK_OBJECT( document ), "button_press_event",
	                    GTK_SIGNAL_FUNC( gI_document_buttonpressed ),
			    NULL );
	gtk_signal_connect_after( GTK_OBJECT( document ), "button_press_event",
				  GTK_SIGNAL_FUNC( gI_document_buttonpressed_after ),
				  NULL );
}

void
gI_document_connect_text_signals( GideDocument *document )
{
        document->insert_id =
		gtk_signal_connect( GTK_OBJECT( document ),
				    "insert_text",
				    GTK_SIGNAL_FUNC( gI_document_insert_text ),
				    NULL ); 
	gtk_signal_connect_after( GTK_OBJECT( document ), 
				  "insert_text",
				  GTK_SIGNAL_FUNC( gI_document_auto_indent_text), 
				  NULL );
        document->delete_id = 
		gtk_signal_connect( GTK_OBJECT( document ),
				    "delete_text",
				    GTK_SIGNAL_FUNC( gI_document_delete_text ),
				    NULL ); 
}

void
gI_document_disconnect_text_signals( GideDocument *doc )
{
        gtk_signal_disconnect( GTK_OBJECT( doc ),
                               doc->insert_id );
        gtk_signal_disconnect( GTK_OBJECT( doc ),
                               doc->delete_id );
}

static void
change_menu_label (GtkWidget *menu_item,
		   char const * const prefix,
		   char const * suffix)
{
	gchar    *text;

	GtkBin   *bin = GTK_BIN(menu_item);
	GtkLabel *label = GTK_LABEL(bin->child);

	g_return_if_fail (label != NULL);

	gtk_widget_set_sensitive (menu_item, suffix != NULL);

	if (suffix == NULL)
		suffix = _("Nothing");

	/* Limit the size of the descriptor to 30 characters */
	text = g_strdup_printf ("%s : %s", prefix, suffix);

	gtk_label_set_text (label, text);

	g_free (text);
}

void
gI_document_set_undo_redo_state( GideDocument *doc,
				 char const * const undo_suffix,
				 char const * const redo_suffix)
{
	GnomeDockItem *dockitem;
	GtkWidget* widget;
	GList* children;

	change_menu_label( gI_menus_get_item_widget(main_window, "/Edit/Undo"),
			   _("Undo"), undo_suffix);
	change_menu_label( gI_menus_get_item_widget(main_window, "/Edit/Redo"),
			   _("Redo"), redo_suffix);

	if (gI_document_is_changed(doc) && !undo_suffix) 
		gI_document_set_changed_state(doc, FALSE);

	dockitem = gnome_app_get_dock_item_by_name( GNOME_APP( main_window ),
	           GNOME_APP_TOOLBAR_NAME );
	widget = gnome_dock_item_get_child(dockitem);
	for(children = GTK_TOOLBAR(widget)->children; children;
		children = children->next)
	{
		GtkToolbarChild* child;

		child = children->data;
		if(child->type == GTK_TOOLBAR_CHILD_BUTTON)
		{
			char* haha;
			gtk_label_get(GTK_LABEL(child->label), &haha);

			if(!strcmp(haha, "Undo"))
			{
				gtk_widget_set_sensitive(child->widget, undo_suffix != NULL);
			}
			else if(!strcmp(haha, "Redo"))
			{
				gtk_widget_set_sensitive(child->widget, redo_suffix != NULL);
			}
		}
	}
}

gboolean
gI_document_is_free( GideDocument *document )
{
	g_return_val_if_fail( document, FALSE );

	if (document->filename || document->changed)
		return FALSE;

	return TRUE;
}

gboolean
gI_document_is_changed( GideDocument *document )
{
	return document->changed;
}

void
gI_document_set_changed_state( GideDocument *document, gboolean state )
{
	document->changed = state;

	if (state) {
		gtk_signal_emit( GTK_OBJECT(document),
				 doc_signals[DOC_MODIFIED_SIGNAL] );
	} else {
		gtk_signal_emit( GTK_OBJECT(document),
				 doc_signals[DOC_UNMODIFIED_SIGNAL] );
	}
}

void
gI_document_set_readonly_state( GideDocument *document, gboolean state )
{
	document->read_only = state;
	gI_text_set_editable( document, !state );
		
	if (state) {
		gtk_signal_emit( GTK_OBJECT(document),
				 doc_signals[DOC_READONLY_SIGNAL] );
	} else {
		gtk_signal_emit( GTK_OBJECT(document),
				 doc_signals[DOC_UNREADONLY_SIGNAL] );
	}
}

void
gI_document_insert( GideDocument *document, GdkFont *font, GdkColor *fore,
		    GdkColor *back, const char *chars, gint length )
{
	g_return_if_fail( document != NULL );

	if( document->read_only )
	{
		return;
	}

	gI_text_insert( document, font, fore, back,
	                chars, length );
}

void
gI_document_insert_text_at_point( GideDocument *document, gchar *str )
{
	GtkEditable *widget;
	GtkArg posArg;
	gint pos;

	if( !document )
		return;

	widget = GTK_EDITABLE( document );

	if( document->read_only )
		return;

	posArg.name = "GtkEditable::text_position";
	gtk_object_getv( GTK_OBJECT(widget), 1, &posArg );

	pos = posArg.d.int_data;
	gtk_editable_insert_text( widget, str, strlen(str), &pos );
}


void
gI_document_delete( GideDocument *document, gint from, gint to )
{
	g_return_if_fail( document != NULL );
	g_return_if_fail( (!document->read_only) );
	
	gtk_editable_delete_text( GTK_EDITABLE( document ),
	                          from, to );
}

void
gI_document_delete_selection( GideDocument *document )
{
	g_return_if_fail( document );
	g_return_if_fail( !(document->read_only) );

	gtk_editable_delete_selection( GTK_EDITABLE( document ) );
}

static gint
gI_document_search_text( GideDocument *document, SearchReplaceState *state )
{
	gchar *buf;
	glong start, len, text_length;
	glong hit = 0;

	buf = (gchar *) g_malloc0( sizeof( state->search ) );
	len = strlen( state->search );
	text_length = gI_text_get_length( document );
	start = state->back ? state->start - 1 : state->start;
	
	gI_text_freeze( document );

	for( ; state->back ?  start >=0 : start<=(text_length-len);
	       state->back ? start-- : start++) {
	       
		buf = gtk_editable_get_chars( GTK_EDITABLE( document ),
					      start, start+len );

		if (buf) {
			if( state->cs )
				hit = strcmp( buf, state->search );
			else
				hit = strcasecmp( buf, state->search );

			if( hit == 0 ) {
				
				/* scroll... */
				__goto_point( start );
				gtk_editable_select_region( GTK_EDITABLE( document ),
							    start, start+len );

				if ( state->back )
					gI_text_set_point(document, start);
				else
					gI_text_set_point(document, start+len);
					
				gI_text_thaw( document );
				
				return start;
			}
			g_free( buf );
		}
	}

	gI_text_thaw( document );
	
	return -1;
}


static gboolean
gI_document_search_regex( GideDocument *document, SearchReplaceState *state )
{
#ifdef HAVE_GNU_REGEXP
	gchar *buf;
	gint flags;
	glong len,text_length;
	gchar errbuf[256];
	gchar errmsg[300];
	regex_t buffer;
	glong compile;
	gchar *doctext;
	regmatch_t regs[1];
	glong match;

	buf = (gchar *) g_malloc0( sizeof( state->search ) );
	len = strlen( state->search );
	text_length = gI_text_get_length( document );

	flags = REG_EXTENDED | REG_NEWLINE;
	if(!state->cs)
		flags |= REG_ICASE;

	/* Compile the regular expression */
	compile = regcomp( &buffer, state->search, flags);
	if( compile != 0 ) {
		regerror( compile, &buffer, errbuf, 256 );
		sprintf( errmsg, _("Regex Compile Error: %s"), errbuf );
		gI_error_dialog( errmsg );
		return -1;
	}

	/* the stuff below needs too much memory, find a better way */
	doctext = gtk_editable_get_chars( GTK_EDITABLE( document ),
					  state->start, text_length );
	match = regexec( &buffer, doctext, 1, regs, 0 );
	if( match != 0 ) {
		g_free( doctext );
		return -1;
	} else {
		/* scroll... */
		__goto_point( regs[0].rm_so + state->start );
		gtk_editable_select_region( GTK_EDITABLE( document ),
					    regs[0].rm_so + state->start,
					    regs[0].rm_eo + state->start );
		
		gI_text_set_point( document, regs[0].rm_eo+state->start );
		regfree( &buffer );
		g_free( doctext );

		return regs[0].rm_so+state->start;
	}
#else
	gI_error_dialog(
		_("Regular expression searching requires GNU regexp lib") );
	return -1;
#endif
}

gint
gI_document_search( GideDocument *document, SearchReplaceState *state )
{
	if (state->re)
		return gI_document_search_regex( document, state );
	else
		return gI_document_search_text( document, state );
}
			       
/*
 ---------------------------------------------------------------------
     Function: gI_document_clear_text()
     Desc: Clears the given Document
 ---------------------------------------------------------------------
*/

void gI_document_clear_text( GideDocument *document )
{
	gint i;
	gint flag;

	i = gI_text_get_length( document );
	if ( i > 0 ) {
		flag = document->changed;
		document->changed = TRUE;
		gtk_editable_delete_text( GTK_EDITABLE( document ), 0, i );
		document->changed = flag;
	}
}

/*
 ---------------------------------------------------------------------
     Function: gI_document_track_movement()
     Desc: Tracks the movement in document
 ---------------------------------------------------------------------
*/
void
gI_document_track_movement( GideDocument *document )
{
	gint height = 14, width = 7;

	g_return_if_fail(document);
	
	document->x = gI_text_last_ver_value( document) +
		gI_text_cursor_pos_y( document ) / height;
	document->y = gI_text_cursor_pos_x( document ) / width;
}

static void
gI_document_buttonpressed( GtkWidget *widget, GdkEventButton *event,
			   gpointer data )
{
	GideDocument *document;

	document = GIDE_DOCUMENT(widget);
	
	check_current_doc();
}

static void
gI_document_buttonpressed_after( GtkWidget *widget, GdkEventButton *event,
				 gpointer data )
{
	GideDocument *document;

	document = GIDE_DOCUMENT(widget);
	
        /* jump to cursor mark */
        gI_text_set_point( document,
			   gtk_editable_get_position( GTK_EDITABLE( document ) ) );

        gI_document_track_movement( document );
        //gI_window_set_statusbar( document );
}

/*
 ---------------------------------------------------------------------
     Function: gI_document_keypressed()
     Desc: This function is called, when a key is pressed
 ---------------------------------------------------------------------
*/

static void
gI_document_keypressed( GtkWidget *widget, GdkEventKey *event,
			gpointer data )
{
	GideDocument *document;

	document = GIDE_DOCUMENT(widget);
	
	gI_document_track_movement( document );

	//gI_window_set_statusbar( document );
}

static void
gI_document_insert_text( GtkEditable *editable,
			 const gchar *new_text,
			 gint new_text_length,
			 gint *position,
			 gpointer data )
{
	gchar *text;

	text = g_new0( gchar, new_text_length );
	strncpy(text, new_text, new_text_length);
	
	cmd_insert_text( GIDE_DOCUMENT(editable), text,
			 *position,((*position)+new_text_length) );

	g_free(text);
}

static void 
gI_document_auto_indent_text( GtkEditable *editable,
			const gchar *new_text,
			gint new_text_length,
			gint *position,
			gpointer data )
{
	gint i, newlines, newline_1;
	gchar *buffer, *whitespace;
	GideDocument *document;

	document = GIDE_DOCUMENT(editable);

	/* just a little error checking */
	if ((new_text_length != 1) || (new_text[0] != '\n'))
		return ;
	
	/* count how many lines we have before our insertion point */
	newlines = newline_1 = 0;
	for (i = *position; i > 0; i--)
	{
		buffer = gtk_editable_get_chars (GTK_EDITABLE (editable), i-1, i);
		if (buffer == NULL)
			continue;
		if (buffer[0] == 10)
		{
			if (newlines > 0)
			{
				g_free (buffer);
				buffer = NULL;
				break;
			}
			else {
				newlines++;
				newline_1 = i;
				g_free (buffer);
				buffer = NULL;
			}
		}
		g_free (buffer);
		buffer = NULL;
	}
	

	/* look at the line before the new insertion point and while the characters are 
	   tabs (9) or spaces (32) add them to the whitespace buffer */
	whitespace = g_malloc0 (newline_1 - i + 2);
	for (i = i; i <= newline_1; i++)
	{
		buffer = gtk_editable_get_chars (GTK_EDITABLE (editable), i, i+1);
		if ((buffer[0] != 32) & (buffer[0] != 9))
		{
			g_free (buffer);
			buffer = NULL;
			break;
		}
		strncat (whitespace, buffer, 1);
		g_free (buffer);
	}

	/* insert the whitespace */
	if (strlen(whitespace) > 0)
	{
		gI_document_insert_text_at_point( document, whitespace );
	}

	g_free (whitespace);

}

static void
gI_document_delete_text( GtkEditable *editable,
			 gint start_pos, gint end_pos,
			 gpointer data )
{
	gchar *text;
	
	text = gtk_editable_get_chars( editable, start_pos, end_pos );

	cmd_delete_text( GIDE_DOCUMENT(editable), text,
			 start_pos, end_pos );
}

static void
gI_document_changed (GtkWidget *widget, gpointer data)
{
	GideDocument *document;

	document = GIDE_DOCUMENT(widget);

	gI_document_set_changed_state( document, TRUE );	
}

static void
gI_document_destroy( GtkWidget *widget, gpointer data)
{
	GideDocument *document;

	document = GIDE_DOCUMENT(widget);

	if( document->filename )
		g_free(	document->filename );
	
	command_list_release (document->undo_commands);
	command_list_release (document->redo_commands);
}

/*
----------------------------------------------------------------------
Function gI_document_show_popup_menu
----------------------------------------------------------------------
*/
gint gI_document_show_popup_menu(GtkWidget *widget,GdkEvent *ev)
{
	if(ev->type==GDK_BUTTON_PRESS)
	{
		GdkEventButton *event=(GdkEventButton*)ev;
		if(event->button==3)
		{
			gnome_popup_menu_do_popup_modal(widget, NULL, NULL, event,
				main_window);

			return TRUE;
		}
	}
	return FALSE;
}


/*
-----------------------------------------------------------------------
Function: gl_document_cmd_swaphc
-----------------------------------------------------------------------
*/
void gI_document_cmd_swaphc(GtkWidget *widget)
{
	size_t len;
	gchar *newfname;
	GideWindow *win = main_window;
	GideDocument *doc;
	newfname=0;
	doc=gI_window_get_current_doc(win);
	if(!doc || !doc->filename)
		return;

	len=strlen(doc->filename);
	while(len){
		if(doc->filename[len]=='.')
			break;

		len--;
	};
	if(strcasecmp(&doc->filename[len],".h")==0) {
		len++;
		newfname=g_malloc(len+4);
		strcpy(newfname,doc->filename);  /* copy original filename */
		strcpy(&newfname[len],"cc");
		if(!file_check_if_exist(newfname,FALSE))  {
			strcpy(&newfname[len],"cpp");
			if(!file_check_if_exist(newfname,FALSE))  {
				newfname[len+1]=0;    /* test if *.c exist */
				if(!file_check_if_exist(newfname,TRUE)) {
					g_free(newfname);    /* must free up before return */
					return ;
				}
			}
		}
	}
	else if(strncasecmp(&doc->filename[len],".c",2)==0 ) {
		len++;
		newfname=g_strdup(doc->filename);
		newfname[len]='h';
		newfname[len+1]=0;
		if(!file_check_if_exist(newfname,TRUE)) {
			g_free(newfname);
			return ;
		}
	}

	if(newfname)
	{
		if( !check_file_open( newfname ) )
		{
			file_open_by_name(win,newfname);
		}
		else
		{
			goto_file( newfname );
		}
		g_free( newfname );
	}
}

/*
-----------------------------------------------------------------------
Function: gl_document_cmd_open_file_at_line
-----------------------------------------------------------------------
*/
void gI_document_cmd_open_file_at_line(GtkWidget *widget)
{
	glong i;
	GideWindow *win = main_window;
	GideDocument *current;
	gchar *line;
	gchar filename[MAXLEN];
	gchar *inc[64];
	gchar *fwp;
	gI_comp_set *cset = NULL;

	current = gI_window_get_current_doc( win );
	if( !current )
		return;

	if( !GTK_EDITABLE( current )->has_selection )
	{
		gI_error_dialog( _("No selection in current document!") );
		return;
	}

	/* get selection */
	if( GTK_EDITABLE( current )->selection_end_pos
	        < GTK_EDITABLE( current )->selection_start_pos )
	{
		line = gtk_editable_get_chars( GTK_EDITABLE( current ),
		                               GTK_EDITABLE( current )->selection_end_pos,
		                               GTK_EDITABLE( current )->selection_start_pos );
	}
	else
	{
		line = gtk_editable_get_chars( GTK_EDITABLE( current ),
		                               GTK_EDITABLE( current )->selection_start_pos,
		                               GTK_EDITABLE( current )->selection_end_pos );
	}

	if( !line )
		return;

	if( !strstr( line, "#include" ) )
	{
		fwp = g_strconcat( getcwd( NULL, STRLEN ), "/", line, NULL );
		if( !check_file_open( fwp ) )
			file_open_by_name( win, fwp );
		else
			goto_file( fwp );
		g_free( line );
		g_free( fwp );
		return;
	}

	if( current->filename )
		cset = get_comp_set_from_filename( current->filename );

	if( strchr( line, '<' ) && strchr( line, '>' ) )
	{
		strncpy( filename, SK_GetBetween( line, '<', '>' ), MAXLEN );

		/* search includes path's, if available */
		if( cset || cfg->incpath )
		{
			if( cset )
			{
				SK_GetFields( cset->incpath, inc, ':' );
			}
			else
			{
				if( cfg->incpath )
				{
					SK_GetFields( cfg->incpath, inc, ':' );
				}
				else
				{
					gI_error_dialog( _("Can't find the included file in the specified paths") );

					g_free( line );
					return;
				}
			}

			for(i=1;i<=atoi(inc[0]);i++)
			{
				if( inc[i] != NULL )
				{
					fwp = g_strconcat( inc[i], "/", filename, NULL );
					if( file_exist( fwp ) )
					{
						if( !check_file_open( fwp ) )
							file_open_by_name( win, fwp );
						else
							goto_file( fwp );
						g_free( fwp );
						g_free( line );
						return;
					}
				}
			}
		}

		gI_error_dialog( _("Can't find the included file in the specified paths") );

		g_free( line );
		return;
	}

	if( strchr( line, '"' ) )
	{
		strncpy( filename, SK_GetBetween( line, '"', '"' ), MAXLEN );
		fwp = g_strconcat( getcwd( NULL, STRLEN ), "/", filename, NULL );
		if( !check_file_open( fwp ) )
			file_open_by_name( win, fwp );
		else
			goto_file( fwp );
		g_free( line );
		g_free( fwp );
		return;
	}

	g_free( line );
}

static void special_mail_doc_ok( GtkWidget *widget, GtkWidget *entry )
{
	GideDocument *current;
	gchar *to = NULL, *smtp = NULL, *email = NULL;
	struct sockaddr_in socks;
	struct sockaddr_in sockd;
	gint skfd;
	struct utsname uts;
	/*	struct hostent *hostent = NULL;
		gchar *hwd;*/
	gchar *buf;
	/*	gchar *user;*/
	glong psize = 0;

	current = gI_window_get_current_doc( main_window );
	if( !current )
		return;

	to = gtk_entry_get_text( GTK_ENTRY( entry ) );

	if( !cfg->smtp || isempty( cfg->smtp ) )
		smtp = g_strdup( "127.0.0.1" );
	else
		smtp = g_strdup( cfg->smtp );

	if( !cfg->email || isempty( cfg->email ) )
		email = getenv( "USER" );
	else
		email = cfg->email;

	skfd = socket( AF_INET, SOCK_STREAM, IPPROTO_TCP );
	if( skfd < 0 )
	{
		perror( "socket" );
	}

	if( uname( &uts ) < 0 )
	{
		perror( "uname" );
	}

	/***
	printf("nodename: %s\n", uts.nodename );
	#ifdef __USE_GNU
	printf("domainname: %s\n", uts.domainname );
	#else
	printf("domainname: %s\n", uts.__domainname );
	#endif

	#ifdef __USE_GNU
		hwd = g_strconcat( uts.nodename, ".", uts.domainname, NULL );
	#else
		hwd = g_strconcat( uts.nodename, ".", uts.__domainname, NULL );
	#endif	

		hostent = gethostbyname( hwd );
	#ifdef __USE_GNU
		if( !hostent && !strcmp( uts.domainname, "(none)" ) )
	#else
		if( !hostent && !strcmp( uts.__domainname, "(none)" ) )
	#endif
		{
			hostent = gethostbyname( uts.nodename );			
		}

		g_free( hwd );

		if( !hostent )
		{
			error_dialog( "\n    Unable to determe local address!    \n", "Address Lookup Error" );
			return;
		}	
	***/

	socks.sin_family = AF_INET;
	socks.sin_port = htons( 0 );
	socks.sin_addr.s_addr = inet_addr( "0.0.0.0" );
	if( bind( skfd, (struct sockaddr *) &socks, sizeof( struct sockaddr_in ) ) < 0 )
		{
			perror( "bind" );
		}

	sockd.sin_family = AF_INET;
	sockd.sin_port = htons( 25 );
	sockd.sin_addr.s_addr = inet_addr( smtp );

	g_free( smtp );

	if( connect( skfd, (struct sockaddr *) &sockd, sizeof( struct sockaddr_in ) ) < 0 )
		{
			buf = g_strconcat( _("Connect Error: "), strerror(errno), NULL );
			gI_error_dialog( buf);
			g_free( buf );
			return;
		}

	/***printf( "connected\n" );***/

	/* process gtk stuff */
	while( gtk_events_pending() || gdk_events_pending() )
		gtk_main_iteration();

	buf = g_strconcat( "HELO", " ", uts.nodename, "\n", NULL );
	write( skfd, buf, strlen( buf ) );
	g_free( buf );

	/***
	#ifdef __USE_GNU
		if( !strcmp( uts.domainname, "(none)" ) )
	#else
		if( !strcmp( uts.__domainname, "(none)" ) )
	#endif
			buf = g_strconcat( "MAIL FROM: ", user, "@", uts.nodename, "\n", NULL ); 
		else
	#ifdef __USE_GNU
			buf = g_strconcat( "MAIL FROM: ", user, "@", uts.nodename, ".", uts.domainname, "\n", NULL ); 
	#else
			buf = g_strconcat( "MAIL FROM: ", user, "@", uts.nodename, ".", uts.__domainname, "\n", NULL ); 
	#endif
	***/
	buf = g_strconcat( "MAIL FROM: ", email, "\n", NULL );

	write( skfd, buf, strlen(buf ) );
	g_free( buf );

	buf = g_strconcat( "RCPT TO: ", to, "\n", NULL );
	write( skfd, buf, strlen(buf ) );
	g_free( buf );

	buf = g_strconcat( "DATA\n", NULL );
	write( skfd, buf, strlen(buf ) );
	g_free( buf );

	buf = g_strconcat( "To: ", to, "\n", NULL );
	write( skfd, buf, strlen(buf ) );
	g_free( buf );

	if( current->filename )
		buf = g_strconcat( "Subject: ", current->filename, "\n", NULL );
	else
		buf = g_strconcat( "Subject: File\n", NULL );
	write( skfd, buf, strlen(buf ) );
	g_free( buf );


	for(psize=0;psize<(gI_text_get_length( current ));psize++)
	{
		buf = gtk_editable_get_chars( GTK_EDITABLE( current ),
		                              psize,
		                              psize+1 );

		write( skfd, buf, 1 );

		g_free( buf );

		gtk_main_iteration_do( FALSE );
	}

	buf = g_strconcat( ".\n", NULL );
	write( skfd, buf, 2 );
	g_free( buf );

	gI_ok_dialog( _("Mail sent.") );
}


void special_mail_doc( GtkWidget *widget, gpointer data )
{
	GideDocument *current;

	current = gI_window_get_current_doc( main_window );
	if( !current )
		return;

	entry_dialog( _("Send To:  "), _("Mail Document"), special_mail_doc_ok );
}


void show_manpage( GtkWidget *widget, gpointer data )
{
	GideDocument *current;

	current = gI_window_get_current_doc( main_window );
	if( !current )
		return;

	if( GTK_EDITABLE( current )->has_selection )
	{
		gchar *mp;

		mp = gtk_editable_get_chars( GTK_EDITABLE( current ),
		                             GTK_EDITABLE( current )->selection_start_pos,
		                             GTK_EDITABLE( current )->selection_end_pos );

		gman( mp, NULL );

		g_free( mp );
	}
	else
	{
		gchar *mp;
		gint point, noc;
		gchar *ptr;

		/* get the next gchars, max. 50 */
		point = gI_text_get_point( current );
		if( point+50 > gI_text_get_length( current ) )
		{
			noc = gI_text_get_length( current ) - point;
		}
		else
		{
			noc = 50;
		}

		mp = gtk_editable_get_chars( GTK_EDITABLE( current ),
		                             point, point+noc  );

		ptr = mp;

		while( *ptr != ' ' && *ptr != '(' && *ptr != ';' )
			ptr++;

		*ptr = '\0';

		if( !isempty( mp ) )
		{
			gman( mp, NULL );
		}

		g_free( mp );
	}
}

void special_insert_file( GtkWidget *widget, gpointer data )
{
	GideDocument *current;
	GList *files = NULL;
	FILE *f;
	gchar *filename;
	gchar buf[STRLEN];

	current = gI_window_get_current_doc( main_window );
	if( !current )
		return;

	files = gI_file_sel_new( _("Insert File..."), FALSE, FALSE );
	if( files && files->data )
	{
		filename = (gchar *) files->data; 

		f = fopen( filename, "r" );
	        if( !f )
	        {
	       		g_snprintf( buf, sizeof(buf), _("File Error: %s"), strerror( errno ) );
		        gI_error_dialog( buf);
		        return;
		}

	        while( fgets( buf, sizeof(buf), f ) )
       		{
	        	gI_document_insert( current, NULL, NULL, NULL, buf, strlen( buf ) );
     		}

	        fclose( f );
	}
}

void
special_remove_current_file( GtkWidget *widget, gpointer data )
{
	GideDocument *current;
	gchar errmsg[200];

	current = gI_window_get_current_doc( main_window );
	if( !current )
		return;

	if( !current->filename )
		return;

	if( remove( current->filename ) < 0 )
	{
		g_snprintf( errmsg, sizeof(errmsg), _("File Error: %s"), strerror( errno ) );
		gI_error_dialog( errmsg);
		return;
	}

	check_current_doc();
}

/* this is for g_list_foreach callback function */
static void update_document_CB( GideDocument *doc, GtkStyle* style )
{
	if ( doc )
		gtk_widget_set_style( GTK_WIDGET(doc), style );
}

void update_documents_style(  )
{
	g_list_foreach( main_window->documents, (GFunc) update_document_CB, cfg->style );
}



void check_current_doc(void)
{
	GideDocument *current;
	glong ret;
	gchar msg[STRLEN];

	current = gI_window_get_current_doc( main_window );
	if( !current )
		return;

	if( !current->filename )
		return;

	ret = check_doc_changed( current );

	switch( ret )
	{
	case 0:
		{
			break;
		}

	case 1:
		{
			if( current->changed )
			{
				g_snprintf( msg, sizeof(msg), _("The file\n'%s'\nhas been changed from outside the editor!\nIf You reload, You'll loose the changes You did in gIDE!\nDo You really want to reload it?"), current->filename );
			}
			else
			{
				g_snprintf( msg, sizeof(msg), _("The file\n'%s'\nhas been changed from outside the editor!\nDo You want to reload it?"), current->filename );
			}

			if( gI_ask_dialog( msg) == 0) /* YES */
			{
				file_reload( NULL, main_window );
				break;
			}
			else
			{
				gI_document_set_changed_state( current, TRUE );
				break;
			}
		}

	case 2:
		{
			g_snprintf( msg, sizeof(msg), _("The file\n'%s'\nis not longer available.\nChoose YES to close it, or NO to keep it!"), current->filename );
			if( gI_ask_dialog( msg) == 0) /* YES */
			{
				file_close( NULL, main_window );
				break;
			}
			else
			{
				gI_document_set_changed_state( current, TRUE );
				break;
			}
		}
	}
}












