/* $Header: /cvs/gnome/gIDE/src/gI_file.c,v 1.16 2000/04/16 04:59:56 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include "gide.h"
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>

#include "gI_file.h"
#include "gI_compile.h"
#include "gI_common.h"
#include "gI_edit.h"
#include "gI_menus.h"
#include "gI_project.h"

#ifdef HAVE_GTKTEXT_PATCH
#include <gtksctext.h>
#include <gtkeditor/gtkeditor.h>
#include "gI_globpatterns.h"
#endif


/* globals */
static GtkWidget *print_window = NULL;
static GtkWidget *e_print_other;
static glong print_lpr = TRUE;
static GtkWidget *r_print_all;
static GtkWidget *r_print_sel;

/* externs */
extern gchar gide_path[];

#ifdef HAVE_GTKTEXT_PATCH
/* declared in gtkeditor-regex.h and syntaxtable.h */
void            _gtk_editor_destroy_patterns   (GtkEditorHilitePatterns *patterns);
void            _gtk_editor_destroy_stable (GtkEditorSyntaxTable *stable);
#endif

/* Prototypes */
static void file_save_by_name( GideWindow *window, GideDocument *document,
			       gchar *filename );
static gint file_close_dialog( GideWindow *window, GideDocument *current );
static gboolean file_exit_is_changed( GideWindow *window );

/*
 ---------------------------------------------------------------------
     Function: file_strip_name()
     Desc: strip path from filename (unix-style)
 ---------------------------------------------------------------------
*/
gchar *file_strip_name( gchar *filename )
{
	return g_basename(filename);
}

static void
file_launch_external ( gchar *filename )
{
    	gchar launch[STRLEN];
	gchar *ptr;
	
	strcpy( launch, cfg->redit );
	ptr = strstr( launch, "%s" );
	if( ptr ) {
		*ptr = '\0';
		if ( filename ) {
			strcat( launch, filename );
			ptr = strstr( cfg->redit, "%s" );
			ptr++;ptr++;
			strcat( launch, ptr );
		}
	}
	strcat( launch, " &" );
	system( launch );
}


/*
 ---------------------------------------------------------------------
     Function: file_new()
     Desc: Callback-Function /File/New
 ---------------------------------------------------------------------
*/
void
file_new( GtkWidget *widget, gpointer data )
{
	GideDocument *document;
    
	g_assert(IS_GIDE_WINDOW(data));

	if( cfg->use_redit ) {
		file_launch_external( NULL );
	} else {
		document = GIDE_DOCUMENT( gI_document_new() );
		gI_window_add_doc( GIDE_WINDOW(data), document );
	}
}


/*
 ---------------------------------------------------------------------
     Function: file_open()
     Desc: Callback-Function /File/Open
 ---------------------------------------------------------------------
*/
void
file_open( GtkWidget *widget, gpointer data )
{
	GList *files = gI_file_sel_new( _("Open File..."), FALSE, TRUE );

	g_assert(IS_GIDE_WINDOW(data));

	while( files ) {
		gchar *filename = (gchar *) files->data;

		file_open_by_name( GIDE_WINDOW(data), filename );

		files = g_list_next( files );
	}

	gI_file_sel_free_list( files );
}


void
file_open_by_name( GideWindow *window, gchar *filename )
{
    	FILE *file;
    	gchar buf[STRLEN];
    	GideDocument *document;

    	/* only open the file, if its not already opened */
	if ( gI_window_is_open_doc(window, filename)) {
		gI_window_goto_doc_by_file(window, filename);
		return;
	}
	
    	/* If the user chose an external editor */
    	if( cfg->use_redit ) {
		file_launch_external( filename );
		return;
    	}
	
	file = fopen( filename, "r" );
    	if ( !file ) {
        	g_snprintf( buf, STRLEN, _("Unable to Open '%s'"), filename );
        	gI_error_dialog( buf );
        	return;
    	}

	document = gI_window_get_current_doc( window );
    	if ( !document || !gI_document_is_free(document) ) {
		document = GIDE_DOCUMENT( gI_document_new() );
		gI_window_add_doc( window, document );
		gI_document_load_file( document, filename, file );
    	} else {
		gI_document_load_file(document, filename, file);
	}
}

/*
 ---------------------------------------------------------------------
     Function: file_reload()
     Desc: Callback-Function /File/Revert
 ---------------------------------------------------------------------
*/
void
file_reload( GtkWidget *widget, gpointer data )
{
	GideDocument *current;
	gchar *filename;
	FILE *file;
	gchar buf[STRLEN];
	gint pos;

	g_assert(IS_GIDE_WINDOW(data));

	current = gI_window_get_current_doc( GIDE_WINDOW(data) );
	
	if( !current )
		return;

    	if( !current->filename )
        	return;

	if ( !gI_document_is_changed(current) )
		return;
		
	if( gI_ask_dialog( _("The file has been changed,\nDo You want to reload it?")) != 0 )  /* 0 == GNOME_YES */
	{
		return;
	}

	pos = gI_text_get_point( current );
	
    	file = fopen( current->filename, "r" );
    	if( !file ) {
        	g_snprintf( buf, STRLEN,
			    _("Unable to Open '%s'"), current->filename );
        	gI_error_dialog( buf);
        	return; 
    	}

	filename = g_strdup( current->filename );
	gI_document_load_file(current, filename, file);
	g_free( filename );

    	if( gI_text_get_length( current ) >= pos )
        	gI_text_set_point( current, pos );
}


/*
 ---------------------------------------------------------------------
     Function: file_save()
     Desc: Callback-Function /File/Save
 ---------------------------------------------------------------------
*/
void
file_save( GtkWidget *widget, gpointer data )
{
	FILE *file;
	GideWindow *window;
	GideDocument *document;

	g_assert(IS_GIDE_WINDOW(data));

	window = GIDE_WINDOW(data);
	
	document = gI_window_get_current_doc( window );
	if( !document )
		return;

    	if( !document->filename ) {
        	file_save_as( widget, data );
        	return;
    	}
    
    	file = fopen( document->filename, "w" );
    	if( !file ) {
        	gI_error_dialog( _("Unable to Save File") );
        	return;
    	}

	gI_document_save_file( document, document->filename, file );
	
    	gI_window_set_statusbar( window );
}


/*
 ---------------------------------------------------------------------
     Function: file_save_as()
     Desc: Callback-Function /File/Save As
 ---------------------------------------------------------------------
*/
void file_save_as( GtkWidget *widget, gpointer data )
{
	GideWindow *window;
	GideDocument *document;
	GList *files;

	g_assert(IS_GIDE_WINDOW(data));

	window = GIDE_WINDOW( data );
	document = gI_window_get_current_doc( window );	
	
	files = gI_file_sel_new( _("Save As..."), TRUE, FALSE );

	if( files ) {
		gchar *filename = files->data;
		if( !filename )
			return;

	        file_save_by_name( window, document, filename );
	} else {
		return;
	}	

	gI_file_sel_free_list( files );
}

static void
file_save_by_name( GideWindow *window, GideDocument *document,
		   gchar *filename )
{
	FILE *file;
	gchar warnmsg[STRLEN];
	gint btn;

	if( file_exist( filename ) != 0 ) {
		/* overwrite dialog */
		sprintf( warnmsg,
			 _("The File %s already exists. Overwrite it?"),
			 filename );
	        btn = gI_ask_dialog( warnmsg);
        
        	switch (btn) {
			/* YES */
			case 0:
			    	break;

			/* NO, CANCEL or CLOSE */
	    		case 1:
	    		case 2:
	    		case -1:
				return;
				break;
		}
	}

	file = fopen( filename, "w" );
	if( !file ) {
        	gI_error_dialog( _("Unable to Save File") );
        	return;
    	}

	gI_document_save_file( document, filename, file );
}

static void
print_destroy( GtkWidget *widget, gpointer data )
{
	gtk_widget_destroy( print_window );
	print_window = NULL;
}


static void
print_show_lpr( GtkWidget *widget, gpointer data )
{
	gtk_widget_set_sensitive( e_print_other, FALSE );
	print_lpr = TRUE;
}


static void print_show_other( GtkWidget *widget, gpointer data )
{
	gtk_widget_set_sensitive( e_print_other, TRUE );
	print_lpr = FALSE;
}


static void _file_print( GtkWidget *widget, gpointer data )
{
	FILE *file;
	GideDocument *current;
	glong length,i;
	gchar filename[MAXLEN];
	gchar tmpstring[MAXLEN];
	gchar _tmpstring[MAXLEN];
	gchar *ptr;
	gchar *var_ptr;
	gchar *c;

	current = gI_window_get_current_doc( main_window );
	g_return_if_fail( current != NULL );

	if( print_lpr )
    	{
        	/* print using lpr */
        	if( current->changed || current->filename == NULL || GTK_TOGGLE_BUTTON( r_print_sel )->active )
        	{
            		/* strcat still vuln. to overflows, write ANSI-comp. (SK_)strncat */
            		g_snprintf( filename, MAXLEN, "%s/", cfg->tmpdir );
            		if( current->filename != NULL )
                		strcat( filename, file_strip_name( current->filename ) );

			g_snprintf( tmpstring, MAXLEN, ".print.%d%d", (int) time(0), (int)getpid() );
            		strcat( filename, tmpstring );

            		file = fopen( filename, "w" );
            		if( file == NULL )
            		{
                		/* error dialog */
                		g_snprintf( tmpstring, MAXLEN, _("Unable to Open '%s'"), filename );
                		gI_error_dialog( tmpstring);
                		return;
            		}

            		if( GTK_TOGGLE_BUTTON( r_print_sel )->active )
            		{
                		if( !GTK_EDITABLE( current )->has_selection )
                		{
                    			gI_error_dialog( _("No selection in current document!") );
                    			return;
                		}
                
                		for(i=GTK_EDITABLE( current )->selection_start_pos+1;
                    		    i<=GTK_EDITABLE( current )->selection_end_pos;
                    		    i++)
				{
                    			c = gtk_editable_get_chars( GTK_EDITABLE( current ), i-1, i );
					fputc( c[0], file );
					g_free( c );
				}
            		}

            		if( GTK_TOGGLE_BUTTON( r_print_all )->active )
            		{
                		length = gI_text_get_length( current );

    				for(i=1;i<=length;i++)
				{
			        	c = gtk_editable_get_chars( GTK_EDITABLE( current ), i-1, i );
			        	fputc( c[0], file );
					g_free( c );
				}
            		}

            		fclose( file );

            		g_snprintf( tmpstring, MAXLEN, "lpr %s", filename );
            		system( tmpstring );

            		remove( filename );
       		}
        	else
        	{
            		g_snprintf( tmpstring, MAXLEN, "lpr %s", current->filename );
            		system( tmpstring );
        	}

		print_destroy( NULL, NULL );
    	}
    	else
    	{
        	/* print using specified command */
        	ptr = gtk_entry_get_text( GTK_ENTRY( e_print_other ) );
        	if( !ptr || isempty( ptr ) )
        	{
            		return;
        	}

       		/* parse for "file variable" */
        	var_ptr = strstr( ptr, "%s" );
        	if( !var_ptr )
        	{
            		return;
        	}

        	*var_ptr = ' ';
        	var_ptr++;
        	*var_ptr = ' ';

        	var_ptr--;

        	/* backup the line */
        	strcpy( _tmpstring, var_ptr );

        	*var_ptr = '\0';

        	if( current->changed || current->filename == NULL )
        	{
            		/* write to temp file */
            		g_snprintf( filename, MAXLEN, "%s/", cfg->tmpdir );
            		g_snprintf( tmpstring, MAXLEN, ".print.%d%d", (int) time(0), (int) getpid() );
            		strcat( filename, tmpstring );

            		file = fopen( filename, "w" );
            		if( file == NULL )
            		{
                		/* error dialog */
                		g_snprintf( tmpstring, MAXLEN, _("Unable to Open '%s'"), filename );
                		gI_error_dialog( tmpstring);
                		return;
            		}

            		if( GTK_TOGGLE_BUTTON( r_print_all )->active )
            		{
                		length = gI_text_get_length( current );

    				for(i=1;i<=length;i++)
				{
			        	c = gtk_editable_get_chars( GTK_EDITABLE( current ), i-1, i );
			        	fputc( c[0], file );
					g_free( c );
				}
            		}

            		fclose( file );

            		/* build string */
            		strcpy( tmpstring, ptr );
            		strcat( tmpstring, filename );
            		strcat( tmpstring, _tmpstring );

            		/* execute */
            		system( tmpstring );

            		remove( filename );
        	}
        	else
        	{
            		/* build string */
            		strcpy( tmpstring, ptr );
            		strcat( tmpstring, current->filename );
            		strcat( tmpstring, _tmpstring );

            		/* execute */
            		system( tmpstring );
        	}

        	print_destroy( NULL, NULL );
    	}
}


/*
 ---------------------------------------------------------------------
     Function: file_print()
     Desc: Callback-Function /File/Print
 ---------------------------------------------------------------------
*/
void file_print( GtkWidget *widget, gpointer data )
{
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *hsep;
	GtkWidget *button;
	GSList *type_group;
	GtkWidget *label;
 	GtkWidget *r_print_lpr;
    	GtkWidget *r_print_other;
    	GSList *print_group;

    	if( print_window )
        	return;

    	print_window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
    	gtk_widget_set_usize( print_window, 300, 260 );
    	gtk_window_set_title( GTK_WINDOW( print_window ), _("Print...") );
    	gtk_signal_connect( GTK_OBJECT( print_window ), "destroy",
                            GTK_SIGNAL_FUNC( print_destroy ), NULL );

    	vbox = gtk_vbox_new( FALSE, 0 );
    	gtk_container_add( GTK_CONTAINER( print_window ), vbox );
    	gtk_widget_show( vbox );

    	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 10 );
    	gtk_widget_show( hbox );

    	label = gtk_label_new( _("Print Using: ") );
    	gtk_box_pack_start( GTK_BOX( hbox ), label, FALSE, TRUE, 5 );
    	gtk_widget_show( label );

    	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
    	gtk_widget_show( hbox );

    	r_print_lpr = gtk_radio_button_new_with_label( NULL, "lpr" );
    	type_group = gtk_radio_button_group( GTK_RADIO_BUTTON( r_print_lpr ) );
    	gtk_box_pack_start( GTK_BOX( hbox ), r_print_lpr, FALSE, TRUE, 5 );
    	gtk_signal_connect( GTK_OBJECT( r_print_lpr ), "clicked",
			    GTK_SIGNAL_FUNC( print_show_lpr ), NULL );
    	gtk_widget_show( r_print_lpr );

    	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
    	gtk_widget_show( hbox );

    	r_print_other = gtk_radio_button_new_with_label( type_group, _("Command") );
    	gtk_box_pack_start( GTK_BOX( hbox ), r_print_other, FALSE, TRUE, 5 );
    	gtk_signal_connect( GTK_OBJECT( r_print_other ), "clicked",
                            GTK_SIGNAL_FUNC( print_show_other ), NULL );
    	gtk_widget_show( r_print_other );

    	e_print_other = gtk_entry_new_with_max_length( 255 );
    	gtk_box_pack_start( GTK_BOX( hbox ), e_print_other, FALSE, TRUE, 10 );
    	gtk_widget_show( e_print_other );
    	gtk_widget_set_sensitive( e_print_other, FALSE );

    	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 10 );
    	gtk_widget_show( hbox );

   	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 10 );
    	gtk_widget_show( hbox );

    	label = gtk_label_new( _("Print Area: ") );
    	gtk_box_pack_start( GTK_BOX( hbox ), label, FALSE, TRUE, 5 );
    	gtk_widget_show( label );
    
    	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
    	gtk_widget_show( hbox );
    
    	r_print_all = gtk_radio_button_new_with_label( NULL, "all" );
    	print_group = gtk_radio_button_group( GTK_RADIO_BUTTON( r_print_all ) );
    	gtk_box_pack_start( GTK_BOX( hbox ), r_print_all, FALSE, TRUE, 5 );
    	gtk_widget_show( r_print_all );

    	r_print_sel = gtk_radio_button_new_with_label( print_group, "selection" );
    	gtk_box_pack_start( GTK_BOX( hbox ), r_print_sel, FALSE, TRUE, 10 );
    	gtk_widget_show( r_print_sel );

    	hsep = gtk_hseparator_new();
    	gtk_box_pack_start( GTK_BOX( vbox ), hsep, FALSE, TRUE, 10 );
    	gtk_widget_show( hsep );

    	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
    	gtk_widget_show( hbox );

    	button = gnome_stock_button( GNOME_STOCK_BUTTON_OK );
    	gtk_box_pack_start( GTK_BOX( hbox ), button, TRUE, TRUE, 15 );
    	gtk_signal_connect( GTK_OBJECT( button ), "clicked",
                       	    GTK_SIGNAL_FUNC( _file_print ), NULL );
    	gtk_widget_show( button );

    	button = gnome_stock_button( GNOME_STOCK_BUTTON_CANCEL );
    	gtk_box_pack_start( GTK_BOX( hbox ), button, TRUE, TRUE, 15 );
    	gtk_signal_connect( GTK_OBJECT( button ), "clicked",
                       	    GTK_SIGNAL_FUNC( print_destroy ), NULL );
    	gtk_widget_show( button );

    	gtk_widget_show( print_window );
}


/*
 ---------------------------------------------------------------------
     Function: file_close()
     Desc: Callback-Function /File/Close
 ---------------------------------------------------------------------
*/
void
file_close( GtkWidget *widget, gpointer data)
{
	GideWindow *window;
	GideDocument *document;

	g_assert(IS_GIDE_WINDOW(data));
	window = GIDE_WINDOW(data);
	document = gI_window_get_current_doc( window );

	if( !document )
		return;

    	if( gI_document_is_changed( document ) ) { 
		if( file_close_dialog(window, document) == 2 )
			return; 
    	}

	gtk_object_destroy( GTK_OBJECT(document) );
}


/* -------------------------------------*/
/* file_close_dialog()                       */
/* return TRUE if user press NO and file is NOT saved      */
/* Return FALSE if file will save or user press cancel button*/
/* Use this instead of file_close_changed_dialog */
/* --------------------------------------*/
static gint
file_close_dialog( GideWindow *window, GideDocument *current )
{
	gchar *filename;
	gchar HelpString[STRLEN];
	gint btn;

	if( !current->filename ) {
		gchar *label;

		label = g_strdup( gI_window_get_doc_label(window, current) );
		
        	filename = g_strdup(file_strip_name( label ));
	} else {
		filename = g_strdup( current->filename ); 
	}

	g_snprintf( HelpString, STRLEN,
		    _("The file \"%s\" has been modified!\nSave it?\n"),
		    filename );
	g_free(filename);
	
	btn = gI_ask_dialog( HelpString );
	if ( btn == 0 ) {
		if( !current->filename )
			file_save_as( NULL, window );
		else
			file_save( NULL, window );
	}

	return btn;
}


static void write_history( GideDocument *document, FILE *history )
{
	if( !document )
        	return;

	if( !document->filename )
        	return;

	fprintf( history, "%s\n", document->filename );
}


/*
 ---------------------------------------------------------------------
     Function: file_exit()
     Desc: Callback-Function /File/Exit
 ---------------------------------------------------------------------
*/
void file_exit( GtkWidget *widget, gpointer data )
{
	gint nod;
	GideWindow *window;
	FILE *history;
    	gchar historywp[MAXLEN];
    	gI_project *project;

	g_assert(IS_GIDE_WINDOW(data));

	window = GIDE_WINDOW(data);
	
	/* close opened project */
	if( (project = gI_project_get_current()) ) {
    	gI_project_close( NULL, NULL );
	}

	/* save filenames */
	g_snprintf( historywp, MAXLEN, "%s/%s", gide_path, HISTORY );
	history = fopen( historywp, "w" );
	if( history ) {
    	g_list_foreach( window->documents,
			(GFunc) write_history,
			(gpointer) history );
    	fclose( history );
	}

	nod = gI_window_num_changed_docs( window );

	if( nod > 0 ) {
		if (!file_exit_is_changed( window ))
			gtk_main_quit();
	} else {
		gtk_main_quit();
	}
}

static gboolean
file_exit_is_changed( GideWindow *window )
{
    	GideDocument *document;
	gboolean canceled;
	gint nod,i;

    	nod = gI_window_num_docs( main_window );
	canceled = FALSE;
	
    	for( i=0; i<nod; i++ ) {
        	gI_window_goto_doc_by_index( window, i );
        	document = gI_window_get_current_doc( window );
        	if( gI_document_is_changed(document) ) {
            		if (file_close_dialog( window, document ) == 2) {
				canceled = TRUE;
			}
        	}
    	}

	return canceled;
}

static void
_file_autosave( GideDocument *document, gpointer data )
{
	FILE *docfile;
	gchar str[STRLEN];
	glong length,i;
	gchar *c;

    	/* don't autosave, if no document is given */
    	if( !document )
        	return;

    	/* don't autosave, if the document has no filename (i.e. "Untitled") */
    	if( !document->filename )
        	return;

    	/* don't autosave, if document is unchanged */
    	if( !document->changed )
        	return;

    	docfile = fopen( document->filename, "w" );
    	if( !docfile )
    	{
        	g_snprintf( str, STRLEN, _("Unable to Save '%s'!"), document->filename );
        	gI_error_dialog( str);
        	return;
    	}

    	length = gI_text_get_length( document );
    	for(i=1;i<=length;i++)
	{
        	c = gtk_editable_get_chars( GTK_EDITABLE( document ), i-1, i );
		fputc( c[0], docfile );
		g_free( c );
	}

    	fclose( docfile );

    	gI_document_set_changed_state( document, FALSE );

	document->last_mod = get_last_mod( document->filename );

	gI_window_set_statusbar( main_window );
}


void file_autosave( GtkWidget *widget, gpointer data )
{
	gchar sb_msg[STRLEN];

	/*g_print( "autosave!!\n" );*/
	/* remove timeout */
	gtk_timeout_remove( main_window->timeout_id );

    	/*push message to statusbar*/
    	g_snprintf( sb_msg, STRLEN, _("Autosaving...") );
    	gtk_statusbar_push( GTK_STATUSBAR( main_window->statusbar ), 2, sb_msg );

    	g_list_foreach( main_window->documents, (GFunc) _file_autosave, NULL );

	/* re-add timeout */
	main_window->timeout_id = gtk_timeout_add( cfg->autosave_freq * 1000, (GtkFunction) file_autosave, NULL ); 
}


gint file_check_if_exist( gchar *fname, gint askcreate )
{
	gchar txt[512];
	FILE *file;
	struct stat sta;
	gint e;
        
	e=stat(fname,&sta);
	if( !e && S_ISREG(sta.st_mode) )
	{   	/* file exist and is a regular file */
		return( 1 );	
	}

	if( askcreate && e==-1 )
	{
	    sprintf( txt, _("The file\n'%s'\ndoes not exist!\nDo you want to create the file?"), fname );
	    if( gI_ask_dialog( txt ) == 0 ) /*GNOME_YES )*/
	    {
	        file = fopen( fname, "w" );	/* this is not a good way!! */
		if( file ) 
		{
			fclose( file );
		}

		return( 1 );
	    }
	}
	else
	{
		if( !e )
		{ 
		    sprintf( txt, _("The file '%s' is not a regular file!"), fname );
                    gI_error_dialog( txt );
                }
	}

	return( 0 );
}


/*
 * this function checks if a document has been changed from
 * outside (i.e. the file mod time has changed)
 * return: 0 - not changed
 * 	   1 - changed
 *	   2 - does not (longer) exist
 */
glong check_doc_changed( GideDocument *doc )
{
	if( !doc )
		return( 0 );

	if( !doc->filename )
		return( 0 );

	if( !file_exist( doc->filename ) )
		return( 2 );

	if( get_last_mod( doc->filename ) != doc->last_mod )
		return( 1 );

	return( 0 );
}


/*
 * Close all open files
 */
void
gI_file_close_all(GtkWidget *widget, gpointer data)
{
	gint								i;

	for(i = 0; i < gI_window_num_docs(main_window); i++)
	{
		gtk_notebook_set_page(GTK_NOTEBOOK(main_window->notebook), i);
		file_close(NULL, main_window);
	}
}


/*
 * Save all files
 */
void
gI_file_save_all(GtkWidget *widget, gpointer data)
{
	gint i;
	GideWindow *window;
	gint nod;
	GideDocument *document;

	g_return_if_fail(IS_GIDE_WINDOW(data));

	window = GIDE_WINDOW(data);

	nod = gI_window_num_changed_docs(window);

	if(nod > 0) {
		for(i = 0; i < gI_window_num_docs(window); i++) {
			gtk_notebook_set_page(GTK_NOTEBOOK(window->notebook), i);
			document = gI_window_get_current_doc(window);
			if (gI_document_is_changed(document)) {
				file_save(NULL, window);
			}
		}
	}
}







