/* $Header: /cvs/gnome/gIDE/src/gI_project_window.c,v 1.4 2000/04/09 16:37:35 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gide.h"
#include "gI_project.h"
#include "gI_project_window.h"
#include "gI_todo.h"
#include "gI_common.h"
#include "gI_menus.h"

/* globals */
static GtkWidget*						project_window;
static GtkWidget*						e_target;
static GtkWidget*						target_list;
static GtkWidget*						e_prjname;
static GtkWidget*						e_prjver;
static GtkWidget*						lib_list;
static GtkWidget*						e_lib;
static GtkWidget*						e_cpar;
static GtkWidget*						e_cinc;
static GtkWidget*						e_lpar;
static GtkWidget*						e_llib;
static GtkWidget*						e_prjroot;
static GtkWidget*						e_prjfdir;
static GtkWidget*						e_changelog;
static GtkWidget*						e_mtarget;
static glong							lib_selected = -1;
static glong							target_selected = -1;
static GtkWidget*						files_list;

/* prototypes */
static void project_window_destroy(GtkWidget* widget, gpointer data);
static void project_window_target_list_select(GtkWidget* widget,
	gint row, gint column, GdkEventButton* bevent);
static void project_window_target_add(GtkWidget* widget, gpointer data);
static void project_window_target_remove(GtkWidget* widget, gpointer data);
static void project_window_sources_list_select(GtkWidget* widget,
	gint row, gint column, GdkEventButton* bevent);
static void project_window_sources_add_ok(GtkWidget* widget,
	GtkFileSelection* file_sel);
static void project_window_sources_add_cancel(GtkWidget* widget,
	GtkFileSelection* file_sel);
static void project_window_sources_add(GtkWidget* widget, gpointer data);
static void project_window_sources_remove(GtkWidget* widget, gpointer data);
static gboolean project_window_target_exists(gI_project* project,
	gchar* target);
static void project_window_sources_add_target_dialog(GtkWidget* widget,
	GtkWidget* entry);
static void project_window_sources_add_target(GtkWidget* widget, gpointer data);
static void project_window_lib_list_select(GtkWidget* widget, gint row,
	gint column, GdkEventButton* bevent);
static void project_window_lib_add(GtkWidget* widget, gpointer data);
static void project_window_lib_remove(GtkWidget* widget, gpointer data);
static void project_window_ok(void);
static GtkWidget* project_window_create_project_page(void);
static GtkWidget* project_window_create_targets_page(void);
static GtkWidget* project_window_create_sources_page(void);
static GtkWidget* project_window_create_libraries_page(void);
static GtkWidget* project_window_create_parameters_page(void);
static void project_window_clicked(GnomeDialog* dlg, int button,
	gpointer data);
static void move_up_entries(gI_project* project, glong target, glong source);

/*
 * Destroy the project New/Edit/etc. window
 */
static void
project_window_destroy(
	GtkWidget*							widget,
	gpointer							data
)
{
	gI_project* project = gI_project_get_current();

	if(project && project->todoPane)
	{
		gI_todo_pane_destroy(project->todoPane);
		project->todoPane = NULL;
	}

	if(project_window)
	{
		gnome_dialog_close(GNOME_DIALOG(project_window));
		project_window = NULL;
	}

	gI_project_update_menu(main_window);
}


/*
 * Callback function for (un)selecting target
 */
static void
project_window_target_list_select(
	GtkWidget*							widget,
	gint								row,
	gint								column,
	GdkEventButton*						bevent
)
{
	GList*								selection;

	selection = GTK_CLIST(widget)->selection;
	if(!selection)
	{
		target_selected = -1;
	}
	else
	{
		target_selected = row;
	}
}


/*
 * Callback function for adding new target
 */
static void
project_window_target_add(
	GtkWidget*							widget,
	gpointer							data
)
{
	gchar*								ptr;
	gchar*								insrow[2];

	ptr = gtk_entry_get_text(GTK_ENTRY(e_target));
	if(!ptr || isempty(ptr))
	{
		return;
	}

	insrow[0] = ptr;
	insrow[1] = NOT_YET_SUPPORTED;
	gtk_clist_append(GTK_CLIST(target_list), insrow);
	gtk_entry_set_text(GTK_ENTRY(e_target), "");
}


/*
 * Callback function for removing target
 */
static void
project_window_target_remove(
	GtkWidget*							widget,
	gpointer							data
)
{
	gchar*								target_to_remove;
	glong								i;

	if(target_selected == -1)
	{
		gI_error_dialog(_("No target selected!"));
		return;
	}

	gtk_clist_get_text(GTK_CLIST(target_list), target_selected, 0,
		&target_to_remove);
	for(i = 0; i < (GTK_CLIST(files_list)->rows); i++)
	{
		if(!strcmp(target_to_remove, (gchar*)gtk_clist_get_row_data(
			GTK_CLIST(files_list), i)))
		{
			gtk_clist_remove(GTK_CLIST(files_list), i);
			break;
		}
	}

	gtk_clist_remove(GTK_CLIST(target_list), target_selected);
	target_selected = -1;
}


/*
 * Callback function for clicking source file
 */
static void
project_window_sources_list_select(
	GtkWidget*							widget,
	gint								row,
	gint								column,
	GdkEventButton*						bevent
)
{
	gchar*								build;

	if(!bevent)
	{
		return;
	}

	if(bevent->type == GDK_2BUTTON_PRESS || bevent->type == GDK_3BUTTON_PRESS)
	{
		if(column == 3)
		{
			gtk_clist_get_text(GTK_CLIST(files_list), row, column, &build);
			if(!strncmp(build, "Yes", 3))
			{
				gtk_clist_set_text(GTK_CLIST(files_list), row, column, "No");
			}
			else
			{
				gtk_clist_set_text(GTK_CLIST(files_list), row, column, "Yes");
			}
		}
	}
}


/*
 * Callback function for add source ok button
 */
static void
project_window_sources_add_ok(
	GtkWidget*							widget,
	GtkFileSelection*					file_sel
)
{
	gint								newrow;
	gint								rownum;
	gchar*								insrow[4];
	gchar*								target;
	gchar*								sel_filename;
	gchar*								filename;
	GList*								rowp;

	gtk_clist_get_text(GTK_CLIST(target_list), target_selected, 0, &target);

	sel_filename = gtk_file_selection_get_filename(
		GTK_FILE_SELECTION(file_sel));

	rowp = GTK_CLIST(file_sel->file_list)->row_list;
	rownum = 0;
	newrow = 0;

	while(rowp)
	{
		if(GTK_CLIST_ROW(rowp) -> state == GTK_STATE_SELECTED)
		{
			if(gtk_clist_get_cell_type(GTK_CLIST(file_sel->file_list), rownum,
				0) == GTK_CELL_TEXT)
			{
				gtk_clist_get_text(GTK_CLIST(file_sel->file_list), rownum, 0,
					&filename);
				if(filename != NULL)
				{
					if(GTK_WIDGET_VISIBLE(file_sel))
					{
						gtk_widget_hide(GTK_WIDGET(file_sel));
					}
					{
						gchar*			fwrp;
						gchar*			p;
						gchar*			q;
						gI_project* project = gI_project_get_current();

						project->prjroot = g_strdup(
							gtk_entry_get_text(GTK_ENTRY(e_prjroot)));

						q = p = get_path(sel_filename);
						if(project->prjroot && !isempty(project->prjroot) &&
							!strncmp(p, project->prjroot,
							strlen(project->prjroot)))
						{
							p += strlen(project->prjroot);
						}

						if(!isempty(p))
						{
							if(p[strlen(p) - 1] == '/' || filename[0] == '/')
							{
								fwrp = g_strdup_printf("%s%s", p, filename);
							}
							else
							{
								fwrp = g_strdup_printf("%s/%s", p, filename);
							}
						}
						else
						{
							fwrp = g_strdup_printf("%s", filename);
						}

						g_free(q);

						insrow[0] = fwrp;
						insrow[1] = SOURCE;
						insrow[2] = target;
						insrow[3] = YES;

						newrow = gtk_clist_append(GTK_CLIST(files_list),
							insrow);
						gtk_clist_set_row_data(GTK_CLIST(files_list), newrow,
							(gpointer)target);
						g_free(fwrp);
					}
				}
			}
		}
		rownum++;
		rowp = g_list_next(rowp);
	}

	if(GTK_WIDGET_VISIBLE(file_sel))
	{
		gtk_widget_hide(GTK_WIDGET(file_sel));
	}
}


/*
 * Callback function for adding sources cancel button
 */
static void
project_window_sources_add_cancel(
	GtkWidget*							widget,
	GtkFileSelection*					file_sel
)
{
	if(GTK_WIDGET_VISIBLE(file_sel))
	{
		gtk_widget_hide(GTK_WIDGET(file_sel));
	}
}


/*
 * Callback function for adding sources
 */
static void
project_window_sources_add(
	GtkWidget*							widget,
	gpointer							data
)
{
	static GtkWidget*					file_sel = NULL;

	if(target_selected == -1)
	{
		gI_error_dialog(_("No target selected!"));
		return;
	}

	if(!file_sel)
	{
		file_sel = gtk_file_selection_new(_("Add file to project..."));
		gtk_clist_set_selection_mode(
			GTK_CLIST(GTK_FILE_SELECTION(file_sel)->file_list),
			GTK_SELECTION_EXTENDED);

		gtk_signal_connect(GTK_OBJECT(file_sel), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroy), file_sel);
		gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(file_sel)->ok_button),
			"clicked", GTK_SIGNAL_FUNC(project_window_sources_add_ok),
			file_sel);
		gtk_signal_connect(
			GTK_OBJECT(GTK_FILE_SELECTION(file_sel)->cancel_button), "clicked",
			GTK_SIGNAL_FUNC(project_window_sources_add_cancel), file_sel);
	}

	if(GTK_WIDGET_VISIBLE(file_sel))
	{
		return;
	}
	else
	{
		gtk_widget_show(file_sel);
	}
}


/*
 * Callback function for removing sources
 */
static void
project_window_sources_remove(
	GtkWidget*							widget,
	gpointer							data
)
{
	gint								rownum;
	GList*								rowp;

	rowp = GTK_CLIST(files_list)->row_list;
	rownum = 0;

	/* this was a bit tricky but it works fine now :-) */
	while(rowp)
	{
		if(GTK_CLIST_ROW(rowp)->state == GTK_STATE_SELECTED)
		{
			rowp = g_list_next(rowp);
			gtk_clist_remove(GTK_CLIST(files_list), rownum);
		}
		else
		{
			rowp = g_list_next(rowp);
			rownum++;
		}
	}
}


/*
 * Returns TRUE if the specified target exists within the project
 */
static gboolean
project_window_target_exists(
	gI_project*							project,
	gchar*								target
)
{
	glong								i;

	for(i = 0; i < project->targets_no; i++)
	{
		if(!strcmp(project->targets[i]->name, target))
		{
			return TRUE;
		}
	}

	return FALSE;
}

/*
 * Callback function for add target dialog
 */
static void
project_window_sources_add_target_dialog(
	GtkWidget*							widget,
	GtkWidget*							entry
)
{
	gchar*								target;
	gchar*								insrow[3];
	glong								row;
	gchar*								selected_target;

	gtk_clist_get_text(GTK_CLIST(target_list), target_selected, 0,
		&selected_target);
	target = gtk_entry_get_text(GTK_ENTRY(entry));

	if(!target || isempty(target))
	{
		return;
	}

	if(!project_window_target_exists(gI_project_get_current(), target))
	{
		gI_error_dialog(_("Specified target does not exist!"));
		return;
	}

	insrow[0] = target;
	insrow[1] = TARGET;
	insrow[2] = selected_target;
	row = gtk_clist_append(GTK_CLIST(files_list), insrow);
	gtk_clist_set_row_data(GTK_CLIST(files_list), row,
		(gpointer)selected_target);
}


/*
 * Callback function for add target within sources window
 */
static void
project_window_sources_add_target(
	GtkWidget*							widget,
	gpointer							data
)
{
	if(target_selected == -1)
	{
		gI_error_dialog(_("No target selected!"));
		return;
	}

	entry_dialog("Target", "Add target",
		project_window_sources_add_target_dialog);
}


/*
 * Callback function for library list
 */
static void
project_window_lib_list_select(
	GtkWidget*							widget,
	gint								row,
	gint								column,
	GdkEventButton*						bevent
)
{
	GList*								selection;

	selection = GTK_CLIST(widget)->selection;
	if(!selection)
	{
		lib_selected = -1;
	}
	else
	{
		lib_selected = row;
	}
}


/*
 * Callback function for adding library
 */
static void
project_window_lib_add(
	GtkWidget*							widget,
	gpointer							data
)
{
	gchar*								insrow[1];
	gchar*								ptr;

	ptr = gtk_entry_get_text(GTK_ENTRY(e_lib));
	if(!ptr || isempty(ptr))
	{
		return;
	}

	insrow[0] = ptr;
	gtk_clist_append(GTK_CLIST(lib_list), insrow);
	gtk_entry_set_text(GTK_ENTRY(e_lib), "");
}


/*
 * Callback function for removing library
 */
static void
project_window_lib_remove(
	GtkWidget*							widget,
	gpointer							data
)
{
	if(lib_selected == -1)
	{
		gI_error_dialog(_("No Library selected!"));
		return;
	}

	gtk_clist_remove(GTK_CLIST(lib_list), lib_selected);
	lib_selected = -1;
}


/*
 * Callback function for ok button
 */
static void
project_window_ok(
	void
)
{
	gchar*								ptr;
	gchar*								selected_target;
	gchar*								selected_lib;
	gchar*								target_target;
	gchar*								target_file;
	gchar*								file_comment;
	gchar								msg[STRLEN];
	glong								i;
	glong								j;
	glong								sources_no;
	FILE*								prjfile;
	gchar*								build;
	gI_project*							project;

	project = gI_project_get_current();

	/* save project */
	ptr = gtk_entry_get_text(GTK_ENTRY(e_prjname));
	if(ptr)
	{
		if(isempty(ptr))
		{
			gI_error_dialog(_("Project Name is missing!"));
			return;
		}
		if(project->name)
		{
			g_free(project->name);
		}
		project->name = g_strdup(ptr);
	}

	ptr = gtk_entry_get_text(GTK_ENTRY(e_prjver));
	if(ptr)
	{
		if(project->version)
		{
			g_free(project->version);
		}
		project->version = g_strdup(ptr);
	}

	ptr = gtk_entry_get_text(GTK_ENTRY(e_changelog));
	if(ptr)
	{
		if(project->changelog)
		{
			g_free(project->changelog);
		}
		project->changelog = g_strdup(ptr);
	}

	ptr = gtk_entry_get_text(GTK_ENTRY(e_prjroot));
	if(ptr)
	{
		if(project->prjroot)
		{
			g_free(project->prjroot);
		}
		project->prjroot = g_strdup(ptr);
	}

	ptr = gtk_entry_get_text(GTK_ENTRY(e_prjfdir));
	if(ptr)
	{
		if(project->prjfdir)
		{
			g_free(project->prjfdir);
		}
		project->prjfdir = g_strdup(ptr);
	}

	for(i = 0; i < GTK_CLIST(target_list)->rows; i++)
	{
		if(i >= MAX_PROJECT_TARGETS)
		{
			g_snprintf(msg, sizeof(msg), _("\nUnable to save more than %d"
				" targets per project! If you need\n more, then change the"
				" \"#define MAX_PROJECT_TARGETS\"\nin structs.h and"
				" recompile.\n"), MAX_PROJECT_TARGETS);
			gI_error_dialog(msg);
			break;
		}
		gtk_clist_get_text(GTK_CLIST(target_list), i, 0, &selected_target);
		project->targets[i]->name = (gchar*) realloc(project->targets[i]->name,
			(strlen(selected_target) + 1) * sizeof(gchar));
		strcpy(project->targets[i]->name, selected_target);
		sources_no = -1;
		for(j = 0; j < GTK_CLIST(files_list)->rows; j++)
		{
			if(j >= MAX_PROJECT_FILES)
			{
				g_snprintf(msg, sizeof(msg), _("\nUnable to save more than %d"
					" files per project! If you need\n more, then change the"
					" \"#define MAX_PROJECT_FILES\"\nin structs.h and"
					" recompile.\n"), MAX_PROJECT_FILES);
				gI_error_dialog(msg);
				break;
			}
			gtk_clist_get_text(GTK_CLIST(files_list), j, 0, &target_file);
			target_target = (gchar*)gtk_clist_get_row_data(
				GTK_CLIST(files_list), j);
			if(!strcmp(selected_target, target_target))
			{
				sources_no++;
				gtk_clist_get_text(GTK_CLIST(files_list), j, 1, &file_comment);
				if(!strcmp(file_comment, SOURCE))
				{
					gchar*				p;

					if(project->targets[i]->sources[sources_no])
					{
						g_free(project->targets[i]->sources[sources_no]);
					}

					if(!strncmp(target_file, project->prjroot,
						strlen(project->prjroot)))
					{
						p = target_file + strlen(project->prjroot);
						project->targets[i]->sources[sources_no] = g_strdup(p);
					}
					else
					{
						project->targets[i]->sources[sources_no] =
							g_strdup(target_file);
					}

					gtk_clist_get_text(GTK_CLIST(files_list), j, 3, &build);
					if(!strncmp(build, "Yes", 3))
					{
						project->targets[i]->build[sources_no] = 1;
					}
					else
					{
						project->targets[i]->build[sources_no] = 0;
					}
				}
				else
				{
					project->targets[i]->sources[sources_no] = (gchar*)realloc(
						project->targets[i]->sources[sources_no],
						(strlen(target_file) + 2) * sizeof(gchar));
					sprintf(project->targets[i]->sources[sources_no], "%%%s",
						target_file);
				}
			}
		}
		if(sources_no == -1)
		{
			sources_no = 0;
		}
		project->targets[i]->sources_no = sources_no + 1;
	}
	project->targets_no = GTK_CLIST(target_list)->rows;

	for(i = 0; i < GTK_CLIST(lib_list)->rows; i++)
	{
		if(i >= MAX_PROJECT_FILES)
		{
			g_snprintf(msg, sizeof(msg), _("\nUnable to save more than %d"
				" libraries per\nproject! If you need more, change the"
				" \"#define MAX_PROJECT_FILES\"\nin structs.h and"
				" recompile.\n"), MAX_PROJECT_FILES);
			gI_error_dialog(msg);
			break;
		}
		gtk_clist_get_text(GTK_CLIST(lib_list), i, 0, &selected_lib);
		project->libs[i] = (gchar*)realloc(project->libs[i],
			(strlen(selected_lib) + 1) * sizeof(gchar));
		strcpy(project->libs[i], selected_lib);
	}
	project->libs_no = i;

	ptr = gtk_entry_get_text(GTK_ENTRY(e_cpar));
	if(ptr)
	{
		project->cpar = (gchar*)realloc(project->cpar,
			(strlen(ptr) + 1) * sizeof(gchar));
		strcpy(project->cpar, ptr);
	}

	ptr = gtk_entry_get_text(GTK_ENTRY(e_cinc));
	if(ptr)
	{
		project->cinc = (gchar*)realloc(project->cinc,
			(strlen(ptr) + 1) * sizeof(gchar));
		strcpy(project->cinc, ptr);
	}

	ptr = gtk_entry_get_text(GTK_ENTRY(e_lpar));
	if(ptr)
	{
		project->lpar = (gchar*)realloc(project->lpar,
			(strlen(ptr) + 1) * sizeof(gchar));
		strcpy(project->lpar, ptr);
	}

	ptr = gtk_entry_get_text(GTK_ENTRY(e_llib));
	if(ptr)
	{
		project->llib = (gchar*)realloc(project->llib,
			(strlen(ptr) + 1) * sizeof(gchar));
		strcpy(project->llib, ptr);
	}

	ptr = gtk_entry_get_text(GTK_ENTRY(e_mtarget));
	if(ptr)
	{
		project->mtarget = (gchar*)realloc(project->mtarget,
			(strlen(ptr) + 1) * sizeof(gchar));
		strcpy(project->mtarget, ptr);
	}

	if(!isempty(project->prjfdir) && project->filename &&
		!strstr(project->filename, project->prjfdir))
	{
		g_free(project->filename);
		project->filename = NULL;
	}

	if(!isempty(project->prjfdir))
	{
		/* good way? */
		strcpy(prj_path, project->prjfdir);
	}

	if(!project->filename)
	{
		if(!isempty(project->version))
		{
			project->filename = g_strconcat(prj_path, "/", project->name,
				"-", project->version, ".prj", NULL);
		}
		else
		{
			project->filename = g_strconcat(prj_path, "/", project->name,
				".prj", NULL);
		}
	}

	prjfile = fopen(project->filename, "w");
	if(!prjfile)
	{
		/* error handling */
		gI_error_dialog(_("Unable to Create Project-File"));
		return;
	}

	fprintf(prjfile, "; gIDE %s - Project File\n\n", VERSION);

	fprintf(prjfile, "PROJECT: %s\n", project->name);
	fprintf(prjfile, "VERSION: %s\n", project->version);
	fprintf(prjfile, "CHANGELOG: !%s!\n", project->changelog);
	fprintf(prjfile, "MTARGET: !%s!\n", project->mtarget);
	fprintf(prjfile, "PRJROOT: !%s!\n", project->prjroot);
	fprintf(prjfile, "PRJFDIR: !%s!\n", project->prjfdir);

	fprintf(prjfile, "$TARGETS: %d\n", project->targets_no);
	for(i = 0; i < project->targets_no; i++)
	{
		fprintf(prjfile, "%s\n", project->targets[i]->name);
	}
	fprintf(prjfile, "$END TARGETS\n");

	for(j = 0; j < project->targets_no; j++)
	{
		fprintf(prjfile, "$SOURCES: %d\n", project->targets[j]->sources_no);
		fprintf(prjfile, "%ld\n", j);
		for(i = 0; i < project->targets[j]->sources_no; i++)
		{
			fprintf(prjfile, "%s\n", project->targets[j]->sources[i]);
			fprintf(prjfile, "%d\n", project->targets[j]->build[i]);
		}
		fprintf(prjfile, "$END SOURCES\n");
	}

	fprintf(prjfile, "$LIBS: %d\n", project->libs_no);
	for(i = 0; i < project->libs_no; i++)
	{
		fprintf(prjfile, "%s\n", project->libs[i]);
	}
	fprintf(prjfile, "$END LIBS\n");

	fprintf(prjfile, "CPAR: !%s!\n", project->cpar);
	fprintf(prjfile, "CINC: !%s!\n", project->cinc);

	fprintf(prjfile, "LPAR: !%s!\n", project->lpar);
	fprintf(prjfile, "LLIB: !%s!\n", project->llib);

	gI_todo_list_write_to_file(project->todoList, prjfile,
		"$Begin_Todo List", "$End_Todo List");

	fclose(prjfile);

	project_window_destroy(NULL, NULL);

	if(cfg->prjftree)
	{
		gtk_widget_show(main_window->tree_box);
		gI_project_files_tree(main_window->tree_viewport, project);
	}
}


/*
 * Create project page for project notebook
 */
static GtkWidget*
project_window_create_project_page(
	void
)
{
	GtkWidget*							book_vbox;
	GtkWidget*							vbox;
	GtkWidget*							frame;
	GtkWidget*							hbox;

	book_vbox = gtk_vbox_new(FALSE, 0);

	frame = gtk_frame_new(_("Project"));
	gtk_box_pack_start(GTK_BOX(book_vbox), frame,
		FALSE, TRUE, 10);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 10);

	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Project Name       ")),
		FALSE, TRUE, 5);

	e_prjname = gtk_entry_new_with_max_length(128);
	gtk_box_pack_start(GTK_BOX(hbox), e_prjname, TRUE, TRUE, 5);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 10);

	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Project Version    ")),
		FALSE, TRUE, 5);

	e_prjver = gtk_entry_new_with_max_length(16);
	gtk_box_pack_start(GTK_BOX(hbox), e_prjver, TRUE, TRUE, 5);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 10);

	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Main Target Name")),
		FALSE, TRUE, 5);

	e_mtarget = gtk_entry_new_with_max_length(255);
	gtk_box_pack_start(GTK_BOX(hbox), e_mtarget, TRUE, TRUE, 5);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 10);

	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("ChangeLog-File  ")),
		FALSE, TRUE, 5);

	e_changelog = gtk_entry_new_with_max_length(255);
	gtk_box_pack_start(GTK_BOX(hbox), e_changelog, TRUE, TRUE, 5);

	frame = gtk_frame_new(_("Parameters"));
	gtk_box_pack_start(GTK_BOX(book_vbox), frame, FALSE, TRUE, 10);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 10);

	gtk_box_pack_start(GTK_BOX(hbox),
		gtk_label_new(_("Project Files Root Dir:")), FALSE, TRUE, 5);

	e_prjroot = gtk_entry_new_with_max_length(255);
	gtk_box_pack_start(GTK_BOX(hbox), e_prjroot, TRUE, TRUE, 5);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 10);

	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Project-File Dir:")),
		FALSE, TRUE, 5);

	e_prjfdir = gtk_entry_new_with_max_length(255);
	gtk_box_pack_start(GTK_BOX(hbox), e_prjfdir, TRUE, TRUE, 5);

	return book_vbox;
}


/*
 * Create targets page for project notebook
 */
static GtkWidget*
project_window_create_targets_page(
	void
)
{
	GtkWidget*							book_vbox;
	GtkWidget*							frame;
	GtkWidget*							vbox;
	GtkWidget*							hbox;
	GtkWidget*							scrolled_window;
	gchar* target_list_titles[] = { "Target", "Comment" };
	GtkWidget*							button;

	book_vbox = gtk_vbox_new(FALSE, 0);

	frame = gtk_frame_new(_("Targets"));
	gtk_box_pack_start(GTK_BOX(book_vbox), frame, TRUE, TRUE, 10);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 10);

	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

	gtk_box_pack_start(GTK_BOX(hbox), scrolled_window, TRUE, TRUE, 5);

	target_list = gtk_clist_new_with_titles(2, target_list_titles);
	gtk_clist_column_titles_passive(GTK_CLIST(target_list));
	gtk_clist_set_column_width(GTK_CLIST(target_list), 0, 250);
	gtk_container_add(GTK_CONTAINER(scrolled_window), target_list);
	gtk_signal_connect(GTK_OBJECT(target_list), "select_row",
		GTK_SIGNAL_FUNC(project_window_target_list_select), NULL);
	gtk_signal_connect(GTK_OBJECT(target_list), "unselect_row",
		GTK_SIGNAL_FUNC(project_window_target_list_select), NULL);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 5);

	button = gtk_button_new_with_label(_("   New Target   "));
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, TRUE, 5);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(project_window_target_add), NULL);

	button = gtk_button_new_with_label(_("  Remove Target  "));
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, TRUE, 5);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(project_window_target_remove), NULL);

	e_target = gtk_entry_new_with_max_length(255);
	gtk_box_pack_start(GTK_BOX(hbox), e_target, TRUE, TRUE, 5);

	return book_vbox;
}


/*
 * Create sources page for project notebook
 */
static GtkWidget*
project_window_create_sources_page(
	void
)
{
	GtkWidget*							book_vbox;
	GtkWidget*							frame;
	GtkWidget*							vbox;
	GtkWidget*							scrwin;
	GtkWidget*							hbox;
	GtkWidget*							button;
	gchar* files_list_titles[] = { "File", "Comment", "Target", "Build" };

	book_vbox = gtk_vbox_new(FALSE, 0);

	frame = gtk_frame_new(_("Sources"));
	gtk_box_pack_start(GTK_BOX(book_vbox), frame, TRUE, TRUE, 10);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	scrwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_box_pack_start(GTK_BOX(vbox), scrwin, TRUE, TRUE, 5);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

	files_list = gtk_clist_new_with_titles(4, files_list_titles);
	gtk_clist_column_titles_passive(GTK_CLIST(files_list));
	gtk_clist_set_selection_mode(GTK_CLIST(files_list),
		GTK_SELECTION_EXTENDED);

	gtk_clist_set_column_width(GTK_CLIST(files_list), 0, 300);
	gtk_clist_set_column_width(GTK_CLIST(files_list), 1, 60);
	gtk_clist_set_column_width(GTK_CLIST(files_list), 2, 100);
	gtk_container_add(GTK_CONTAINER(scrwin), files_list);
/*	gtk_box_pack_start(GTK_BOX(vbox), files_list, TRUE, TRUE, 5);*/
	gtk_signal_connect(GTK_OBJECT(files_list), "select_row",
		GTK_SIGNAL_FUNC(project_window_sources_list_select), NULL);
	gtk_widget_show(files_list);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 10);

	button = gtk_button_new_with_label(_("   Add Files   "));
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, TRUE, 5);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(project_window_sources_add), NULL);

	button = gtk_button_new_with_label(_("  Remove Files  "));
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, TRUE, 5);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(project_window_sources_remove), NULL);

	button = gtk_button_new_with_label(_("   Add Target   "));
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, TRUE, 20);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(project_window_sources_add_target), NULL);

	return book_vbox;
}


/*
 * Create libraries page for project notebook
 */
static GtkWidget*
project_window_create_libraries_page(
	void
)
{
	GtkWidget*							book_vbox;
	GtkWidget*							frame;
	GtkWidget*							vbox;
	GtkWidget*							scrwin;
	GtkWidget*							hbox;
	GtkWidget*							button;

	book_vbox = gtk_vbox_new(FALSE, 0);

	frame = gtk_frame_new(_("Libraries"));
	gtk_box_pack_start(GTK_BOX(book_vbox), frame, TRUE, TRUE, 10);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	scrwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_box_pack_start(GTK_BOX(vbox), scrwin, TRUE, TRUE, 5);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

	lib_list = gtk_clist_new(1);
	gtk_clist_column_titles_hide(GTK_CLIST(lib_list));
	gtk_clist_column_titles_passive(GTK_CLIST(lib_list));
	gtk_container_add(GTK_CONTAINER(scrwin), lib_list);
	gtk_signal_connect(GTK_OBJECT(lib_list), "select_row",
		GTK_SIGNAL_FUNC(project_window_lib_list_select), NULL);
	gtk_signal_connect(GTK_OBJECT(lib_list), "unselect_row",
		GTK_SIGNAL_FUNC(project_window_lib_list_select), NULL);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 10);

	button = gtk_button_new_with_label(_("   Add   "));
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, TRUE, 5);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(project_window_lib_add), NULL);

	button = gtk_button_new_with_label(_("  Remove  "));
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, TRUE, 5);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(project_window_lib_remove), NULL);

	e_lib = gtk_entry_new_with_max_length(255);
	gtk_box_pack_start(GTK_BOX(hbox), e_lib, TRUE, TRUE, 15);

	return book_vbox;
}


/*
 * Create parameters page for project notebook
 */
static GtkWidget*
project_window_create_parameters_page(
	void
)
{
	GtkWidget*							book_vbox;
	GtkWidget*							frame;
	GtkWidget*							vbox;
	GtkWidget*							hbox;

	book_vbox = gtk_vbox_new(FALSE, 0);

	frame = gtk_frame_new(_("Compile"));
	gtk_box_pack_start(GTK_BOX(book_vbox), frame, FALSE, TRUE, 10);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 10);

	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Options        ")),
		FALSE, TRUE, 5);

	e_cpar = gtk_entry_new_with_max_length(255);
	gtk_box_pack_start(GTK_BOX(hbox), e_cpar, TRUE, TRUE, 5);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 10);

	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Include path")),
		FALSE, TRUE, 5);

	e_cinc = gtk_entry_new_with_max_length(255);
	gtk_box_pack_start(GTK_BOX(hbox), e_cinc, TRUE, TRUE, 5);

	frame = gtk_frame_new(_("Link"));
	gtk_box_pack_start(GTK_BOX(book_vbox), frame, FALSE, TRUE, 10);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 10);

	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Options  ")),
		FALSE, TRUE, 5);

	e_lpar = gtk_entry_new_with_max_length(255);
	gtk_box_pack_start(GTK_BOX(hbox), e_lpar, TRUE, TRUE, 5);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 10);

	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Lib path")),
		FALSE, TRUE, 5);

	e_llib = gtk_entry_new_with_max_length(255);
	gtk_box_pack_start(GTK_BOX(hbox), e_llib, TRUE, TRUE, 5);

	return book_vbox;
}


/*
 * Button callback for prototype generator
 */
static void
project_window_clicked(
	GnomeDialog*						dlg,
	int									button,
	gpointer							data
)
{
	switch(button)
	{
	case 0:
		project_window_ok();
		break;
	case 1:
		if(!gI_project_get_current()->name)
		{
			gI_project_close(NULL, NULL);
		}
		project_window_destroy(NULL, NULL);
		break;
	}
}


/*
 * Create project definition window.
 * Has different modes for creating a new project or editing an existing one.
 */
void
gI_project_window_create(
	gI_project*							project,
	GideProjectWindowMode				mode
)
{
	gchar								mode_str[32];
	GtkWidget*							notebook;
	GtkBox*								vbox;
	glong								i;
	glong								j;
	glong								row;
	gchar*								insrow[4];

	if(project_window)
	{
		return;
	}

	switch(mode)
	{
	case GI_PW_NEW:
		strcpy(mode_str, _("New Project"));
		break;

	case GI_PW_EDIT:
		strcpy(mode_str, _("Edit Project"));
		break;

	default:
		strcpy(mode_str, _("Project"));
	}

	project_window = gnome_dialog_new(mode_str, GNOME_STOCK_BUTTON_OK,
		GNOME_STOCK_BUTTON_CANCEL, NULL);
	gnome_dialog_set_parent(GNOME_DIALOG(project_window),
		GTK_WINDOW(main_window));

	vbox = GTK_BOX(GNOME_DIALOG(project_window)->vbox);

	notebook = gtk_notebook_new();
	gtk_box_pack_start(vbox, notebook, TRUE, TRUE, 5);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(notebook), TRUE);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook), TRUE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), TRUE);

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
		project_window_create_project_page(), gtk_label_new(_("Project")));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
		project_window_create_targets_page(), gtk_label_new(_("Targets")));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
		project_window_create_sources_page(), gtk_label_new(_("Sources")));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
		project_window_create_libraries_page(), gtk_label_new(_("Libraries")));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
		project_window_create_parameters_page(),
		gtk_label_new(_("Parameters")));

	if(!project->todoList)
	{
		g_print("Creating todo list in project window creation code\n");
		project->todoList = gI_todo_list_create();
	}

	project->todoPane = gI_todo_pane_create_from_todo(project->todoList);

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
		project->todoPane->vbox, gtk_label_new(_("Todo")));

	gtk_widget_show_all(GTK_WIDGET(vbox));

	gnome_dialog_set_default(GNOME_DIALOG(project_window), 0);

	gtk_signal_connect(GTK_OBJECT(project_window), "clicked",
		GTK_SIGNAL_FUNC(project_window_clicked), NULL);

	/* if edit mode, insert data */
	if(mode == GI_PW_EDIT)
	{
		/* We're apparently not guaranteed to have strings for these.
		   gtk_entry doesn't like it if we pass a null string. */

		if(project->name)
		{
			gtk_entry_set_text(GTK_ENTRY(e_prjname), project->name);
		}
		if(project->version)
		{
			gtk_entry_set_text(GTK_ENTRY(e_prjver), project->version);
		}
		if(project->changelog)
		{
			gtk_entry_set_text(GTK_ENTRY(e_changelog), project->changelog);
		}
		if(project->mtarget)
		{
			gtk_entry_set_text(GTK_ENTRY(e_mtarget), project->mtarget);
		}

		for(i = 0; i < project->targets_no; i++)
		{
			insrow[0] = project->targets[i]->name;
			insrow[1] = NOT_YET_SUPPORTED;
			gtk_clist_append(GTK_CLIST(target_list), insrow);

			for(j = 0; j < project->targets[i]->sources_no; j++)
			{
				if(project->targets[i]->sources[j][0] == '%')
				{
					insrow[0] = project->targets[i]->sources[j] + 1;
					insrow[1] = TARGET;
					insrow[2] = project->targets[i]->name;
					insrow[3] = NO;
					row = gtk_clist_append(GTK_CLIST(files_list), insrow);
					gtk_clist_set_row_data(GTK_CLIST(files_list), row,
						(gpointer)project->targets[i]->name);
				}
				else
				{
					insrow[0] = project->targets[i]->sources[j];
					insrow[1] = SOURCE;
					insrow[2] = project->targets[i]->name;
					if(project->targets[i]->build[j])
					{
						insrow[3] = YES;
					}
					else
					{
						insrow[3] = NO;
					}
					row = gtk_clist_append(GTK_CLIST(files_list), insrow);
					gtk_clist_set_row_data(GTK_CLIST(files_list), row,
						(gpointer)project->targets[i]->name);
				}
			}
		}

		for(i = 0; i < project->libs_no; i++)
		{
			insrow[0] = project->libs[i];
			gtk_clist_append(GTK_CLIST(lib_list), insrow);
		}

		/* Don't pass null pointers to gtk_entry... */
		if(project->cpar)
		{
			gtk_entry_set_text(GTK_ENTRY(e_cpar), project->cpar);
		}
		if(project->cinc)
		{
			gtk_entry_set_text(GTK_ENTRY(e_cinc), project->cinc);
		}
		if(project->lpar)
		{
			gtk_entry_set_text(GTK_ENTRY(e_lpar), project->lpar);
		}
		if(project->llib)
		{
			gtk_entry_set_text(GTK_ENTRY(e_llib), project->llib);
		}
	}

	if(project->prjroot)
	{
		gtk_entry_set_text(GTK_ENTRY(e_prjroot), project->prjroot);
	}
	if(project->prjfdir)
	{
		gtk_entry_set_text(GTK_ENTRY(e_prjfdir), project->prjfdir);
	}
	else
	{
		gtk_entry_set_text(GTK_ENTRY(e_prjfdir), prj_path);
	}

	/* Go to first page... */
	gtk_notebook_set_page(GTK_NOTEBOOK(notebook), 0);

	gtk_widget_show(project_window);

	gI_project_update_menu(main_window);
}


static void
move_up_entries(
	gI_project*							project,
	glong								target,
	glong								source
)
{
	glong								i;

	if(!project)
	{
		return;
	}

	for(i = source; i < project->targets[target]->sources_no - 1; i++)
	{
		project->targets[target]->sources[i] = realloc(
			project->targets[target]->sources[i],
			strlen(project->targets[target]->sources[i+1]) + 1);
		strcpy(project->targets[target]->sources[i],
			project->targets[target]->sources[i + 1]);
	}

	g_free(project->targets[target]->sources[
		project->targets[target]->sources_no]);
	project->targets[target]->sources_no--;
}


/*
 * Remove a file from the project and project window
 */
void
gI_project_window_remove_file(
	GtkWidget*							widget,
	gchar*								file
)
{
	glong								i;
	glong								j;
	gI_project*							project;

	project = gI_project_get_current();
	if(!project)
	{
		return;
	}

	for(i = 0; i < project->targets_no; i++)
	{
		for(j = 0; j < project->targets[i]->sources_no; j++)
		{
			if(!strcmp(project->targets[i]->sources[j], file))
			{
				move_up_entries(project, i, j);

				if(project_window && GTK_WIDGET_VISIBLE(project_window))
				{
					gtk_clist_remove(GTK_CLIST(files_list), j);
				}
			}
		}
	}

	/* it's easier to regenerate the tree, than to correct it */
	if(cfg->prjftree)
	{
		gtk_widget_show(main_window->tree_box);
		gI_project_files_tree(main_window->tree_viewport, project);
	}
}


/*
 * Update the sensitivity of relevant menu items
 */
void
gI_project_window_update_menu(
	GideWindow*							window
)
{
	if(gI_project_get_current() != NULL)
	{
		if(project_window != NULL)
		{
			gI_menus_set_sensitive(window, "/Project/Edit Project", FALSE);
			gI_menus_set_sensitive(window, "/Project/Close Project", FALSE);
			gI_menus_set_sensitive(window, "/Project/Build Project", FALSE);
			gI_menus_set_sensitive(window, "/Project/Create Makefile", FALSE);
		}
		else
		{
			gI_menus_set_sensitive(window, "/Project/Edit Project", TRUE);
		}
		gI_menus_set_sensitive(window, "/Project/New Project", FALSE);
	}
	else
	{
		gI_menus_set_sensitive(window, "/Project/Edit Project", FALSE);
		if(project_window != NULL)
		{
			gI_menus_set_sensitive(window, "/Project/New Project", FALSE);
		}
		else
		{
			gI_menus_set_sensitive(window, "/Project/New Project", TRUE);
		}
	}

}
