/* gnome-sql - GUI front-end
 * Copyright (c) 1998 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gsqlfe.h"
#include <gda-command.h>
#include <gda-recordset.h>

/* function prototypes */
static void close_current_browser (GtkWidget *, gpointer );
static void refresh_grids (GtkWidget *, gpointer );
static void select_table_cb (GtkWidget *, gpointer );

/* global variables */
extern gint l_openconns;
static GList *l_openwins = NULL;

/* private functions */
static void
close_current_browser (GtkWidget *w, gpointer data)
{
  struct GSQLFE_BrowserInfo *info = browser_get_current_window();
  if (info != NULL)
    {
      gint current = gtk_notebook_page_num(GTK_NOTEBOOK(ui_get_work_area()),
                                           info->tab);
      gtk_notebook_remove_page(GTK_NOTEBOOK(ui_get_work_area()), current);
      l_openwins = g_list_remove(l_openwins, (gpointer) info);
      g_free((gpointer) info);
      l_openconns--;
    }
}

static void
refresh_grids (GtkWidget *w, gpointer data)
{
  struct GSQLFE_BrowserInfo *info;
  /* select_table_cb() expects the table list widget as its 
     first argument and the current browser as the second one */
  info = browser_get_current_window();
  if (info != NULL && info->type == GSQLFE_BROWSER_TABLES)
    select_table_cb(GTK_WIDGET(info->list), (gpointer) info);
}

static void
select_table_cb (GtkWidget *w, gpointer data)
{
  struct GSQLFE_BrowserInfo *info = (struct GSQLFE_BrowserInfo *) data;
  if (w != NULL && info != NULL)
    {
      GtkObject *list_item;
#ifdef HAVE_GDA
      Gda_Recordset* rowset;
#else      
      GnomeSqlRowset *rowset;
#endif      
      gchar *table_name, *query;
      GList *node = g_list_first(GTK_LIST(w)->selection);
      if (node != NULL)
        {
          /* get table name */
          list_item = GTK_OBJECT(node->data);
          table_name = gtk_object_get_data(list_item, glb_list_item_data_key);
          /* show table's description */
#ifdef HAVE_GDA
	  rowset = gda_connection_open_schema(info->db, GDA_Connection_GDCN_SCHEMA_COLS, GDA_Connection_TABLE_NAME, table_name);
#else	    
          rowset = gnome_sql_describe_table(info->db, table_name);
#endif
          if (rowset != NULL)
            info->grid1 = grid_show_resultset(info->grid1, rowset);
          else
            {
              info->grid1 = grid_new_empty(info->grid1);
#ifdef HAVE_GDA
	      ui_show_error("Dummy Error Placeholder");
#else	      
              ui_show_error(gnome_sql_get_last_error(info->db));
#endif
            }
#ifdef HAVE_GDA
	  gda_recordset_close(rowset);
#else	  
          gnome_sql_close_rowset(info->db, rowset);
#endif
          gtk_scrolled_window_add_with_viewport(
                     GTK_SCROLLED_WINDOW(info->grid_scroll1), info->grid1);
          /* show table's data */
          query = g_strdup_printf("SELECT * FROM %s", table_name);
#ifdef HAVE_GDA
	  {
	    Gda_Command* cmd;
	    gulong       reccount;
	    
	    cmd = gda_command_new();
	    gda_command_set_connection(cmd, info->db);
	    gda_command_set_text(cmd, query);
	    rowset = gda_command_execute(cmd, &reccount, 0);
	    gda_command_free(cmd);
	  }
#else
	    
          rowset = gnome_sql_open_rowset(info->db, query);
#endif
          if (rowset != NULL)
            info->grid2 = grid_show_resultset(info->grid2, rowset);
          else
            {
              info->grid2 = grid_new_empty(info->grid2);
#ifdef HAVE_GDA
	      ui_show_error("Dummy Error Placeholder");
#else
	      ui_show_error(gnome_sql_get_last_error(info->db));
#endif
            }
          g_free((gpointer) query);
#ifdef HAVE_GDA
	  gda_recordset_close(rowset);
#else
	  gnome_sql_close_rowset(info->db, rowset);
#endif
          gtk_scrolled_window_add_with_viewport(
                     GTK_SCROLLED_WINDOW(info->grid_scroll2), info->grid2);
        }
    }
}

/* close all browser windows related with this connection */
#ifdef HAVE_GDA
void
browser_close_all(Gda_Connection* gdb)
#else     
void
browser_close_all (GnomeDatabase *gdb)
#endif     
{
  GList *node = g_list_first(l_openwins);
  while (node != NULL)
    {
      struct GSQLFE_BrowserInfo *info = 
             (struct GSQLFE_BrowserInfo *) node->data;
      /* move to next item in list */
      node = g_list_next(node);
      if (info != NULL && info->db == gdb)
        {
          gint this_one =
              gtk_notebook_page_num(GTK_NOTEBOOK(ui_get_work_area()),
                                    info->tab);
          gtk_notebook_remove_page(GTK_NOTEBOOK(ui_get_work_area()),
                                   this_one);
          /* free all memory */
          l_openwins = g_list_remove(l_openwins, info);
          g_free((gpointer) info);
        }
    }
}

struct GSQLFE_BrowserInfo *
browser_get_current_window (void)
{
  gint this_one;
  gint current = gtk_notebook_current_page(GTK_NOTEBOOK(ui_get_work_area()));
  if (current >= 0)
    {
      GList *node = g_list_first(l_openwins);
      struct GSQLFE_BrowserInfo *info;
      while (node != NULL)
        {
          info = node->data;
          this_one = gtk_notebook_page_num(GTK_NOTEBOOK(ui_get_work_area()),
                                           info->tab);
          if (current == this_one) 
            return (info);
          node = g_list_next(node);
        }
    }
  return (NULL);
}

/* open a new browser window */
GnomeUIInfo browsertoolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("New"), N_("Create new object"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_NEW, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Refresh"), N_("Refresh grids"),
    refresh_grids, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_REFRESH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Script"), N_("Generate script from object's definition"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_INDEX, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Close"), N_("Close this window"),
    close_current_browser, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_CLOSE, 0, 0, NULL },
  GNOMEUIINFO_END
};

#ifdef HAVE_GDA
void
browser_open_tables(Gda_Connection* gdb)
#else
void
browser_open_tables (GnomeDatabase *gdb)
#endif
{
  struct GSQLFE_BrowserInfo *info;
  gchar label[64], *str;
#ifndef HAVE_GDA
  GList *tbl;
#else
  Gda_Recordset* tbl;
#endif  
  /* check parameters */
  g_return_if_fail(gdb != NULL);
  /* create browser object */
  if ((info = g_malloc(sizeof(struct GSQLFE_BrowserInfo))))
    {
      info->type = GSQLFE_BROWSER_TABLES;
      info->db = gdb;
      /* create browser window */
      info->tab = gtk_table_new(3, 3, FALSE);
#ifdef HAVE_GDA
      str = gda_connection_get_dsn(info->db);
#else
      str = gnome_sql_get_current_database(info->db);
#endif
      sprintf(label, "Tables: %s", str);
      gtk_notebook_append_page(GTK_NOTEBOOK(ui_get_work_area()), info->tab,
                               gtk_label_new(label));
      g_free((gpointer) str);
      /* create common toolbar */
      info->toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, 
                                      GTK_TOOLBAR_ICONS);
      gnome_app_fill_toolbar(GTK_TOOLBAR(info->toolbar), browsertoolbar, NULL);
      gtk_table_attach(GTK_TABLE(info->tab), info->toolbar, 0, 4, 0, 1,
                       GTK_FILL | GTK_SHRINK,
                       GTK_FILL | GTK_SHRINK, 
                       3, 3);
      gtk_widget_show(info->toolbar);
      /* create scrolled window to put list inside */
      info->scrolled_window = gtk_scrolled_window_new(NULL, NULL);
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(info->scrolled_window),
                                     GTK_POLICY_AUTOMATIC,
                                     GTK_POLICY_AUTOMATIC);
      gtk_table_attach(GTK_TABLE(info->tab), info->scrolled_window, 0, 1, 1, 3,
                       GTK_FILL | GTK_EXPAND,
                       GTK_SHRINK | GTK_EXPAND | GTK_FILL,
                       3, 3);
      gtk_widget_show(info->scrolled_window);
      /* create table list */
      info->list = gtk_list_new();
      gtk_signal_connect(GTK_OBJECT(info->list), "selection_changed",
                         GTK_SIGNAL_FUNC(select_table_cb), (gpointer) info);
      gtk_list_set_selection_mode(GTK_LIST(info->list), GTK_SELECTION_SINGLE);
      gtk_scrolled_window_add_with_viewport(
                     GTK_SCROLLED_WINDOW(info->scrolled_window), info->list);
#ifdef HAVE_GDA
      tbl = gda_connection_open_schema(info->db, GDA_Connection_GDCN_SCHEMA_TABLES, 0, 0);
      if (tbl == NULL)
        ui_show_error("Dummy Error Placeholder");
      else ui_fill_list(GTK_LIST(info->list), tbl);
#else      
      tbl = gnome_sql_get_tables(info->db);
      if (tbl == NULL)
        ui_show_error(gnome_sql_get_last_error(info->db));
      else ui_fill_list(GTK_LIST(info->list), tbl);
#endif
      gtk_widget_show(info->list);
      /* create grid 1 */
      info->grid_scroll1 = gtk_scrolled_window_new(NULL, NULL);
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(info->grid_scroll1),
                                     GTK_POLICY_AUTOMATIC,
                                     GTK_POLICY_AUTOMATIC);
      gtk_table_attach(GTK_TABLE(info->tab), info->grid_scroll1, 1, 4, 1, 2,
                       GTK_SHRINK | GTK_EXPAND | GTK_FILL,
                       GTK_SHRINK | GTK_EXPAND | GTK_FILL,
                       3, 3);
      gtk_widget_show(info->grid_scroll1);
      info->grid1 = grid_new_empty(NULL);
      gtk_scrolled_window_add_with_viewport(
                     GTK_SCROLLED_WINDOW(info->grid_scroll1), info->grid1);
      /* create grid 2 */
      info->grid_scroll2 = gtk_scrolled_window_new(NULL, NULL);
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(info->grid_scroll2),
                                     GTK_POLICY_AUTOMATIC,
                                     GTK_POLICY_AUTOMATIC);
      gtk_table_attach(GTK_TABLE(info->tab), info->grid_scroll2, 1, 4, 2, 3,
                       GTK_SHRINK | GTK_EXPAND | GTK_FILL,
                       GTK_SHRINK | GTK_EXPAND | GTK_FILL,
                       3, 3);
      gtk_widget_show(info->grid_scroll2);
      info->grid2 = grid_new_empty(NULL);
      gtk_scrolled_window_add_with_viewport(
                     GTK_SCROLLED_WINDOW(info->grid_scroll2), info->grid2);
      /* register this window */
      gtk_notebook_set_page(GTK_NOTEBOOK(ui_get_work_area()), l_openconns);
      l_openwins = g_list_append(l_openwins, (gpointer) info);
      gtk_widget_show(info->tab);
      l_openconns++;
    }
  else ui_show_error("Cannot allocate %d bytes", 
                     sizeof(struct GSQLFE_BrowserInfo));
}









