/* gnome-sql - GUI front-end
 * Copyright (c) 1998 by Rodrigo Moya

 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gsqlfe.h"

/* global variables */
static gchar *grid_cols[] = { "" };

/* private functions */
static GtkWidget *
create_grid (gint cnt, gchar *titles[])
{
  GtkWidget *grid;
  /* create widget */
  grid = gtk_clist_new_with_titles(cnt, titles);
  /* set options */
  gtk_clist_set_selection_mode(GTK_CLIST(grid), GTK_SELECTION_SINGLE);
  gtk_clist_moveto(GTK_CLIST(grid), 0, 0, 0.0, 0.0);
  gtk_widget_show(grid);
  return (grid);
}

/* create new DB grid */
GtkWidget *
grid_new_empty (GtkWidget *old_grid)
{
  GtkWidget *grid;
  /* destroy old grid */
  if (old_grid != NULL)
    gtk_widget_destroy(old_grid);
  /* create new one */
  grid = create_grid(1, grid_cols);
  return (grid);
}

#ifdef HAVE_GDA
GtkWidget*
grid_show_resultset (GtkWidget *old_grid, Gda_Recordset *rowset)
{
  GtkWidget *grid = NULL;
  GPtrArray *columns;
  gchar *col_name;
  gboolean first = TRUE;
  register gint cnt, total_fields;

  /* check parameters */
  g_return_val_if_fail(rowset != NULL, old_grid);

  /* destroy the old grid */
  if (old_grid != NULL)
    gtk_widget_destroy(old_grid);

  /* create result set columns array */
  columns = g_ptr_array_new();
  if (columns != NULL)
    {
      gulong position = gda_recordset_move(rowset, 1, 0);
      /* add data to grid */
      while (position != GDA_RECORDSET_INVALID_POSITION && !gda_recordset_eof(rowset))
	{
	  GPtrArray *row;
	  GDA_Field *field;
	  register int col;
	  row = g_ptr_array_new();
	  total_fields = gda_recordset_rowsize(rowset);
	  /* if first time, set columns */
	  if (first)
	    {
	      for (cnt = 0; cnt < total_fields; cnt++)
		{
		  col_name = gda_recordset_field_idx(rowset, cnt)->name;
		  if (col_name != NULL)
		    g_ptr_array_add(columns, (gconstpointer) col_name);
		}
	      /* create grid */
	      grid = create_grid(total_fields, (gchar **) columns->pdata);
	      first = FALSE;
	    }
	  /* add row to grid */
	  for (col = 0; col < total_fields; col++)
	    {
	      field = gda_recordset_field_idx(rowset, col);
	      if (field != NULL)
		{
		  g_ptr_array_add(row, gda_stringify_value(0, 0, field));
		  gtk_clist_set_column_auto_resize(GTK_CLIST(grid), col, TRUE);
		}
	    }
	  gtk_clist_freeze(GTK_CLIST(grid));
	  gtk_clist_append(GTK_CLIST(grid), row->pdata);
	  gtk_clist_thaw(GTK_CLIST(grid));
	  g_ptr_array_free(row, TRUE);
	  position = gda_recordset_move(rowset, 1, 0);
	}
      /* if not EOF, there was an error */
      if (!gda_recordset_eof(rowset))
	ui_show_error("Dummy Error Placeholder");
      g_ptr_array_free(columns, TRUE);
    }
  /* create grid if not yet */
  if (grid == NULL) grid = grid_new_empty(NULL);
  return (grid);
}

#else  

/* populate a grid with all rows in a result set */
GtkWidget *
grid_show_resultset (GtkWidget *old_grid, GnomeSqlRowset *rowset)
{
  GtkWidget *grid = NULL;
  GPtrArray *columns;
  gchar *col_name;
  gboolean first = TRUE;
  register gint cnt, total_fields;
  /* check parameters */
  g_return_val_if_fail(rowset != NULL, old_grid);
  /* destroy the old grid */
  if (old_grid != NULL)
    gtk_widget_destroy(old_grid);
  /* create result set columns array */
  columns = g_ptr_array_new();
  if (columns != NULL)
    {
      gboolean fetched = gnome_sql_rowset_fetch_first(rowset);
      /* add data to grid */
      while (fetched)
	{
	  GPtrArray *row;
	  gchar *field;
	  register int col;
	  row = g_ptr_array_new();
	  total_fields = gnome_sql_rowset_get_field_count(rowset);
	  /* if first time, set columns */
	  if (first)
	    {
	      for (cnt = 0; cnt < total_fields; cnt++)
		{
		  col_name = gnome_sql_rowset_get_field_name(rowset, cnt);
		  if (col_name != NULL)
		    g_ptr_array_add(columns, (gconstpointer) col_name);
		}
	      /* create grid */
	      grid = create_grid(total_fields, (gchar **) columns->pdata);
	      first = FALSE;
	    }
	  /* add row to grid */
	  for (col = 0; col < total_fields; col++)
	    {
	      field = gnome_sql_rowset_get_field_value(rowset, 0, col);
	      if (field != NULL)
		{
		  g_ptr_array_add(row, field);
		  gtk_clist_set_column_auto_resize(GTK_CLIST(grid), col, TRUE);
		}
	    }
	  gtk_clist_freeze(GTK_CLIST(grid));
	  gtk_clist_append(GTK_CLIST(grid), row->pdata);
	  gtk_clist_thaw(GTK_CLIST(grid));
	  g_ptr_array_free(row, TRUE);
	  fetched = gnome_sql_rowset_fetch_next(rowset);
	}
      /* if not EOF, there was an error */
      if (!gnome_sql_rowset_is_eof(rowset))
	ui_show_error(gnome_sql_get_last_error(rowset->database));
      g_ptr_array_free(columns, TRUE);
    }
  /* create grid if not yet */
  if (grid == NULL) grid = grid_new_empty(NULL);
  return (grid);
}
#endif      
