/* gnome-sql - GUI front-end
 * Copyright (c) 1998 by Rodrigo Moya

 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gsqlfe.h"
#ifdef HAVE_GDA
#include <gda-command.h>
#endif

/* global variables */
GList *l_last_commands = NULL;  /* not static, because settings.c needs it */

/* private functions */
static void
select_dialog_clicked_cb (GnomeDialog *dialog, gint button_number, 
                          gpointer data)
{
  GtkList *list = (GtkList *) data;
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  switch (button_number)
    {
    case 0 :						/* 'Ok' button */
      if (info != NULL && list->selection != NULL)
        {
          GList *node = g_list_first(list->selection);
          if (node != NULL && node->data != NULL)
            {
              gchar *command;
              GtkObject *list_item;
              /* get command string */
              list_item = GTK_OBJECT(node->data);
              command = gtk_object_get_data(list_item, glb_list_item_data_key);
              if (command != NULL)
                {
                  /* set buffer text */
                  gtk_editable_delete_text(GTK_EDITABLE(info->cmd_win), 0,
                               gtk_text_get_length(GTK_TEXT(info->cmd_win)));
                  gtk_text_freeze(GTK_TEXT(info->cmd_win));
                  gtk_text_insert(GTK_TEXT(info->cmd_win), NULL, NULL, NULL,
                                  command, strlen(command));
                  gtk_text_thaw(GTK_TEXT(info->cmd_win));
                }
            }
        }
    case 1 :						/* 'Cancel' button */
      gnome_dialog_close(dialog);
      break;
    }
}

/* clipboard stuff */
void
sql_clear_window (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      /* empty SQL command window */
      gtk_editable_delete_text(GTK_EDITABLE(info->cmd_win), 0,
                               gtk_text_get_length(GTK_TEXT(info->cmd_win)));
      /* empty grid */
      info->grid = grid_new_empty(info->grid);
      gtk_scrolled_window_add_with_viewport(
                     GTK_SCROLLED_WINDOW(info->scrolled_window), info->grid);
    }
}

void
sql_copy_text (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    gtk_editable_copy_clipboard(GTK_EDITABLE(info->cmd_win));
}

void
sql_cut_text (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    gtk_editable_cut_clipboard(GTK_EDITABLE(info->cmd_win));
}

void
sql_load_script (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      gchar *filename = ui_get_file_name("Select SQL Script");
      if (filename != NULL)
	{
	  ui_load_file_in_text(GTK_TEXT(info->cmd_win), filename);
	  /* free memory */
	  g_free((gpointer) filename);
	}
    }
}

void
sql_paste_text (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    gtk_editable_paste_clipboard(GTK_EDITABLE(info->cmd_win));
}

/* return last executed command */
gchar *
sql_get_last_command (void)
{
  GList *node = g_list_last(l_last_commands);
  return (node != NULL ? (gchar *) node->data : NULL);
}

/* requery grid */
void
sql_refresh_grid (GtkWidget *w, gpointer data)
{
  gchar *command;
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      command = sql_get_last_command();
      if (command != NULL)
        {
#ifndef HAVE_GDA
          GnomeSqlRowset *rowset = gnome_sql_open_rowset(info->db, command);
#else
	  Gda_Command*   cmd    = gda_command_new;
	  Gda_Recordset* rowset;
	  gulong         reccount;
	  
	  gda_command_set_text(cmd, sql_get_last_command);
	  gda_command_set_connection(cmd, info->db);
	  rowset = gda_command_execute(command, &reccount, 0);
#endif	  
          if (rowset != NULL)
            {
              info->grid = grid_show_resultset(info->grid, rowset);
              gtk_scrolled_window_add_with_viewport(
                     GTK_SCROLLED_WINDOW(info->scrolled_window), info->grid);
#ifdef HAVE_GDA
	      gda_recordset_close(rowset);
#else
	      gnome_sql_close_rowset(info->db, rowset);
#endif
            }
          else
#ifdef HAVE_GDA
	    ui_show_error("Dummy Error Placeholder");
#else
	    ui_show_error(gnome_sql_get_last_error(info->db));
#endif	    
        }
    }
}

/* run SQL buffer as a script */
void
sql_run_script (GtkWidget *w, gpointer data)
{
  gchar *script;
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      /* get text from command window */
      script = gtk_editable_get_chars(GTK_EDITABLE(info->cmd_win), 0, 
                           gtk_text_get_length(GTK_TEXT(info->cmd_win)));
      if (script != NULL)
        {
          gchar *token = strtok(script, ";");
          while (token != NULL)
            {
#ifndef HAVE_GDA
              switch (gnome_sql_execute_command(info->db, token))
                {
                case GNOME_SQL_RESULT_ERROR :
                  ui_show_error(gnome_sql_get_last_error(info->db));
                  break;
                default :
                  sql_store_command(token);
                }
              /* get next token */
#else
	      gulong reccount;
	      Gda_Command* cmd = gda_command_new();
	      gda_command_set_connection(cmd, info->db);
	      gda_command_set_text(cmd, token);
	      if (!gda_command_execute(info->db, &reccount, token))
		ui_show_error("Dummy Error Placeholder");
	      else
		sql_store_command(token);
#endif	      
              token = strtok(NULL, ";");
            }
        }
    }
}

/* save SQL window to file */
void
sql_save_script (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      gchar *buffer, *filename;
      buffer = gtk_editable_get_chars(GTK_EDITABLE(info->cmd_win), 0, 
				      gtk_text_get_length(GTK_TEXT(info->cmd_win)));
      if (buffer != NULL)
	{
	  /* prompt user for file name */
	  filename = ui_get_file_name("Save SQL Buffer");
	  if (filename != NULL)
	    {
	      FILE *file = fopen(filename, "w");
	      if (file != NULL)
		{
		  fwrite(buffer, gtk_text_get_length(GTK_TEXT(info->cmd_win)), 1,
			 file);
		  if (ferror(file))
		    ui_show_error("Error writing to file %s", filename);
		  /* close file */
		  fclose(file);
		}
	      else ui_show_error("Error opening file %s", filename);
	      /* free memory */
	      g_free((gpointer) filename);
	    }
	}
    }
}

/* let the user choose one of the last executed commands */
void
sql_select_command (GtkWidget *w, gpointer data)
{
  GtkWidget *dialog, *scrolled_window, *list;
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      /* create the dialog box */
      dialog = gnome_dialog_new("Select SQL Command",
                                             GNOME_STOCK_BUTTON_OK,
                                             GNOME_STOCK_BUTTON_CANCEL,
                                             NULL);
      gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);              
      /* create scrolled window to put list inside */
      scrolled_window = gtk_scrolled_window_new(NULL, NULL);
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
                                     GTK_POLICY_AUTOMATIC,
                                     GTK_POLICY_AUTOMATIC);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), scrolled_window,
                         TRUE, TRUE, GNOME_PAD);
      gtk_widget_show(scrolled_window);
      /* create table list */
      list = gtk_list_new();
      gtk_signal_connect(GTK_OBJECT(dialog), "clicked",
                         GTK_SIGNAL_FUNC(select_dialog_clicked_cb), 
                         (gpointer) list); 
      if (l_last_commands != NULL)
        ui_fill_list(GTK_LIST(list), l_last_commands);
      gtk_list_set_selection_mode(GTK_LIST(list), GTK_SELECTION_SINGLE);
      gtk_scrolled_window_add_with_viewport(
                     GTK_SCROLLED_WINDOW(scrolled_window), list);
      gtk_widget_show(list);
      /* finally, show the dialog box */
      gtk_widget_show(dialog);
    }
}

/* store a command on linked list */
void
sql_store_command (const gchar *cmd)
{
  gchar *new_str;
  g_return_if_fail(cmd != NULL);
  if ((new_str = g_strdup(cmd)))
    {
      new_str = g_strstrip(new_str);
      l_last_commands = g_list_append(l_last_commands, (gpointer) new_str);
      if (g_list_length(l_last_commands) > MAX_COMMAND_LIST)
        {
          GList *node = g_list_first(l_last_commands);
          l_last_commands = g_list_remove(l_last_commands, node->data);
        }
    }
  else ui_show_error("Cannot allocate %ld bytes", strlen(cmd));
}
