/* GNOME DB libary
 * Copyright (C) 1998,1999 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnome.h>
#include <gda.h>
#include <libgnorba/gnorba.h>

#include "gda-connection.h"
#include "gda-server.h"

static GHashTable* factories = 0;


static guint
hash_id_hash(gconstpointer data)
{
  return (guint)data;
}

static gint
guint_id_guint(gconstpointer d1, gconstpointer d2)
{
  return d1 < d2 ? -1 : d1 == d2 ? 0 : 1;
}

/*
 * this function will change as soon as the client side
 * error objects are here
 */
static gint
Exception(CORBA_Environment *ev)
{
  switch (ev->_major)
    {
    case CORBA_SYSTEM_EXCEPTION:
      fprintf(stderr,"CORBA system exception %s.\n", CORBA_exception_id(ev));
      return -1;
    case CORBA_USER_EXCEPTION:
      fprintf(stderr,"CORBA user exception %s.\n", CORBA_exception_id(ev));
      return -1;
    default:
    }
  return 0;
}

static int
get_corba_connection(Gda_Connection* cnc)
{
  CORBA_Environment ev;
  GDA_ConnectionFactory factory;

    if (!factories)
    factories = g_hash_table_new(hash_id_hash, guint_id_guint);
  factory = g_hash_table_lookup(factories, cnc->provider);
  if (!factory)
    {
      factory = goad_server_activate_with_id(NULL, cnc->provider, 0, NULL);
    }
  CORBA_exception_init(&ev);
  if (!factory)
    {
      fprintf(stderr,"Cannot open factory\n");
      return -1;
    }
  cnc->connection = GDA_ConnectionFactory_create(factory, &ev);
  if (Exception(&ev) < 0)
    return -1;
  return 0;
}



/**
 * gda_connection_new:
 * @orb: The ORB
 *
 * Allocates space for a client side connection object
 *
 * Returns: the pointer to the allocated object
 */

Gda_Connection*
gda_connection_new(CORBA_ORB orb)
{
  Gda_Connection* cnc;
  
  g_return_val_if_fail(orb != NULL, NULL);
  
  cnc = g_new0(Gda_Connection, 1);
  return cnc;
}

/**
 * gda_connection_free:
 * @cnc: the connection
 *
 * If the connection is open the connection is closed and all
 * associated objects (commands, recordsets, errors) are deleted. The
 * memory is freed. The connection object cannot be used any more.
 *
 */
void
gda_connection_free(Gda_Connection* cnc)
{
  if (cnc->is_open)
    {
      if (cnc->commands)
	g_warning("Commands still associated with gda_connection");
      if (cnc->recordsets)
	g_warning("Recordsets still associated with gda_connection");
      if (cnc->errors)
	g_warning("Errors still associated with gda_connection");
      gda_connection_close(cnc);
    }
  if (cnc->provider)
    g_free(cnc->provider);
  if (cnc->default_db)
    g_free(cnc->default_db);
  if (cnc->database)
    g_free(cnc->database);
  if (cnc->user)
    g_free(cnc->user);
  if (cnc->passwd)
    g_free(cnc->passwd);
  g_free(cnc);
}  

/**
 * gda_connection_list_provider:
 *
 * Searches the GNORBA database for GDA servers and returns a Glist of 
 * Gda_Server structs. 
 *
 * Returns: a GList of GDA servers structs
 */
GList*
gda_connection_list_providers(void)
{
  GList*          retval = 0;
  GoadServerList* servlist;
  GoadServer*     slist;
  gint            i;
  Gda_Server*     server;
  
  servlist = goad_server_list_get();
  slist = servlist->list;
  for (i = 0; slist[i].repo_id; i++)
    {
      if (strcmp(*slist[i].repo_id, "IDL:GDA/ConnectionFactory:1.0")  == 0)
	{
	  fprintf(stderr,"Found server '%s'\n", slist[i].server_id);
	  server = gda_server_new();
	  server->name     = g_strdup(slist[i].server_id);
	  server->location = g_strdup(slist[i].location_info);
	  server->comment  = g_strdup(slist[i].description);
	  server->repo_id  = g_strdup(*(slist[i].repo_id));
	  server->type     = slist[i].type;
	  retval = g_list_append(retval, server);
	}
    }
  goad_server_list_free(servlist);
  return retval;
}

/**
 * gda_connection_set_provider:
 * @cnc: connection object
 * @name: name of the provider
 *
 * Registers @name as the provider for this connection. This
 * should be the GOAD id of the CORBA server to
 * use
 *
 */

void
gda_connection_set_provider(Gda_Connection* cnc, gchar* name)
{
  g_return_if_fail(cnc != 0);
  g_return_if_fail(name != 0);

  if (cnc->provider)
    g_free(cnc->provider);
  cnc->provider = g_strdup(name);
}

/**
 * gda_connection_get_provider:
 * @cnc: Connection object
 *
 * Retuns the provider used for this connection
 *
 * Returns: a reference to the provider name
 */
 
const gchar*
gda_connection_get_provider(Gda_Connection* cnc)
{
  g_return_val_if_fail(cnc != NULL, NULL);

  return g_strdup(cnc->provider);
}

/**
 * gda_connection_list_datasources:
 * @cnc: the connection object:
 *
 * List all datasources which can be used with this connection object.
 *
 * Returns: a GList with the DSNs of all known data sources.
 */
GList*
gda_connection_list_datasources(Gda_Connection* cnc)
{
  CORBA_Environment       ev;
  GDA_Connection_DSNlist* dsnlist;
  GList*                  retval = 0;
  gint                    idx;
  
  CORBA_exception_init(&ev);
  if (!cnc->connection)
    get_corba_connection(cnc);
  dsnlist = GDA_Connection_listSources(cnc->connection, &ev);
  for (idx = 0; idx < dsnlist->_length; idx++)
    {
      gchar* dsn = g_strdup(dsnlist->_buffer[idx]);
      size_t endidx = strcspn(dsn, " \t");
      dsn[endidx] = '\0';
      retval = g_list_append(retval, dsn);
    }
  CORBA_free(dsnlist);
  return retval;
}

/**
 * gda_connection_set_default_db:
 * @cnc: the connection object
 * @dsn: the DSN of the default database
 *
 * Registers the DSN as the name of the default DB to access when the connection is opened
 *
 */
void
gda_connection_set_default_db(Gda_Connection* cnc, gchar* dsn)
{
  g_return_if_fail(cnc != NULL);
  g_return_if_fail(dsn != NULL);

  if (cnc->default_db)
    g_free(cnc->default_db);
  cnc->default_db = g_strdup(dsn);
}

/**
 * gda_connection_open:
 * @cnc: the connection object which describes the server and
 *        the database name to which the connection shuld be opened
 * @dsn: The DSN of the database. Can be NULL.
 * @user: The username for authentication to the database. Can be NULL.
 * @pwd: The password for authentication to the database. Can be NULL.
 * 
 * The function activates the correct CORBA server (defined with he
 * #gda_connection_set_provider() function. Then it
 * tries to open the database using the DSN or the default database
 * as a data source. If @user or @pwd is not NULL, it will overwrite the
 * appropriate entry in the DSN passed as @par2. Entries in the DSN have the form
 * <key> = <value> seperated from the database name . Currently the DSN is not
 * parsed.
 *
 * Returns: 0 on success, -1 on error
 */
gint
gda_connection_open(Gda_Connection* cnc, gchar* dsn, gchar* user, gchar* pwd)
{
  gchar*                db_to_use;
  gchar*                user_to_use;
  gchar*                pwd_to_use;
  CORBA_Environment     ev;
  gint                  rc;
  
  g_return_val_if_fail(cnc != NULL, -1);
  g_return_val_if_fail(cnc->connection == CORBA_OBJECT_NIL, -1);
  g_return_val_if_fail(user != NULL, -1);
  g_return_val_if_fail(pwd != NULL, -1);

  if (!dsn)
    db_to_use = cnc->default_db;
  else
    db_to_use = dsn;
  user_to_use = user;
  pwd_to_use = pwd;

  cnc->database = g_strdup(db_to_use);
  cnc->user = g_strdup(user_to_use);
  cnc->passwd = g_strdup(pwd_to_use);

  if (get_corba_connection(cnc) < 0)
      return -1;

  rc = GDA_Connection_open(cnc->connection, db_to_use, user_to_use, pwd_to_use, &ev);
  if (Exception(&ev) < 0)
    return -1;
  
  if (rc < 0)
    {
      fprintf(stderr,"connection opend failed\n");
      return -1;
    }
  cnc->is_open = 1;
  return 0;
}

/**
 * gda_connection_close:
 * @cnc: The connection object which should be closed
 *
 * Closes the connection object and all associated #Gda_Recordset objects.
 * #Gda_Command objects are not closed, but can't be used anymore. Transactions
 * pending on this objects are aborted and an error is returned.
 *
 * Returns: 0 on success, -1 on error
 */
gint
gda_connection_close(Gda_Connection* cnc)
{
  CORBA_Environment     ev;
  gint                  rc = 0;
  
  g_return_val_if_fail(cnc != NULL, -1);
  g_return_val_if_fail(cnc->connection != NULL, -1);

  /* Here we should notify all recordset and command objects */
  
  CORBA_exception_init(&ev);
  GDA_Connection_close(cnc->connection, &ev);
  if (Exception(&ev) < 0)
    rc = -1;
  cnc->is_open = 0;
  CORBA_Object_release(cnc->connection, &ev);
  cnc->connection = CORBA_OBJECT_NIL;
  Exception(&ev);
  return rc;
}

/**
 * gda_connection_open_schema:
 * @cnc: Connection object
 * @t: Query type
 * @Varargs: Constraint values 
 * 
 * Retrieves meta data about the data source to which the
 * connection object is connected.
 * The @constraints is a list of enum/string pairs.
 * The end of the list must be marked by the special value
 * GDA_Connection_no_CONSTRAINT
 *
 * Returns: The recordset which holds the results.
 */

 

Gda_Recordset*
gda_connection_open_schema(Gda_Connection* cnc, GDA_Connection_QType t, ...)
{

  va_list ap;
  CORBA_Environment ev;
  GDA_Connection_ConstraintSeq* constraints;
  GDA_Connection_ConstraintType constraint_type;
  GDA_Connection_Constraint*    c;
  GList*                        tmp_constraints = 0;
  GList*                        ptr;
  gint                          count = 0;
  gint                          index;
  Gda_Recordset*                rs;
  CORBA_Object                  corba_rs;
  
  g_return_val_if_fail(cnc != 0, 0);
  g_return_val_if_fail(cnc->connection != 0, 0);

  va_start(ap, t);
  while (1)
    {
      gchar* constraint_value;

      constraint_type = va_arg(ap, int);
      if (constraint_type == GDA_Connection_no_CONSTRAINT)
	break;
      constraint_value = va_arg(ap, char*);
      c = g_new0(GDA_Connection_Constraint, 1);
      c->ctype = constraint_type;
      c->value = g_strdup(constraint_value);
      tmp_constraints = g_list_append(tmp_constraints, c);
      count++;
    }
  constraints =  GDA_Connection_ConstraintSeq__alloc();
  constraints->_buffer = CORBA_sequence_GDA_Connection_Constraint_allocbuf(count);
  constraints->_length = count;

  ptr = tmp_constraints;
  index = 0;

  while (count)
    {
      memcpy(&constraints->_buffer[index], ptr->data,
	     sizeof(GDA_Connection_Constraint));
      index++;
      count--;
      ptr = g_list_next(ptr);
    }
  CORBA_exception_init(&ev);
  fprintf(stderr,"client: gda_connection_open_schema: constraints._maximum = %d\n", constraints->_maximum);
  fprintf(stderr,"                                    constraints._length  = %d\n", constraints->_length);
  
  corba_rs = GDA_Connection_openSchema(cnc->connection, t, constraints, &ev);
  
  /* @mla@ FIXME: free allocated storage */
  if (Exception(&ev) < 0)
    return 0;
  rs = gda_recordset_new();
  rs->eof = 0;
  rs->bof = 0;
  rs->corba_rs = corba_rs;
  return rs;
}

/**
 * gda_connection_get_errors:
 * @cnc: the connection object
 *
 * Returns a list of all errors for this connection object. This
 * function also clears the error list. The errors are stored in LIFO
 * order, so the last error which happend is stored as the first
 * element in the list. Advancing the list means to get back in time and 
 * retrieve earlier errors.
 *
 * The farther away from the client the error happens, the later it
 * is in the queue. If an error happens on the client and then in the
 * CORBA layer and then on the server, you will get the server error
 * first.
 *
 * Returns: a #Glist holding the error objects.
 */
GList*
gda_connection_get_errors(Gda_Connection* cnc)
{
  GList* retval;

  retval = cnc->errors;
  cnc->errors = 0;
  return retval;
}

