/* gnome-sql - GUI front-end
 * Copyright (c) 1998 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gsqlfe.h"

/* global variables */
static GList *l_connections = NULL;
gint l_openconns = 0;

/* private functions */

/* close current connection */
void
db_close_connection (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      if (!gda_connection_close(info->db))
        {
          gint current = 
              gtk_notebook_page_num(GTK_NOTEBOOK(ui_get_work_area()),
                                    info->tab);
          /* remove all related windows */
          browser_close_all(info->db);
          /* remove the connection itself */
          l_connections = g_list_remove(l_connections, (gpointer) info);
	  gda_connection_free(info->db);
          g_free((gpointer) info);
          gtk_notebook_remove_page(GTK_NOTEBOOK(ui_get_work_area()),
                                   current);
          l_openconns--;
        }
      else ui_show_error("Cannot close connection");
    }
}

/* close all open connections */
void
db_close_all_connections (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info;
  gint current;
  GList *node = g_list_first(l_connections);
  while (node != NULL)
    {
      info = (struct GSQLFE_ConnectionInfo *) node->data;
      /* move to next item in list */
      node = g_list_next(node);
      if (!gda_connection_close(info->db))
        {
          current = gtk_notebook_page_num(GTK_NOTEBOOK(ui_get_work_area()),
                                          info->tab);
          /* remove all related windows */
          browser_close_all(info->db);
          /* remove the connection itself */
          l_connections = g_list_remove(l_connections, (gpointer) info);
          g_free((gpointer) info);
          gtk_notebook_remove_page(GTK_NOTEBOOK(ui_get_work_area()), current);
          l_openconns--;
        }
      else ui_show_error("Cannot close connection");
    }
}

/* commit changes */
void
db_commit (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      ui_show_error("Not already implemented");
    }
}

/* execute command */
void
db_exec_command (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      gchar *query;
      Gda_Recordset *rowset;
      /* open result set */
      query = gtk_editable_get_chars(GTK_EDITABLE(info->cmd_win), 0, 
                           gtk_text_get_length(GTK_TEXT(info->cmd_win)));
      if (query != NULL)
        {
          Gda_Command *cmd = gda_command_new();
          gulong reccount;
          gda_command_set_connection(cmd, info->db);
          gda_command_set_text(cmd, query);
          rowset = gda_command_execute(cmd, &reccount, 0);
          if (rowset != NULL)
            {
              info->grid = grid_show_resultset(info->grid, rowset, 
				   GTK_CONTAINER(info->scrolled_window));
              gda_recordset_close(rowset);
	      gda_recordset_free(rowset);
	      gda_command_free(cmd);
              sql_store_command(query);
            }
          else ui_show_error("Error executing query");
        }
    }
}

struct GSQLFE_ConnectionInfo *
db_get_current_connection (void)
{
  gint this_one;
  gint current = gtk_notebook_current_page(GTK_NOTEBOOK(ui_get_work_area()));
  if (current >= 0)
    {
      GList *node = g_list_first(l_connections);
      struct GSQLFE_ConnectionInfo *info;
      while (node != NULL)
        {
          struct GSQLFE_BrowserInfo *brwinfo;
          info = node->data;
          this_one = gtk_notebook_page_num(GTK_NOTEBOOK(ui_get_work_area()),
                                                info->tab);
          if (current == this_one) 
            return (info);
          /* try searching browser windows */
          brwinfo = browser_get_current_window();
          if (brwinfo && brwinfo->db == info->db)
              return (info);
          node = g_list_next(node);
        }
    }
  return (NULL);
}

/* create workspace for new connection */
GnomeUIInfo sqltoolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("Run"), N_("Execute SQL command"),
    db_exec_command, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Refresh"), N_("Refresh grid"),
    sql_refresh_grid, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_REFRESH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear SQL window"),
    sql_clear_window, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_TRASH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Print"), N_("Print grid"),
      NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK, 
      GNOME_STOCK_PIXMAP_PRINT, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Open"), N_("Load script"),
    sql_load_script, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save"), N_("Save current buffer"),
    sql_save_script, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_SAVE, 0, 0, NULL },
    GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Cut"), N_("Cut selected text to clipboard"),
    sql_cut_text, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_CUT, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Copy"), N_("Copy selected text to clipboard"),
    sql_copy_text, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_COPY, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Paste"), N_("Paste text from clipboard"),
    sql_paste_text, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_PASTE, 0, 0, NULL },
  GNOMEUIINFO_END
};

void
db_new_connection (Gda_Connection *gdb)
{
  struct GSQLFE_ConnectionInfo *info;
  gchar label[64];
  /* check parameters */
  g_return_if_fail(gdb != NULL);
  /* add connection to private list */
  info = (struct GSQLFE_ConnectionInfo *) 
          g_malloc(sizeof(struct GSQLFE_ConnectionInfo));
  if (info != NULL)
    {
      /* initialize connection structure */
      info->db = gdb;
      info->tab = gtk_table_new(3, 2, FALSE);
      gtk_widget_show(info->tab);
      sprintf(label, "%s@%s", gda_connection_get_user(gdb),
                              gda_connection_get_dsn(gdb));
      gtk_notebook_append_page(GTK_NOTEBOOK(ui_get_work_area()), info->tab,
                               gtk_label_new(label));
      gtk_notebook_set_page(GTK_NOTEBOOK(ui_get_work_area()), l_openconns);
      info->toolbar = ui_new_toolbar_in_table(info->tab, 0, 0, 2, 1);
      gnome_app_fill_toolbar(GTK_TOOLBAR(info->toolbar), sqltoolbar, NULL);
      info->cmd_win = ui_new_text_in_table(info->tab, 0, 1, 2, 2);
      /* scrolled window to put grid on */
      info->scrolled_window = gtk_scrolled_window_new(NULL, NULL);
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(info->scrolled_window),
                                     GTK_POLICY_AUTOMATIC,
                                     GTK_POLICY_AUTOMATIC);
      gtk_table_attach(GTK_TABLE(info->tab), info->scrolled_window, 0, 2, 2, 3,
                       GTK_SHRINK | GTK_EXPAND | GTK_FILL,
                       GTK_EXPAND | GTK_FILL,
                       3, 3);
      gtk_widget_show(info->scrolled_window);
      /* create grid */
      info->grid = grid_new_empty(NULL, GTK_CONTAINER(info->scrolled_window));
      /* finally add the node to the list */
      l_connections = g_list_append(l_connections, (gpointer) info);
      l_openconns++;
    }
  else ui_show_error("Cannot allocate %d bytes", 
                     sizeof(struct GSQLFE_ConnectionInfo));
}

/* start or stop trace */
void
db_set_trace (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      ui_show_error("Not yet implemented");
    }
}

/* discard all changes made */
void
db_rollback (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      ui_show_error("Not yet implemented");
    }
}

/* start a new transaction */
void
db_start_transaction (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      ui_show_error("Not yet implemented");
    }
}
