/* GNOME DB libary
 * Copyright (C) 1998,1999 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnome.h>
#include <gda.h>

#include "gda-field.h"



#define ENUM_TO_STR(e) case (e): strncpy(bfr, #e, length); break
#define min(a,b)       (a) < (b) ? (a) : (b)



/**
 * gda_fieldtype_2_string:
 * @bfr: bufferspace where the printed name of the type of the field
 * is placed into.
 * @length: the size of the buffer
 * @f: a pointer to the field.
 *
 * Returns: a pointer to the buffer.
 */

gchar* 
gda_fieldtype_2_string(gchar* bfr, gint length, Gda_Field* f)
{
  if (!bfr)
    {
      bfr = g_new0(gchar, 20);
      length = 20;
    }
  switch (f->attributes->gdaType)
    {
      ENUM_TO_STR(GDA_TypeNull);
      ENUM_TO_STR(GDA_TypeBigint);
      ENUM_TO_STR(GDA_TypeBinary);
      ENUM_TO_STR(GDA_TypeBoolean);
      ENUM_TO_STR(GDA_TypeBstr);
      ENUM_TO_STR(GDA_TypeChar);
      ENUM_TO_STR(GDA_TypeCurrency);
      ENUM_TO_STR(GDA_TypeDate);
      ENUM_TO_STR(GDA_TypeDbDate);
      ENUM_TO_STR(GDA_TypeDbTime);
      ENUM_TO_STR(GDA_TypeDbTimestamp);
      ENUM_TO_STR(GDA_TypeDecimal);
      ENUM_TO_STR(GDA_TypeDouble);
      ENUM_TO_STR(GDA_TypeError);
      ENUM_TO_STR(GDA_TypeInteger);
      ENUM_TO_STR(GDA_TypeLongvarbin);
      ENUM_TO_STR(GDA_TypeLongvarchar);
      ENUM_TO_STR(GDA_TypeLongvarwchar);
      ENUM_TO_STR(GDA_TypeNumeric);
      ENUM_TO_STR(GDA_TypeSingle);
      ENUM_TO_STR(GDA_TypeSmallint);
      ENUM_TO_STR(GDA_TypeTinyint);
      ENUM_TO_STR(GDA_TypeUBigint);
      ENUM_TO_STR(GDA_TypeUSmallint);
      ENUM_TO_STR(GDA_TypeVarchar);
      ENUM_TO_STR(GDA_TypeVarbin);
      ENUM_TO_STR(GDA_TypeVarwchar);
      ENUM_TO_STR(GDA_TypeFixchar);
      ENUM_TO_STR(GDA_TypeFixbin);
      ENUM_TO_STR(GDA_TypeFixwchar);
    }
  return bfr;
}


gchar*
gda_stringify_value(gchar* bfr, gint maxlen, Gda_Field* f)
{
  gchar*     retval = 0;
  
  if (bfr)
    retval = bfr;

  if (gda_field_isnull(f))
    return g_strdup(_("<NULL>"));

  switch (f->attributes->gdaType)
    {
    case GDA_TypeNull:
      if (bfr)
	retval = strncpy(bfr, _("<Unknown GDA Type(NULL)>"), maxlen);
      else
	retval = g_strdup(_("<Unknown GDA Type(NULL)>"));
      break;
    case GDA_TypeBigint:
      if (!bfr)
	{
	  retval = g_new0(gchar,20);
	  maxlen = 20;
	}
      g_snprintf(retval, maxlen, "%Ld", gda_field_bigint(f));
      break;
    case GDA_TypeBoolean:
      {
	gchar* retstr;
	if (gda_field_boolean(f))
	  retstr = _("TRUE");
	else
	  retstr = _("FALSE");
	if (bfr)
	  retval = strncpy(bfr, retstr, maxlen);
	else
	  retval = g_strdup(retstr);
      }
      break;
    case GDA_TypeLongvarchar:
    case GDA_TypeVarchar:
      if (!bfr)
	{
	  retval = g_strdup(f->real_value->_u.v._u.lvc);
	}
      else
	{
	  strncpy(retval, f->real_value->_u.v._u.lvc,
		  min(maxlen, strlen(f->real_value->_u.v._u.lvc)));
	  retval[min(maxlen, strlen(f->real_value->_u.v._u.lvc))] = '\0';
	}
      break;
    case GDA_TypeInteger:
    case GDA_TypeSmallint:
      if (!bfr)
	{
	  retval = g_new0(gchar, 20);
	  maxlen = 20;
	}
      g_snprintf(retval, maxlen, "%d", f->real_value->_u.v._u.si);
      break;
    case GDA_TypeChar:
      if (!bfr)
	retval = g_strdup(f->real_value->_u.v._u.lvc);
      else
	{
	  strncpy(retval, f->real_value->_u.v._u.lvc,
		  min(maxlen, strlen(f->real_value->_u.v._u.lvc)));
	  retval[min(maxlen, strlen(f->real_value->_u.v._u.lvc))] = '\0';
	}
      break;
    case GDA_TypeDecimal:
    case GDA_TypeNumeric:
      if (!bfr)
	retval = g_strdup(f->real_value->_u.v._u.lvc);
      else
	{
	  strncpy(retval, f->real_value->_u.v._u.lvc,
		  min(maxlen, strlen(f->real_value->_u.v._u.lvc)));
	  retval[min(maxlen, strlen(f->real_value->_u.v._u.lvc))] = '\0';
	}
      break;
    case GDA_TypeSingle:
      if (!bfr)
	{
	  retval = g_new0(gchar, 20);
	  maxlen = 20;
	}
      g_snprintf(retval, maxlen, "%g", f->real_value->_u.v._u.f);
      break;
    case GDA_TypeDouble:
      if (!bfr)
	{
	  retval = g_new0(gchar, 20);
	  maxlen = 20;
	}
      g_snprintf(retval, maxlen, "%g", f->real_value->_u.v._u.dp);
      break;
    case GDA_TypeBinary:
    case GDA_TypeBstr:
    case GDA_TypeCurrency:
    case GDA_TypeDate:
    case GDA_TypeDbDate:
    case GDA_TypeDbTime:
    case GDA_TypeDbTimestamp:
    case GDA_TypeError:
    case GDA_TypeLongvarbin:
    case GDA_TypeLongvarwchar:
      
    case GDA_TypeTinyint:
    case GDA_TypeUBigint:
    case GDA_TypeUSmallint:
    case GDA_TypeVarbin:
    case GDA_TypeVarwchar:
    case GDA_TypeFixchar:
    case GDA_TypeFixbin:
    case GDA_TypeFixwchar:
      if (!bfr)
	{
	  retval = g_new0(gchar, 128);
	  maxlen = 128;
	}
      printf("stringify for valuetype [%d]'%s' NYI\n",
	     f->attributes->gdaType, gda_fieldtype_2_string(0, 20, f));
      retval[0] = '\0';
      break;
    }
  return retval;
}

/**
 * gda_field_new:
 * Allocate space for a new field object.
 *
 * Returns: the pointer to the new field object.
 */

Gda_Field*
gda_field_new(void)
{
  Gda_Field* rc = g_new0(Gda_Field,1);
  return rc;
}


/**
 * gda_field_free:
 * @f: a pointer to a Gda_Field object
 *
 * Free the memory allocated for this field object.
 *
 */

void
gda_field_free(Gda_Field* f)
{
  g_free(f);
}


/**
 * gda_field_actual_size:
 * @f: a pointer to the field.
 *
 * Calculates the number of bytes the value fo the field needs.
 *
 * Returns: the number of bytes the value fo the field needs or 0 if
 * the field has a NULL value.
 */

gint
gda_field_actual_size(Gda_Field* f)
{
  if (!f || gda_field_isnull(f))
    return 0;
  
  switch (gda_field_type(f))
    {
    case GDA_TypeTinyint:
      return sizeof(CORBA_char);
    case GDA_TypeBigint:
      return sizeof(CORBA_long_long);
    case GDA_TypeBoolean:
      return sizeof(CORBA_boolean);
    case GDA_TypeDate:
      return sizeof(GDA_Date);
    case GDA_TypeDbDate:
      return sizeof(GDA_DbDate);
    case GDA_TypeDbTime:
      return sizeof(GDA_DbTime);
    case GDA_TypeDbTimestamp:
      return sizeof(GDA_DbTimestamp);
    case GDA_TypeCurrency:
    case GDA_TypeDecimal:
    case GDA_TypeNumeric:
      return strlen(gda_field_longvarchar(f));
    case GDA_TypeDouble:
      return sizeof(CORBA_double);
      /* case GDA_TypeError: NYI */
    case GDA_TypeInteger:
      return sizeof(CORBA_long);
    case GDA_TypeVarbin:
    case GDA_TypeVarwchar:
    case GDA_TypeLongvarwchar:
    case GDA_TypeLongvarbin:
      return gda_field_varbin_length(f);
    case GDA_TypeFixbin:
    case GDA_TypeFixwchar:
    case GDA_TypeFixchar:
      return gda_field_fixbin_length(f);
    case GDA_TypeChar:
    case GDA_TypeVarchar:
    case GDA_TypeLongvarchar:
      return strlen(gda_field_longvarchar(f));
    case GDA_TypeSingle:
      return sizeof(CORBA_float);
    case GDA_TypeSmallint:
      return sizeof(CORBA_short);
    case GDA_TypeUBigint:
      return sizeof(CORBA_unsigned_long_long);
    case GDA_TypeUSmallint:
      return sizeof(CORBA_unsigned_short);
    }
  g_warning("gda_field_actual_size: unknown GDA Type %d\n", gda_field_type(f));
  return -1;
};
  
