#include <orb/orbit.h>
#include <ctype.h>
#include <stdio.h>
#include <signal.h>
#include <gnome.h>
#include <libgnorba/gnorba.h>

#include "odbc.h"
#include "sqldriver.h"

#include "gda-odbcsrv.h"
#include "gda-types.h"
#include "gda-connection.h"
#include "gda-command.h"
#include "gda-error.h"
#include "gda.h"


void Exception( CORBA_Environment* ev )
{
  switch( ev->_major )
    {
    case CORBA_SYSTEM_EXCEPTION:
      g_log("GDA ODBC", G_LOG_LEVEL_DEBUG, "CORBA system exception %s.\n",
	       CORBA_exception_id(ev));
      exit ( 1 );
    case CORBA_USER_EXCEPTION:
      g_log("GDA ODBC", G_LOG_LEVEL_DEBUG, "CORBA user exception: %s.\n",
	       CORBA_exception_id( ev ) );
      exit ( 1 );
    default:
      break;
    }
}


      

GDA_Connection
gda_ConnectionFactory_create(PortableServer_Servant _obj,
			     CORBA_Environment*     ev)
{
  GDAConnection_Servant *new_connection_servant;
  GDA_Connection         new_connection;
  GDAConnectionFactory_Servant* cf_servant = (GDAConnectionFactory_Servant*)_obj;
  PortableServer_ObjectId*      oid;
  
  new_connection_servant = g_new0(GDAConnection_Servant, 1);
  new_connection_servant->base._private = 0;
  new_connection_servant->base.vepv     = &poa_gda_connection__vepv;
  new_connection_servant->poa           = cf_servant->poa;
  new_connection_servant->cnc           = gda_odbc_connection_new();

  POA_GDA_Connection__init((PortableServer_Servant) new_connection_servant, ev);
  Exception(ev);
  
  oid = PortableServer_POA_activate_object(cf_servant->poa, new_connection_servant, ev);
  Exception(ev);

  new_connection = PortableServer_POA_servant_to_reference(cf_servant->poa, (PortableServer_Servant)new_connection_servant, ev);
  Exception(ev);

  return new_connection;
}

CORBA_long
gda_Connection_open(PortableServer_Servant _obj,
		    const CORBA_char*            dsn,
		    const CORBA_char*            user,
		    const CORBA_char*            passwd,			  
		    CORBA_Environment*      env)
{
  GDAConnection_Servant* cnc = (GDAConnection_Servant*) _obj;

  if (gda_odbc_connection_open(cnc->cnc, dsn, user, passwd) < 0)
    {
      fprintf(stderr,"gda_Connection_open Errr left\n");
      return -1;
    }
  else
    {
      return 0;
    }
}

CORBA_long
GDA_odbc_begin_transaction (PortableServer_Servant _obj, CORBA_Environment* ev)
{
  GDAConnection_Servant* cnc_servant = (GDAConnection_Servant*)_obj;
  Gda_ODBC_Connection*   cnc = cnc_servant->cnc;
  GDA_NotSupported*      exception;
  
  if (SQLSetConnectOption(cnc->hdbc, SQL_AUTOCOMMIT, SQL_AUTOCOMMIT_OFF) == SQL_SUCCESS)
    return 0;
  exception = GDA_NotSupported__alloc();
  exception->errormsg = CORBA_string_dup("Transactions not Supported");
  CORBA_exception_set(ev, CORBA_USER_EXCEPTION, ex_GDA_NotSupported, exception);
  return -1;
}


CORBA_long
GDA_odbc_commit_transaction (PortableServer_Servant _obj, CORBA_Environment* ev)
{
  GDAConnection_Servant* cnc_servant = (GDAConnection_Servant*)_obj;
  Gda_ODBC_Connection*   cnc = cnc_servant->cnc;
  GDA_NotSupported*      exception;
  
  if (SQLTransact(cnc->henv, cnc->hdbc, SQL_COMMIT) == SQL_SUCCESS)
    return 0;
  exception = GDA_NotSupported__alloc();
  exception->errormsg = CORBA_string_dup("Transactions not Supported");
  CORBA_exception_set(ev, CORBA_USER_EXCEPTION, ex_GDA_NotSupported, exception);
  return -1;
}

CORBA_long
GDA_odbc_rollback_transaction (PortableServer_Servant _obj, CORBA_Environment* ev)
{
  GDAConnection_Servant* cnc_servant = (GDAConnection_Servant*)_obj;
  Gda_ODBC_Connection*   cnc         = cnc_servant->cnc;
  GDA_NotSupported*      exception;
  
  if (SQLTransact(cnc->henv, cnc->hdbc, SQL_ROLLBACK) == SQL_SUCCESS)
    return 0;
  exception = GDA_NotSupported__alloc();
  exception->errormsg = CORBA_string_dup("Transactions not Supported");
  CORBA_exception_set(ev, CORBA_USER_EXCEPTION, ex_GDA_NotSupported, exception);
  return -1;
}

GDA_Recordset
gda_Connection_openSchema   (PortableServer_Servant _obj,
			     const GDA_Connection_QType t,
			     const GDA_Connection_ConstraintSeq * constraints, CORBA_Environment * ev)
{
  GDAConnection_Servant*   cnc = (GDAConnection_Servant*) _obj;
  GDARecordset_Servant*    new_recset_servant;
  GDA_Recordset            new_recset;
  Gda_ODBC_Recordset*      recset;
  PortableServer_ObjectId* oid;
  Gda_ODBC_Error           e;
  
  memset(&e, '\0', sizeof(e));
  if ((recset = gda_odbc_connection_open_schema(cnc->cnc, &e, t, constraints->_buffer, constraints->_length)) == 0)
    {
      gchar bfr[128];
      GDA_NotSupported* exception = GDA_NotSupported__alloc();
      g_snprintf(bfr, sizeof(bfr), "SCHEMA type %d not supported\n", t);
      exception->errormsg = CORBA_string_dup(bfr);
      CORBA_exception_set(ev, CORBA_USER_EXCEPTION, ex_GDA_NotSupported, exception);
      return CORBA_OBJECT_NIL;
    }
  new_recset_servant = g_new0(GDARecordset_Servant, 1);
  new_recset_servant->base._private = 0;
  new_recset_servant->base.vepv = &poa_gda_recordset__vepv;
  new_recset_servant->poa = cnc->poa;
  new_recset_servant->recset = recset;

  POA_GDA_Recordset__init((PortableServer_Servant)new_recset_servant, ev);
  Exception(ev);

  oid = PortableServer_POA_activate_object(new_recset_servant->poa, new_recset_servant, ev);
  Exception(ev);

  new_recset = PortableServer_POA_servant_to_reference(new_recset_servant->poa, new_recset_servant, ev);
  Exception(ev);

  return new_recset;
}

CORBA_long
gda_Connection_close(PortableServer_Servant _obj,
		    CORBA_Environment*      env)
{
  GDAConnection_Servant* cnc = (GDAConnection_Servant*) _obj;
  PortableServer_POA     poa = cnc->poa;
  GDA_Connection         old_connection;

  
  old_connection = PortableServer_POA_servant_to_reference(cnc->poa,
							   (PortableServer_Servant)cnc, env);
  Exception(env);
  
  if (gda_odbc_connection_close(cnc->cnc) < 0)
    {
      fprintf(stderr," gda_odbc_connection_close returns an error\n");
      return -1;
    }
  PortableServer_POA_deactivate_object(poa,
				       PortableServer_POA_servant_to_id(poa,cnc, env),
				       env);
  Exception(env);
  POA_GDA_Connection__fini((POA_GDA_Connection*)cnc, env);
  Exception(env);
  return 0;
}

GDA_Command
gda_Connection_createCommand(PortableServer_Servant _obj,
			     CORBA_Environment*     ev)
{
  
  GDACommand_Servant*      new_command_servant;
  GDA_Command              new_command;
  GDAConnection_Servant*   cnc                  = (GDAConnection_Servant*)_obj;
  PortableServer_ObjectId* oid;

  new_command_servant = g_new0(GDACommand_Servant, 1);
  new_command_servant->base._private = 0;
  new_command_servant->base.vepv     = &poa_gda_command__vepv;
  new_command_servant->poa           = cnc->poa;
  new_command_servant->cmd           = gda_odbc_cmd_new();

  gda_odbc_cmd_set_connection(new_command_servant->cmd, cnc->cnc);
  
  POA_GDA_Command__init((PortableServer_Servant) new_command_servant, ev);
  Exception(ev);

  oid = PortableServer_POA_activate_object(new_command_servant->poa, new_command_servant, ev);

  new_command = PortableServer_POA_servant_to_reference(new_command_servant->poa,
							(PortableServer_Servant)new_command_servant, ev);
  Exception(ev);
  return new_command;
}


GDA_Recordset
gda_Command_execute(PortableServer_Servant _obj,
		    const CORBA_char*      cmd,
		    CORBA_long*            affected,
		    const GDA_CmdParameterSeq* params,
		    CORBA_long             options,
		    CORBA_Environment*     ev)
{
  GDACommand_Servant*      command_servant = (GDACommand_Servant*)_obj;
  GDARecordset_Servant*    new_recset_servant;
  GDA_Recordset            new_recset;
  Gda_ODBC_Recordset*      recset;
  PortableServer_ObjectId* oid;
  Gda_ODBC_Error*          error;
  
  if (command_servant->cmd->cmd) {
    fprintf(stderr,"WARNING WARNING: command string still existing\n");
    g_free(command_servant->cmd->cmd);
  }
  command_servant->cmd->cmd = g_strdup(cmd);
  error = gda_odbc_error_new();
  recset = gda_odbc_cmd_execute(command_servant->cmd, error, params, 0);
  if (!recset)
    {
      if (error->description)
	{
	  GDA_DriverError* exception = GDA_DriverError__alloc();
	  exception->errors._length = 1;
	  exception->errors._buffer = CORBA_sequence_GDA_Error_allocbuf(1);
	  exception->errors._buffer[0].description = CORBA_string_dup(error->description);
	  exception->errors._buffer[0].number      = error->number;
	  exception->errors._buffer[0].source      = CORBA_string_dup(error->source);
	  exception->errors._buffer[0].sqlstate    = CORBA_string_dup(error->sqlstate);
	  exception->errors._buffer[0].nativeMsg   = CORBA_string_dup(error->native);
	  exception->realcommand = CORBA_string_dup("<Unknown>");
	  CORBA_exception_set(ev, CORBA_USER_EXCEPTION, ex_GDA_DriverError, exception);
	}
      gda_odbc_error_free(error);
      return CORBA_OBJECT_NIL;
    }
  gda_odbc_error_free(error);
  new_recset_servant = g_new0(GDARecordset_Servant, 1);
  new_recset_servant->base._private = 0;
  new_recset_servant->base.vepv     = &poa_gda_recordset__vepv;
  new_recset_servant->poa           = command_servant->poa;
  new_recset_servant->recset        = recset;

  POA_GDA_Recordset__init((PortableServer_Servant)new_recset_servant, ev);
  Exception(ev);

  oid = PortableServer_POA_activate_object(new_recset_servant->poa, new_recset_servant, ev);
  Exception(ev);

  new_recset = PortableServer_POA_servant_to_reference(new_recset_servant->poa, new_recset_servant, ev);
  Exception(ev);

  return new_recset;
}

CORBA_long
GDA_odbc_recset_close(PortableServer_Servant _obj,
		      CORBA_Environment*      env)
{
  GDARecordset_Servant*  rs = (GDARecordset_Servant*) _obj;
  PortableServer_POA     poa = rs->poa;

  if (gda_odbc_recset_close(rs->recset) < 0)
    {
      fprintf(stderr,"GDA_odbc_recset_close: error during close\n");
      return -1;
    }
  PortableServer_POA_deactivate_object(poa,
				       PortableServer_POA_servant_to_id(poa,rs, env),
				       env);
  Exception(env);
  POA_GDA_Recordset__fini((POA_GDA_Recordset*)rs, env);
  Exception(env);
  return 0;
}

GDA_Recordset_Chunk*
gda_Recset_fetch(PortableServer_Servant _obj, CORBA_long count, CORBA_Environment* ev)
{

  GDARecordset_Servant*  recset_servant = (GDARecordset_Servant*)_obj;
  Gda_ODBC_Recordset*    rs = recset_servant->recset;
  GDA_Row*               row;
  GDA_Recordset_Chunk*   chunk;
  GDA_Field*             field;
  Gda_ODBC_Field*        server_field;
  GList*                 ptr;
  gint                   rowidx = 0;
  gint                   colidx;
  gint                   rc;
  
  chunk = GDA_Recordset_Chunk__alloc();
  chunk->_buffer = CORBA_sequence_GDA_Row_allocbuf(count);
  chunk->_length = 0;
  for (rowidx = 0; rowidx < count; rowidx++)
    {
      chunk->_buffer[rowidx]._length = 0;
      chunk->_buffer[rowidx]._buffer = 0;
    }
  
  rowidx = 0;
  
  do {
      gint  rowlength = g_list_length(rs->fields);
      ptr = rs->fields;
      
      row = &chunk->_buffer[rowidx];
      row->_buffer = CORBA_sequence_GDA_Field_allocbuf(rowlength);
      row->_length = rowlength;
      row->_maximum = 0;
      field = row->_buffer;

      while(ptr)
	{
	  field->realValue._d = 1;
	  server_field = ptr->data;
	  server_field->value = &field->realValue._u.v;
	  server_field->value->_u.lvc = 0;
	  ptr = g_list_next(ptr);
	  field++;
	}

      
      rc = gda_odbc_recset_move_next(rs);
      
      if (rc != 0)
	{
	  chunk->_length = count;
	  break;
	}
      ptr = rs->fields;

      field = &row->_buffer[0];
      colidx = 0;
      while(ptr)
	{
	  server_field = ptr->data;
	  field->actualSize   = server_field->actual_length;
	  field->realValue._d = server_field->actual_length == SQL_NULL_DATA;
	  field->shadowValue._d   = 1;
	  field->originalValue._d = 1;
	  if (field->realValue._u.v._d == GDA_TypeBinary)
	    {
	      fprintf(stderr, "         field->rv._u.v._u.lvb._maximum = %d\n", field->realValue._u.v._u.lvb._maximum);
	      if (field->actualSize == -1)
		{
		  field->realValue._u.v._u.lvb._length = 0;
		  field->realValue._u.v._u.lvb._maximum = 0;
		  /* hei, it's a NULL value */
		  CORBA_free(field->realValue._u.v._u.lvb._buffer);
		  field->realValue._d = 1;
		}
	      else
		field->realValue._u.v._u.lvb._length = field->actualSize;
	    }
	  ptr = g_list_next(ptr);
	  field = &row->_buffer[++colidx];
	}
      rowidx++;
  }while (rowidx < count);
  chunk->_length = rowidx;
  return chunk;
}

GDA_RowAttributes*
gda_Recset_describe(PortableServer_Servant _obj, CORBA_Environment* ev)
{
  GDARecordset_Servant*  recset_servant = (GDARecordset_Servant*)_obj;
  Gda_ODBC_Recordset*    rs = recset_servant->recset;
  GDA_RowAttributes*     rc;
  GList*                 ptr;
  Gda_ODBC_Field*        server_field;
  GDA_FieldAttributes*   field;
  gint                   idx;

  rc = GDA_RowAttributes__alloc();
  rc->_length = g_list_length(rs->fields);
  fprintf(stderr,"gda_Recset_describe: length = %d\n", rc->_length);
  rc->_buffer = CORBA_sequence_GDA_FieldAttributes_allocbuf(rc->_length);
  rc->_maximum = 0;

  idx = 0;
  ptr = rs->fields;
  while (ptr)
    {
      field = &rc->_buffer[idx];
      server_field = ptr->data;
      
      field->name        = CORBA_string_dup(server_field->name);
      fprintf(stderr,"gda_Recset_describe: sending field description for '%s'\n", field->name);
      field->definedSize = server_field->defined_length;
      
      field->scale       = server_field->num_scale;
      field->gdaType     = gda_odbc_sql2gdatype(server_field->sql_type);
      field->nativeType  = server_field->sql_type;
      ptr = g_list_next(ptr);
      idx++;
    }
  return rc;
}
  
CORBA_Object
connection_factory__create  (PortableServer_POA poa,   gpointer *impl_ptr,  CORBA_Environment *ev)
{
  GDAConnectionFactory_Servant* servant;
  PortableServer_ObjectId *objid;
  CORBA_Object retval;

  servant = g_new0(GDAConnectionFactory_Servant, 1);
  servant->base._private = NULL;
  servant->base.vepv     = &poa_gda_connectionfactory__vepv;
  servant->poa           = poa;

  POA_GDA_ConnectionFactory__init((PortableServer_Servant)servant, ev);
  objid = PortableServer_POA_activate_object(poa, servant, ev);
  CORBA_free(objid);
  Exception(ev);
  retval =  PortableServer_POA_servant_to_reference(poa, (PortableServer_Servant)servant, ev);
  *impl_ptr = servant;
  return retval;
}

GDA_Connection_DSNlist* gda_Connection_listSources(PortableServer_Servant _obj,
						   CORBA_Environment* ev)
{
  gpointer                 odbc_iterator;
  gchar                    odbc_path[PATH_MAX];
  gchar*                   dsn_name;
  gchar*                   section_name;
  GDA_Connection_DSNlist*  retval;
  GList*                   data_sources = 0;
  GList*                   ptr;
  gint                     idx;
  
  retval = GDA_Connection_DSNlist__alloc();
  retval->_length = 0;
  retval->_buffer = 0;
  retval->_maximum = 0;
  
  /* user servers */

  g_snprintf(odbc_path, sizeof(odbc_path), "=%s/.odbc.ini", gnome_user_home_dir);
  fprintf(stderr,"gda_Connection_listSources: Pushing '%s'\n", odbc_path);
  gnome_config_push_prefix(odbc_path);
  
  odbc_iterator = gnome_config_init_iterator("=/ODBC Data Sources");
  while ((odbc_iterator = gnome_config_iterator_next(odbc_iterator, &dsn_name, &section_name)))
    {
      data_sources = g_list_append(data_sources, dsn_name);
      fprintf(stderr,"gda_Connection_listSources: got data source name '%s'\n", dsn_name);
      g_free(section_name);
    }
  retval->_length = g_list_length(data_sources);
  retval->_buffer = CORBA_sequence_CORBA_string_allocbuf(retval->_length);
  ptr = data_sources;
  idx = 0;
  while (ptr)
    {
      retval->_buffer[idx] = CORBA_string_dup(ptr->data);
      g_free(ptr->data);
      ptr = g_list_next(ptr);
      idx++;
    }
  g_list_free(data_sources);
  fprintf(stderr,"gda_Connection_listSources: retval->_length  = %d\n", retval->_length);
  fprintf(stderr,"                            retval->_maximum = %d\n", retval->_maximum);
  fprintf(stderr,"                            retval->_buffer  = %p\n", retval->_buffer);
  return retval;
}

      
  

void
connection_factory__destroy (PortableServer_POA poa,   gpointer impl_ptr,   CORBA_Environment *ev)
{
  fprintf(stderr,"connection_factory__destroy called\n");
}
