/* gnome-sql - GUI front-end
 * Copyright (c) 1998 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gsqlfe.h"

/* function prototypes */
static void close_connection (struct GSQLFE_ConnectionInfo *);

/* global variables */
GList *l_connections = NULL;
gint l_openconns = 0;

/* private functions */
static void
close_connection (struct GSQLFE_ConnectionInfo *info)
{
  gint current;
  g_return_if_fail(info != NULL);

  current = gtk_notebook_page_num(GTK_NOTEBOOK(ui_get_work_area()),
                                  info->tab);
  /* remove all related windows */
  browser_close_all(info->db);
  sql_builder_close_all(info->db);
  /* remove the connection itself */
  l_connections = g_list_remove(l_connections, (gpointer) info);
  if (info->command != NULL)
    gda_command_free(info->command);
  if (info->errors != NULL)
    info->errors = error_free_list(info->errors);
  if (info->logfile != NULL)
    g_free(info->logfile);
  gda_connection_free(info->db);
  g_free((gpointer) info);
  gtk_notebook_remove_page(GTK_NOTEBOOK(ui_get_work_area()), current);
  /* refresh the error viewer if it is open */
  error_refresh_viewer(NULL, NULL);
  ui_status_bar_message("Connection closed");
}

/* close current connection */
void
db_close_connection (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      if (gda_connection_close(info->db) >= 0)
        {
          close_connection(info);
          l_openconns--;
        }
      else db_show_last_error(info);
    }
}

/* close all open connections */
void
db_close_all_connections (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info;
  GList *node = g_list_first(l_connections);
  while (node != NULL)
    {
      info = (struct GSQLFE_ConnectionInfo *) node->data;
      /* move to next item in list */
      node = g_list_next(node);
      if (gda_connection_close(info->db) >= 0)
        {
          close_connection(info);
          l_openconns--;
        }
      else db_show_last_error(info);
    }
}

/* commit changes */
void
db_commit (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      if (gda_connection_commit_transaction(info->db) != 0)
	db_show_last_error(info);
      else ui_status_bar_message(N_("Commit executed"));
    }
}

/* execute command */
void
db_exec_command (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      gchar *query;
      Gda_Recordset *rowset;
      /* open result set */
      query = gtk_editable_get_chars(GTK_EDITABLE(info->cmd_win), 0, 
                           gtk_text_get_length(GTK_TEXT(info->cmd_win)));
      if (query != NULL)
        {
          gulong reccount;
	  /* create the command object to be used along the duration
             of the connection and freed when closed */
	  if (info->command == NULL)
	    {
	      info->command = gda_command_new();
	      gda_command_set_connection(info->command, info->db);
	    }
	  gda_command_set_text(info->command, query);
          rowset = gda_command_execute(info->command, &reccount, 0);
          if (rowset != NULL)
            {
              info->grid = grid_show_resultset(info->grid, rowset, 
				                       GTK_CONTAINER(info->scrolled_window));
              gda_recordset_close(rowset);
	      gda_recordset_free(rowset);
            }
          else
	    db_show_last_error(info);
	  sql_store_command(query);
        }
    }
}

/* return current connection */
struct GSQLFE_ConnectionInfo *
db_get_current_connection (void)
{
  gint this_one;
  gint current = gtk_notebook_current_page(GTK_NOTEBOOK(ui_get_work_area()));
  if (current >= 0)
    {
      GList *node = g_list_first(l_connections);
      struct GSQLFE_ConnectionInfo *info;
      while (node != NULL)
        {
          struct GSQLFE_BrowserInfo *brwinfo;
	  struct GSQLFE_SQLBuilderInfo *sqlinfo;
          info = node->data;
          this_one = gtk_notebook_page_num(GTK_NOTEBOOK(ui_get_work_area()),
                                                info->tab);
          if (current == this_one) 
            return (info);
          /* try searching browser windows */
          brwinfo = browser_get_current_window();
          if (brwinfo && brwinfo->db == info->db)
	    return (info);
	  /* try searching SQL builder windows */
	  sqlinfo = sql_builder_get_current_window();
	  if (sqlinfo && sqlinfo->db == info->db)
	    return (info);
          node = g_list_next(node);
        }
    }
  return (NULL);
}

/* create workspace for new connection */
GnomeUIInfo sqltoolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("Run"), N_("Execute SQL command"),
    db_exec_command, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("SQL"), N_("Select SQL Command"),
    sql_select_command, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_UP, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Refresh"), N_("Refresh grid"),
    sql_refresh_grid, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_REFRESH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear SQL window"),
    sql_clear_window, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_TRASH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Print"), N_("Print grid"),
      NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK, 
      GNOME_STOCK_PIXMAP_PRINT, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Open"), N_("Load script"),
    sql_load_script, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save"), N_("Save current buffer"),
    sql_save_script, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_SAVE, 0, 0, NULL },
    GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Cut"), N_("Cut selected text to clipboard"),
    sql_cut_text, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_CUT, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Copy"), N_("Copy selected text to clipboard"),
    sql_copy_text, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_COPY, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Paste"), N_("Paste text from clipboard"),
    sql_paste_text, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_PASTE, 0, 0, NULL },
  GNOMEUIINFO_END
};

void
db_new_connection (Gda_Connection *gdb)
{
  struct GSQLFE_ConnectionInfo *info;
  gchar label[64];
  /* check parameters */
  g_return_if_fail(gdb != NULL);
  /* add connection to private list */
  info = (struct GSQLFE_ConnectionInfo *) 
          g_malloc(sizeof(struct GSQLFE_ConnectionInfo));
  if (info != NULL)
    {
      /* initialize connection structure */
      info->db = gdb;
      info->command = NULL;
      info->errors = NULL;
      info->logging_enabled = FALSE;
      info->logfile = NULL;
      /* create widgets */
      info->tab = gtk_table_new(6, 2, FALSE);
      gtk_widget_show(info->tab);
      sprintf(label, "%s@%s", gda_connection_get_user(gdb),
                              gda_connection_get_dsn(gdb));
      gtk_notebook_append_page_menu(GTK_NOTEBOOK(ui_get_work_area()), info->tab,
                                    gtk_label_new(label), gtk_label_new(label));
      ui_set_notebook_page(GTK_NOTEBOOK(ui_get_work_area()), info->tab);
      info->toolbar = ui_new_toolbar_in_table(info->tab, 0, 0, 2, 1);
      gnome_app_fill_toolbar(GTK_TOOLBAR(info->toolbar), sqltoolbar, NULL);
      info->cmd_win = ui_new_text_in_table(info->tab, 0, 1, 2, 2);
      /* scrolled window to put grid on */
      info->scrolled_window = gtk_scrolled_window_new(NULL, NULL);
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(info->scrolled_window),
                                     GTK_POLICY_AUTOMATIC,
                                     GTK_POLICY_AUTOMATIC);
      gtk_table_attach(GTK_TABLE(info->tab), info->scrolled_window, 0, 2, 2, 6,
                       GTK_SHRINK | GTK_EXPAND | GTK_FILL,
                       GTK_EXPAND | GTK_FILL | GTK_EXPAND,
                       3, 3);
      gtk_widget_show(info->scrolled_window);
      /* create grid */
      info->grid = grid_new_empty(NULL, GTK_CONTAINER(info->scrolled_window));
      /* finally add the node to the list */
      l_connections = g_list_append(l_connections, (gpointer) info);
      l_openconns++;
      error_refresh_viewer(NULL, NULL);
    }
  else ui_show_error("Cannot allocate %d bytes", 
                     sizeof(struct GSQLFE_ConnectionInfo));
}

/* start or stop trace */
void
db_set_trace (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      if (info->logging_enabled)
        {
          /* disable logging */
          if (gda_connection_stop_logging(info->db) == 0)
            {
              info->logging_enabled = FALSE;
              if (info->logfile != NULL) 
                {
                  g_free(info->logfile);
                  info->logfile = NULL;
                }
            }
          else db_show_last_error(info);
        }
      else
        {
          /* enable logging */
          gchar *logfile = ui_get_file_name("Select trace file");
          if (logfile != NULL)
            {
              if (gda_connection_start_logging(info->db, logfile) == 0)
                {
                  if (info->logfile != NULL) g_free(logfile);
                  info->logfile = logfile;
                  info->logging_enabled = TRUE;
                }
              else
                {
                  db_show_last_error(info);
                  g_free(logfile);
                }
            }
        }
    }
}

/* show last database error */
void
db_show_last_error (struct GSQLFE_ConnectionInfo *info)
{
  GList* errors;
  Gda_Error* e;
  GList* ptr;
  GString *errmsg = NULL;

  g_return_if_fail(info != NULL);
  errors = gda_connection_get_errors(info->db);
  ptr = errors;

  if (ptr != NULL)
    {
      /* add new errors to private list */
      info->errors = error_add_list(info->errors, errors);
      while (ptr)
	{
	  if (errmsg == NULL)
	    errmsg = g_string_new("");
	  e = ptr->data;
	  g_string_sprintfa(errmsg, "Location: %s\nError: %s\n", e->source, e->description);
	  ptr = g_list_next(ptr);
	}
      /* only show error if there was an error */
      if (errmsg != NULL)
	{
	  ui_show_error(errmsg->str);
	  g_string_free(errmsg, TRUE);
	}
    }
}

/* discard all changes made */
void
db_rollback (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      if (gda_connection_rollback_transaction(info->db) != 0)
	db_show_last_error(info);
      else ui_status_bar_message(N_("Rollback executed"));
    }
}

/* start a new transaction */
void
db_start_transaction (GtkWidget *w, gpointer data)
{
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL)
    {
      if (gda_connection_begin_transaction(info->db) != 0)
	db_show_last_error(info);
      else ui_status_bar_message(N_("Transaction started"));
    }
}

/* view trace file */
void
db_view_trace_file (GtkWidget *w, gpointer data)
{
  GnomeDialog *dialog;
  GtkWidget *tab, *text;
  struct GSQLFE_ConnectionInfo *info = db_get_current_connection();
  if (info != NULL && info->logging_enabled)
    {
      /* create the dialog box */
      dialog = gnome_dialog_new("Connection Trace File",
                                GNOME_STOCK_BUTTON_OK,
                                NULL);
      gtk_widget_set_usize(GTK_WIDGET(dialog), 400, 400);
      gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
      tab = gtk_table_new(1, 1, TRUE);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), tab,
                         TRUE, TRUE, GNOME_PAD);
      gtk_widget_show(tab);
      text = ui_new_text_in_table(tab, 0, 0, 1, 1);
      ui_load_file_in_text(GTK_TEXT(text), info->logfile);
      gnome_dialog_run_and_close(dialog);
    }
}




