/* gnome-sql - GUI front-end
 * Copyright (c) 1998 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gsqlfe.h"

/* global variables */
extern GtkWidget *wWorkArea, *wStatusBar;
gchar *glb_list_item_data_key = "list_item_data_key";

/* ask user name and password */
static void
login_button_clicked (GtkWidget *w, glong *selected)
{
  *selected = (glong) gtk_object_get_data(GTK_OBJECT(w), "button");
}

gboolean
ui_ask_user_password (gchar *user, gchar *password)
{
  GtkWidget *dialog;
  glong *selected;
  gboolean res;
  /* initialize variables */
  selected = (glong *) g_malloc(sizeof(glong));
  *selected = 0;
  /* create dialog */
  dialog = gnome_dialog_new(N_("User Authentication"), GNOME_STOCK_BUTTON_OK,
                            GNOME_STOCK_BUTTON_CANCEL, NULL);
  gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
  gtk_object_set_data(GTK_OBJECT(g_list_nth(GNOME_DIALOG(dialog)->buttons, 0)),
		      "button", (gpointer) 1);
  gnome_dialog_button_connect(GNOME_DIALOG(dialog),
		              "clicked", 
		              GTK_SIGNAL_FUNC(login_button_clicked),
		              (gpointer) selected);
  gtk_object_set_data(GTK_OBJECT(g_list_nth(GNOME_DIALOG(dialog)->buttons, 1)),
		      "button", (gpointer) 2);
  gnome_dialog_button_connect(GNOME_DIALOG(dialog),
		              "clicked", 
		              GTK_SIGNAL_FUNC(login_button_clicked),
		              (gpointer) selected);
  gtk_widget_show(dialog);
  /* wait for selection from user */
  while (!*selected)
    gtk_main_iteration_do(TRUE);
  if (*selected == 1)
    {
      res = TRUE;
    }
  else res = FALSE;
  /* destroy dialog */
  gnome_dialog_close(GNOME_DIALOG(dialog));
  g_free((gpointer) selected);
  return (res);
}

void
ui_fill_list (GtkList *wlist, GList *mlist)
{
  GList *node;
  /* check parameters */
  g_return_if_fail(wlist != NULL && mlist != NULL);
  /* clear the list and fill it */
  gtk_list_clear_items(wlist, 0, -1);
  for (node = g_list_first(mlist); node != NULL; node = g_list_next(node))
    {
      GtkWidget *w = gtk_list_item_new_with_label(node->data);
      gtk_container_add(GTK_CONTAINER(wlist), w);
      gtk_widget_show(w);
      /* associate label with list item */
      gtk_object_set_data(GTK_OBJECT(w), glb_list_item_data_key,
                          (gchar *) node->data);
    }
}

void
ui_flash_message (const gchar *format, ...)
{
  va_list args;
  gchar sz[256];
  /* build the message string */
  va_start(args, format);
  vsprintf(sz, format, args);
  va_end(args);
  gnome_app_flash(GNOME_APP(glb_app), sz);
}

gint
ui_get_clist_selection (GtkCList *clist)
{
  GList *sel;
  g_return_val_if_fail(clist != NULL, -1);
  sel = GTK_CLIST(clist)->selection;
  return (sel != NULL ? GPOINTER_TO_INT(sel->data) : -1);
}

/* open file selection dialog and return selected file */
static void
filesel_button_clicked (GtkWidget *w, glong *selected)
{
  *selected = (glong) gtk_object_get_data(GTK_OBJECT(w), "button");
}

gchar *
ui_get_file_name (gchar *title)
{
  GtkWidget *filesel;
  glong *selected;
  gchar *res = NULL;
  /* initialize variables */
  selected = (glong *) g_malloc(sizeof(glong));
  *selected = 0;
  /* create dialog */
  filesel = gtk_file_selection_new(title);
  gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
		      "button", (gpointer) 1);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(filesel_button_clicked),
		     (gpointer) selected);
  gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
		      "button", (gpointer) 2);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
		     "clicked", GTK_SIGNAL_FUNC(filesel_button_clicked),
		     (gpointer) selected);
  gtk_widget_show(filesel);
  gtk_file_selection_show_fileop_buttons(GTK_FILE_SELECTION(filesel));
  /* wait for selection from user */
  while (!*selected)
    gtk_main_iteration_do(TRUE);
  if (*selected == 1) /* Ok-button */
    res = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel)));
  else res = NULL;
  /* free memory before exiting */
  gtk_widget_destroy(filesel);
  g_free((gpointer) selected);
  return (res);
}

GtkWidget *
ui_get_work_area (void)
{
  return (wWorkArea);
}

/* insert a file into the current position */
void
ui_insert_file_into_text (GtkText *text, gchar *filename)
{
  FILE *file;
  /* check parameters */
  g_return_if_fail(text != NULL);
  g_return_if_fail(filename != NULL);
  /* open file */
  if ((file = fopen(filename, "r")))
    {
      gtk_text_set_point(text, gtk_text_get_point(text));
      gtk_text_freeze(text);
      while (!feof(file))
	{
	  gchar buffer[1025];
	  memset(buffer, 0, sizeof(buffer));
	  fread(buffer, sizeof(buffer) - 1, 1, file);
	  if (ferror(file))
	    {
	      ui_show_error("Error reading file %s", filename);
	      break;
	    }
	  gtk_text_insert(text, NULL, NULL, NULL, buffer, strlen(buffer));
	}
      gtk_text_thaw(text);
      fclose(file);
      ui_status_bar_message("File %s inserted", filename);
    }
  else ui_show_error("Error opening file %s", filename);
}

/* load a file into a text widget */
void
ui_load_file_in_text (GtkText *text, gchar *filename)
{
  FILE *file;
  /* check parameters */
  g_return_if_fail(text != NULL);
  g_return_if_fail(filename != NULL);
  /* open file */
  if ((file = fopen(filename, "r")))
    {
      gtk_editable_delete_text(GTK_EDITABLE(text), 0,
                               gtk_text_get_length(text));
      gtk_text_freeze(text);
      while (!feof(file))
	{
	  gchar buffer[1025];
	  memset(buffer, 0, sizeof(buffer));
	  fread(buffer, sizeof(buffer) - 1, 1, file);
	  if (ferror(file))
	    {
	      ui_show_error("Error reading file %s", filename);
	      break;
	    }
	  gtk_text_insert(text, NULL, NULL, NULL, buffer, strlen(buffer));
	}
      gtk_text_thaw(text);
      fclose(file);
      ui_status_bar_message("File %s loaded", filename);
    }
  else ui_show_error("Error opening file %s", filename);
}

/* create a GtkCList in a table */
static void
clist_hide_columns_cb (GtkWidget *w, gpointer data)
{
  gtk_clist_column_titles_hide(GTK_CLIST(data));
}

static void
clist_show_columns_cb (GtkWidget *w, gpointer data)
{
  gtk_clist_column_titles_show(GTK_CLIST(data));
}

GnomeUIInfo clistpopupmenu[] =
{
  { GNOME_APP_UI_ITEM, N_("Print grid"), N_("Print grid contents"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_PRINT, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Show columns"), N_("Show column titles"),
    clist_show_columns_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_TOP, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Hide columns"), N_("Hide column titles"),
    clist_hide_columns_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_BOTTOM, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Edit current row..."), N_("Edit currently selected row"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_FONT, 0, 0, NULL },
  GNOMEUIINFO_END
};

GtkWidget *
ui_new_clist_in_table (GtkWidget *table, gint numcols, gchar *cols[],
                       gint x1, gint y1, gint x2, gint y2)
{
  GtkWidget *clist, *scroll, *popup_menu;
  g_return_val_if_fail(table != NULL, NULL);
  scroll = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
                                 GTK_POLICY_AUTOMATIC,
                                 GTK_POLICY_AUTOMATIC);
  gtk_table_attach(GTK_TABLE(table), scroll, x1, x2, y1, y2,
                   GTK_SHRINK | GTK_EXPAND | GTK_FILL,
                   GTK_EXPAND | GTK_FILL,
                   3, 3);
  gtk_widget_show(scroll);
  clist = cols != NULL ? gtk_clist_new_with_titles(numcols, cols) :
                         gtk_clist_new(numcols);
  gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_SINGLE);
  gtk_clist_moveto(GTK_CLIST(clist), 0, 0, 0.0, 0.0);
  gtk_widget_show(clist);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll), clist);
  /* create popup-menu */
  popup_menu = ui_new_popup_menu(clist, &clistpopupmenu, (gpointer) clist);
  return (clist);
}

/* create a GtkLabel widget and attach it to a table */
GtkWidget *
ui_new_label_in_table (GtkWidget *table, gchar *text, gint x1, gint y1,
                       gint x2, gint y2)
{
  GtkWidget *label;
  /* check pointers */
  g_return_val_if_fail(table != NULL, NULL);
  label = gtk_label_new(text);
  gtk_table_attach(GTK_TABLE(table), label, x1, x2, y1, y2, 
                   GTK_SHRINK | GTK_FILL,
                   GTK_SHRINK | GTK_FILL,
                   3, 3);
  gtk_widget_show(label);
}

GtkWidget *
ui_new_combo_in_table (GtkWidget *table, gint x1, gint y1, gint x2, gint y2)
{
  GtkWidget *combo;
  /* check parameters */
  g_return_val_if_fail(table != NULL, NULL);
  combo = gtk_combo_new();
  gtk_table_attach(GTK_TABLE(table), combo, x1, x2, y1, y2,
		   GTK_SHRINK | GTK_FILL,
		   GTK_SHRINK | GTK_FILL,
		   3, 3);
  gtk_widget_show(combo);
  return (combo);
}
 
GtkWidget *
ui_new_entry_in_table (GtkWidget *table, gchar *name, gchar *text, gint x1, gint y1,
                      gint x2, gint y2)
{
  GtkWidget *entry;
  /* check pointers */
  g_return_val_if_fail(table != NULL, NULL);
  entry = gtk_entry_new();
  gtk_widget_set_name(entry, name);
  if (text != NULL)
    gtk_entry_set_text(GTK_ENTRY(entry), text);
  gtk_table_attach(GTK_TABLE(table), entry, x1, x2, y1, y2, 
                   GTK_SHRINK | GTK_FILL,
                   GTK_SHRINK | GTK_FILL,
                   3, 3);
  gtk_widget_show(entry);
  return (entry);
}         

/* popup menu */
GtkWidget *
ui_new_popup_menu (GtkWidget *parent, GnomeUIInfo *ui_info, gpointer data)
{
  GtkWidget *popup_menu;
  g_return_val_if_fail(parent != NULL, NULL);
  g_return_val_if_fail(ui_info != NULL, NULL);
  popup_menu = gnome_popup_menu_new(ui_info);
  gnome_popup_menu_attach(popup_menu, parent, (gpointer) data);
  return (popup_menu);
}

/* text editor */
static void
copy_text_cb (GtkWidget *w, gpointer data)
{
  gtk_editable_copy_clipboard(GTK_EDITABLE(data));
}

static void
cut_text_cb (GtkWidget *w, gpointer data)
{
  gtk_editable_cut_clipboard(GTK_EDITABLE(data));
  gtk_editable_changed(GTK_EDITABLE(data));
}

static void
insert_file_cb (GtkWidget *w, gpointer data)
{
  gchar *filename = ui_get_file_name(_("Select File"));
  if (filename != NULL)
    {
      ui_insert_file_into_text(GTK_TEXT(data), filename);
      gtk_editable_changed(GTK_EDITABLE(data));
      g_free(filename);
    }
}

static void
paste_text_cb (GtkWidget *w, gpointer data)
{
  gtk_editable_paste_clipboard(GTK_EDITABLE(data));
  gtk_editable_changed(GTK_EDITABLE(data));
}

static void
save_to_file_cb (GtkWidget *w, gpointer data)
{
  gchar *filename = ui_get_file_name(_("Select File"));
  if (filename != NULL)
    {
      ui_save_text_to_file(GTK_TEXT(data), filename);
      g_free(filename);
    }
}

GnomeUIInfo textpopupmenu[] =
{
  { GNOME_APP_UI_ITEM, N_("Cut"), 
    N_("Cut selected text to clipboard"),
    cut_text_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_CUT, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Copy"), 
    N_("Copy selected text to clipboard"),
    copy_text_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_COPY, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Paste"), N_("Paste text from clipboard"),
    paste_text_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_PASTE, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Insert file..."), N_("Insert file into current position"),
    insert_file_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save to file..."), N_("Save widget contents to file"),
    save_to_file_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_SAVE_AS, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Print..."), N_("Print buffer"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_PRINT, 0, 0, NULL },
  GNOMEUIINFO_END
};

GtkWidget *
ui_new_text_in_table (GtkWidget *table, gint x1, gint y1, gint x2, gint y2)
{
  GtkWidget *text, *scroll_bar, *box, *popup_menu;
  /* check parameters */
  g_return_val_if_fail(table != NULL, NULL);
  /* create vertical box */
  box = gtk_hbox_new(FALSE, 3);
  gtk_table_attach(GTK_TABLE(table), box, x1, x2, y1, y2,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
                   3, 3);
  /* create text widget */
#ifdef HAVE_GTKEDITOR
  text = gtk_editor_new(NULL, NULL);
#else
  text = gtk_text_new(NULL, NULL);
#endif
  gtk_text_set_editable(GTK_TEXT(text), TRUE);
  gtk_text_set_word_wrap(GTK_TEXT(text), TRUE);
  gtk_widget_set_sensitive(text, TRUE);
  gtk_box_pack_start(GTK_BOX(box), text, TRUE, TRUE, 0);
  gtk_widget_show(text);
  /* create scroll bar */
  scroll_bar = gtk_vscrollbar_new(GTK_TEXT(text)->vadj);
  gtk_box_pack_start(GTK_BOX(box), scroll_bar, FALSE, FALSE, 0);
  gtk_widget_show(scroll_bar);
  gtk_widget_show(box);
  /* create popup-menu */
  popup_menu = ui_new_popup_menu(text, &textpopupmenu, (gpointer) text);
  return (text);
}

GtkWidget *
ui_new_toolbar_in_table (GtkWidget *table, gint x1, gint y1, gint x2, gint y2)
{
  GtkWidget *toolbar;
  g_return_val_if_fail(table != NULL, NULL);
  toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
  gtk_table_attach(GTK_TABLE(table), toolbar, x1, x2, y1, y2,
                   GTK_SHRINK | GTK_FILL,
                   GTK_FILL,
                   3, 3);
  gtk_widget_show(toolbar);
  return (toolbar);
}

GtkWidget *
ui_new_tree_in_table (GtkWidget *table, gint x1, gint y1, gint x2, gint y2)
{
  GtkWidget *scrolled_window, *tree;
  g_return_val_if_fail(table != NULL, NULL);
  /* create scrolled window to put tree inside */
  scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  gtk_table_attach(GTK_TABLE(table), 
		   scrolled_window, x1, x2, y1, y2,
		   GTK_FILL | GTK_SHRINK,
		   GTK_SHRINK | GTK_EXPAND | GTK_FILL,
		   3, 3);
  gtk_widget_show(scrolled_window);
  /* create the tree widget */
  tree = gtk_tree_new();
  gtk_tree_set_selection_mode(GTK_TREE(tree), GTK_SELECTION_SINGLE);
  gtk_tree_set_view_lines(GTK_TREE(tree), TRUE);
  gtk_widget_show(tree);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), 
					tree);
  return (tree);
}

gboolean
ui_save_text_to_file (GtkText *text, gchar *filename)
{
  gchar *buffer;
  /* check parameters */
  g_return_val_if_fail(text != NULL, FALSE);
  g_return_val_if_fail(filename != NULL, FALSE);
  /* get text widget's contents */
  buffer = gtk_editable_get_chars(GTK_EDITABLE(text), 0, gtk_text_get_length(text));
  if (buffer != NULL)
    {
      FILE *file = fopen(filename, "w");
      if (file != NULL)
	{
	  fwrite(buffer, gtk_text_get_length(text), 1,
		 file);
	  if (ferror(file))
	    ui_show_error("Error writing to file %s", filename);
	  /* close file */
	  fclose(file);
	  ui_status_bar_message("Wrote file %s", filename);
	}
      else ui_show_error("Error opening file %s", filename);
    }
}

void
ui_set_notebook_page (GtkNotebook *notebook, GtkWidget *tab)
{
  gint current;
  /* check parameters */
  g_return_if_fail(notebook != NULL);
  g_return_if_fail(tab != NULL);
  /* activate the page by id */
  current = gtk_notebook_page_num(notebook, tab);
  gtk_notebook_set_page(notebook, current);
}

void
ui_show_error (const gchar *format, ...)
{
    GtkWidget *dialog;
    /* build the string */
    va_list args;
    gchar sz[256];
    /* build the message string */
    va_start(args, format);
    vsprintf(sz, format, args);
    va_end(args);
    /* create the dialog box */
    gnome_app_error(GNOME_APP(glb_app), sz);
}

void
ui_show_file_in_window (const gchar *filename)
{
  GtkWidget *dialog, *table, *viewer;
  g_return_if_fail(filename != NULL);
  /* create the dialog box */
  dialog = gnome_dialog_new(filename, GNOME_STOCK_BUTTON_CLOSE, NULL);
  gnome_dialog_set_close(GNOME_DIALOG(dialog), TRUE);
  table = gtk_table_new(1, 1, TRUE);
  gtk_widget_set_usize(table, 550, 300);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table,
                     TRUE, TRUE, GNOME_PAD);
  gtk_widget_show(table);
  viewer = ui_new_text_in_table(table, 0, 0, 1, 1);
  ui_load_file_in_text(GTK_TEXT(viewer), filename);
  //gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
  gtk_widget_show(dialog);
}

void
ui_status_bar_message (const gchar *format, ...)
{
  va_list args;
  gchar sz[256];
  /* build the message string */
  va_start(args, format);
  vsprintf(sz, format, args);
  va_end(args);
  gnome_appbar_set_status(GNOME_APPBAR(wStatusBar), (const gchar *) sz);
}

GtkWidget *
ui_tree_new_subtree (GtkTree *tree, gchar *label)
{
  GtkWidget *item, *subtree;
  g_return_val_if_fail(tree != NULL, NULL);
  g_return_val_if_fail(label != NULL, NULL);
  item = gtk_tree_item_new_with_label(label);
  gtk_tree_append(GTK_TREE(tree), item);
  subtree = gtk_tree_new();
  gtk_tree_item_set_subtree(GTK_TREE_ITEM(item), subtree);
  gtk_widget_show(item);
  return (subtree);
}
