/* GNOME-DB
 * Copyright (c) 1998 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <ctype.h>
#include "gda.h"
#include "gda-common.h"
#include "gda-postgres-types.h"
#include "gda-postgres-command.h"
#include "gda-postgres-connection.h"
#include "gda-postgres-error.h"
#include "gda-postgres-recset.h"

typedef Gda_POSTGRES_Recordset* (*schema_ops_fn)(Gda_POSTGRES_Recordset *,
						 Gda_POSTGRES_Error *,
						 Gda_POSTGRES_Connection *,
						 GDA_Connection_Constraint *,
						 gint );

schema_ops_fn schema_ops[GDA_Connection_GDNC_SCHEMA_LAST] =
{
  0,
};

/* private functions */
static gint
execute_command (Gda_POSTGRES_Connection *cnc, gchar *cmd)
{
  g_return_val_if_fail(cnc != NULL, -1);
  g_return_val_if_fail(cmd != NULL, -1);
  if (PQstatus(cnc->pq_conn) == CONNECTION_OK)
    {
      PGresult *rc = PQexec(cnc->pq_conn, cmd);
      switch (PQresultStatus(rc))
        {
        case PGRES_COMMAND_OK :
        case PGRES_EMPTY_QUERY :
	case PGRES_TUPLES_OK :
        case PGRES_COPY_OUT :
	case PGRES_COPY_IN :
          PQclear(rc);
          return (0);
	case PGRES_BAD_RESPONSE :
	case PGRES_NONFATAL_ERROR :
	case PGRES_FATAL_ERROR :
	  break;
        }
      gda_postgres_error_make(NULL, NULL, cnc, "PQexec");
      if (rc != NULL)
        PQclear(rc);
    }
  return (-1);
}

static Gda_POSTGRES_Recordset *
schema_columns (Gda_POSTGRES_Recordset *recset, Gda_POSTGRES_Error *error,
                Gda_POSTGRES_Connection *cnc, GDA_Connection_Constraint *constraint,
                gint length)
{
  Gda_POSTGRES_Command *cmd;
  GDA_Connection_Constraint *ptr;
  gchar *table_qualifier, *table_owner, *table_name, *column_name;
  /* create the command object */
  cmd = gda_postgres_cmd_new();
  cmd->cnc = cnc;
  /* get table name */
  ptr = constraint;
  while(length)
    {
      switch (ptr->ctype)
	{
	case GDA_Connection_TABLE_CATALOG:
	  table_qualifier = ptr->value;
	  fprintf(stderr, "schema_columns: table_qualifier = '%s'\n", table_qualifier);
	  break;
	case GDA_Connection_TABLE_SCHEMA:
	  table_owner = ptr->value;
	  fprintf(stderr, "schema_columns: table_owner = '%s'\n", table_owner);
	  break;
	case GDA_Connection_TABLE_NAME:
	  table_name = ptr->value;
	  fprintf(stderr, "schema_columns: table_name = '%s'\n", table_name);
	  break;
	case GDA_Connection_COLUMN_NAME:
	  column_name = ptr->value;
	  fprintf(stderr, "schema_columns: column_name = '%s'\n", column_name);
	  break;
	default:
	  fprintf(stderr,"schema_columns: invalid constraint type %d\n", ptr->ctype);
	  return 0;
	  break;
	}
      length--;
      ptr++;
    }
  /* FIXME: missing fields: COLUMN_NAME, DATA_TYPE, TYPE_NAME, DATA_PRECISION,
                            LENGTH, NUMERIC_SCALE, NUMERIC_RADIX, NULLABLE, REMARKS */
  cmd->cmd = g_strdup_printf("SELECT a.relname, c.usename, a.relname "
                             " FROM pg_attribute d, pg_user c, pg_database b, pg_class a "
                             " WHERE a.relowner = b.datdba AND c.usesysid = b.datdba "
                             " AND a.oid = d.attrelid "
                             " AND b.datname = %s "
                             " AND a.relname = %s ",
                             PQdb(cnc->pq_conn), table_name);
  /* not finished */
  return (recset);
}

static Gda_POSTGRES_Recordset *
schema_procedures (Gda_POSTGRES_Recordset *recset, Gda_POSTGRES_Error *error,
                   Gda_POSTGRES_Connection *cnc, GDA_Connection_Constraint *constraint,
                   gint length)
{
  return (recset);
}

static Gda_POSTGRES_Recordset *
schema_tables (Gda_POSTGRES_Recordset *recset, Gda_POSTGRES_Error *error,
	       Gda_POSTGRES_Connection *cnc, GDA_Connection_Constraint *constraint,
	       gint length)
{
  gchar *query;
  Gda_POSTGRES_Command *cmd;
  /* build the command object */
  cmd = gda_postgres_cmd_new();
  cmd->cnc = cnc;
  /* build the query */
  query = g_strdup_printf("SELECT a.relname, c.usename, a.relname, 'TABLE', null "
                          " FROM pg_user c, pg_database b, pg_class a "
                          " WHERE b.datname = '%s' "
                          " AND a.relowner = b.datdba AND c.usesysid = b.datdba"
			  " ORDER BY a.relname",
                          PQdb(cnc->pq_conn));
  cmd->cmd = query;
  recset->cmd = cmd;
  /* execute the command */
  cmd->pq_data = PQexec(cnc->pq_conn, cmd->cmd);
  switch (PQresultStatus(cmd->pq_data))
    {
    case PGRES_EMPTY_QUERY :
    case PGRES_TUPLES_OK :
    case PGRES_COMMAND_OK :
    case PGRES_COPY_OUT :
    case PGRES_COPY_IN :
      return (recset);
    case PGRES_BAD_RESPONSE :
    case PGRES_NONFATAL_ERROR :
    case PGRES_FATAL_ERROR :
      break;
    }
  gda_postgres_error_make(error, recset, cnc, "schema_tables");
  return (0);
}

static void
initialize_schema_ops (void)
{
  schema_ops[GDCN_SCHEMA_TABLES] = schema_tables;
  schema_ops[GDCN_SCHEMA_COLS] = schema_columns;
  schema_ops[GDCN_SCHEMA_PROCS] = schema_procedures;
}

/* create new connection object */
Gda_POSTGRES_Connection *
gda_postgres_connection_new (void)
{
  static gint initialized = 0;
  Gda_POSTGRES_Connection *c = g_new0(Gda_POSTGRES_Connection, 1);
  if (!initialized)
    {
      initialize_schema_ops();
      initialized = 1;
    }
  /* initialize everything to NULL to avoid core dumped */
  c->pq_host = NULL;
  c->pq_port = NULL;
  c->pq_options = NULL;
  c->pq_tty = NULL;
  c->pq_db = NULL;
  c->pq_login = NULL;
  c->pq_pwd = NULL;
  return (c);
}

/* free given connection object */
void
gda_postgres_connection_free (Gda_POSTGRES_Connection *c)
{
  g_return_if_fail(c != NULL);
  gda_postgres_connection_close(c);
  if (c->pq_host != NULL) g_free((gpointer) c->pq_host);
  if (c->pq_port != NULL) g_free((gpointer) c->pq_port);
  if (c->pq_options != NULL) g_free((gpointer) c->pq_options);
  if (c->pq_tty != NULL) g_free((gpointer) c->pq_tty);
  if (c->pq_db != NULL) g_free((gpointer) c->pq_db);
  if (c->pq_login != NULL) g_free((gpointer) c->pq_login);
  if (c->pq_pwd != NULL) g_free((gpointer) c->pq_pwd);
  g_free((gpointer) c);
}

/* open new connection to database server */
static gchar*
get_value(gchar* ptr)
{
  while (*ptr && *ptr != '=')
    ptr++;
  if (!*ptr)
    return 0;
  ptr++;
  if (!*ptr)
    return 0;
  while (*ptr && isspace(*ptr))
    ptr++;
  return (g_strdup(ptr));
}

gint
gda_postgres_connection_open (Gda_POSTGRES_Connection *cnc, const gchar *dsn,
			      const gchar *user, const gchar *password)
{
  gchar *ptr_s, *ptr_e;
  /* get options from connection string */
  ptr_s = dsn;
  while (ptr_s && *ptr_s)
    {
      ptr_e = strchr(ptr_s, ';');
      if (ptr_e) *ptr_e = '\0';
      if (strcasecmp(ptr_s, "HOST") == 0)
	cnc->pq_host = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "DATABASE", strlen("DATABASE")) == 0)
	cnc->pq_db = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "PORT", strlen("PORT")) == 0)
	cnc->pq_port = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "OPTIONS", strlen("OPTIONS")) == 0)
	cnc->pq_options = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "TTY", strlen("TTY")) == 0)
	cnc->pq_tty = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "LOGIN", strlen("LOGIN")) == 0)
	cnc->pq_login = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "PASSWORD", strlen("PASSWORD")) == 0)
	cnc->pq_pwd = get_value(ptr_s);
      ptr_s = ptr_e;
      if (ptr_s)
	ptr_s++;
    }
  if (cnc->pq_login == NULL) 
    cnc->pq_login = user != NULL ? g_strdup(user) : NULL;
  if (cnc->pq_pwd == NULL)
    cnc->pq_pwd = password != NULL ? g_strdup(password) : NULL;
  /* actually establish the connection */
  cnc->pq_conn = PQsetdbLogin(cnc->pq_host, cnc->pq_port, cnc->pq_options,
			      cnc->pq_tty, cnc->pq_db, cnc->pq_login, cnc->pq_pwd);
  if (PQstatus(cnc->pq_conn) != CONNECTION_OK)
    {
      fprintf(stderr, "PostgreSQL error: %s\n", PQerrorMessage(cnc->pq_conn));
      return (-1);
    }
  return (0);
}

/* schemas */
Gda_POSTGRES_Recordset *
gda_postgres_connection_open_schema(Gda_POSTGRES_Connection *cnc,
				    Gda_POSTGRES_Error *e,
				    GDA_Connection_QType t,
				    GDA_Connection_Constraint *constraints,
				    gint length)
{
  Gda_POSTGRES_Recordset* rc = gda_postgres_recset_new();
  schema_ops_fn fn = schema_ops[(gint) t];

  if (fn)
    return fn(rc, e, cnc, constraints, length);
  else
    g_log("GDA POSTGRES", G_LOG_LEVEL_INFO,
	  "schema_open_schema: Unhandled SCHEMA_QTYPE %d\n", (gint) t);
  gda_postgres_recset_free(rc);
  return 0;
}

/* close given connection to database */
gint
gda_postgres_connection_close (Gda_POSTGRES_Connection *c)
{
  /* check connection status */
  if (PQstatus(c->pq_conn) == CONNECTION_OK)
    {
      PQfinish(c->pq_conn);
      c->pq_conn = NULL;
      return (0);
    }
  gda_postgres_error_make(NULL, NULL, c, "gda_postgres_connection_close");
  return (-1);
}

/* BEGIN TRANS */
gint
gda_postgres_connection_begin_transaction (Gda_POSTGRES_Connection *cnc)
{
  return (execute_command(cnc, "BEGIN"));
}

/* COMMIT */
gint
gda_postgres_connection_commit_transaction (Gda_POSTGRES_Connection *cnc)
{
  return (execute_command(cnc, "COMMIT"));
}

/* ROLLBACK */
gint
gda_postgres_connection_rollback_transaction (Gda_POSTGRES_Connection *cnc)
{
  return (execute_command(cnc, "ROLLBACK"));
}

/* logging */
gint
gda_postgres_connection_start_logging (Gda_POSTGRES_Connection *cnc, gchar *filename)
{
  FILE *f;
  gchar *str;
  /* check parameters */
  g_return_val_if_fail(cnc != NULL, -1);
  if (PQstatus(cnc->pq_conn) != CONNECTION_OK)
    return (-1);
  /* open given file or default one if NULL */
  str = filename ? filename : g_strdup_printf("%s/gda.log", g_get_home_dir());
  if (!(f = fopen(str, "w")))
    {
      if (str != filename)
        g_free(str);
      return (-1);
    }
  PQtrace(cnc->pq_conn, f);
  /* free memory */
  if (str != filename)
    g_free(str);
  return (0);
}

gint
gda_postgres_connection_stop_logging (Gda_POSTGRES_Connection *cnc)
{
  /* check parameters */
  g_return_val_if_fail(cnc != NULL, -1);
  if (PQstatus(cnc->pq_conn) != CONNECTION_OK)
    return (-1);
  PQuntrace(cnc->pq_conn);
  return (0);
}
