/* GNOME-DB
 * Copyright (c) 1998 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-postgres-types.h"
#include "gda-postgres-command.h"
#include "gda-postgres-connection.h"

/* private functions */
static gboolean
fill_field_values (Gda_POSTGRES_Recordset *recset, glong row)
{
  register gint cnt = 0;
  GList *node;
  /* check parameters */
  g_return_val_if_fail(recset != NULL, FALSE);
  g_return_val_if_fail(recset->cmd != NULL, FALSE);
  g_return_val_if_fail(recset->cmd->pq_data != NULL, FALSE);
  /* traverse all recordset values */

  node = g_list_first(recset->fields);

  fprintf(stderr,"getting data from row %d\n", row);
  while (node != NULL)
    {
      Gda_POSTGRES_Field *field = (Gda_POSTGRES_Field *) node->data;
      if (field != NULL)
        {
          gchar *native_value;
          /* FIXME: must map Postgres value to GDA value */
          field->value->_d = GDA_TypeVarchar;
          /* get field's value from server */

	  native_value = PQgetvalue(recset->cmd->pq_data, row, cnt);
	  field->value->_d = GDA_TypeVarchar;
	  if (native_value != NULL)
	    {
	      field->actual_length = strlen(native_value) + 1;
	      field->value->_u.lvc = CORBA_string_dup(native_value);
	    }
	  else
	    {
	      field->actual_length = 0;
	      field->value->_u.lvc = 0;
	    }
	}
      node = g_list_next(node);
      cnt++;
    }
  return (TRUE);
}

/* create a new result set object */
Gda_POSTGRES_Recordset *
gda_postgres_recset_new (void)
{
  Gda_POSTGRES_Recordset *rc = g_new0(Gda_POSTGRES_Recordset, 1);
  memset(rc, 0, sizeof(Gda_POSTGRES_Recordset));
  return (rc);
}

/* free Recordset object */
void
gda_postgres_recset_free (Gda_POSTGRES_Recordset *recset)
{
  gda_postgres_recset_close(recset);
  g_free((gpointer) recset);
}

/* cursor movement */
gint
gda_postgres_recset_move_next (Gda_POSTGRES_Recordset *recset)
{
  gint ntuples = PQntuples(recset->cmd->pq_data);

  /* check if we can fetch forward */
  if (recset->pos < PQntuples(recset->cmd->pq_data))
    {
      if (fill_field_values(recset, recset->pos))
        {
          recset->at_end = recset->at_begin = 0;
          recset->pos++;
          return (0);
        }
    }
  else
    {
      recset->at_end = 1;
      return (1);
    }
  return (-1);
}

gint
gda_postgres_recset_move_prev (Gda_POSTGRES_Recordset *recset)
{
  if (recset->pos > 0)
    {
      if (fill_field_values(recset, recset->pos - 1))
        {
          recset->at_begin = recset->at_end = 0;
          recset->pos--;
          return (0);
        }
    }
  else
    {
      recset->at_begin = 1;
      return (1);
    }
  return (-1);
}

/* close given recordset */
gint
gda_postgres_recset_close (Gda_POSTGRES_Recordset *recset)
{
  GList *ptr;
  /* free associated command object */
  if (recset->cmd != NULL)
      gda_postgres_cmd_free(recset->cmd);
  /* free fields' list */
  ptr = recset->fields;
  while (ptr != NULL)
    {
      gda_postgres_field_free((gpointer) ptr->data);
      ptr = g_list_next(ptr);
    }
  g_list_free(recset->fields);
  recset->fields = NULL;
  return (0);
}

/* debugging */
void
gda_postgres_recset_dump (Gda_POSTGRES_Recordset *recset)
{
}
